package com.installshield.extras.product;

/**
 * FileSystemManipulation.java
 *	
 * @author JeffDillon 
 * @version 2.0
 * 
 * This Product Action takes a source and target file or directory
 * and performs the specified action. These actions include copy, move,
 * and delete. If a directory is specified as the source then a recursive
 * copy, move, or delete is done.
 *
 * All operations throw a product exception if a failure occurs
 * except for the delete operation and its target not existing.
 *  
 * Note: You should make sure that the "External Files Bean" does not work 
 * for you before using this bean since that is a officially supported
 * ISMP bean.
 * 
 * Note: This bean does not implement the getRequiredBytes method. That
 * means that if you do a large copy you should add a "AddRequiredBytes" bean 
 * to your project since the size of the target is not added to the required
 * bytes. The getRequiredBytes method is not implemented since the source file
 * or directory probably does not exist when the diskspace check is done.
 *
 */

import com.installshield.util.*;
import com.installshield.product.*;
import com.installshield.wizard.service.file.*;
import com.installshield.wizard.service.ServiceException;

public class FileSystemManipulation extends ProductAction {

    public static final int COPY = 0;
    public static final int MOVE = 1;
    public static final int DELETE = 2;

    public static final int INSTALL = 0;
    public static final int UNINSTALL = 1;

    private String target = "";
    private String source = "";

    private int action = COPY;
    private int runtime = INSTALL;

    private FileService fs;

    /**
     * Performs the specified action at install time
     */
    public void install(ProductActionSupport support) throws ProductException {

        if (runtime == INSTALL) {
            doAction(source, target, action);
        }

    }

    /**
     * Helper controller method to determine the correct action
     * based on the bean properties.
     */
    private void doAction(String source, String target, int action) throws ProductException {

        target = resolveString(target);
        source = resolveString(source);

        // make the paths absolute
        if (! FileUtils.isAbsolute(source)) {
            source =
                FileUtils.appendSeparator(getParentComponent().getAbsoluteInstallLocation())
                    + source;
        }
        if (! target.equals("")) {
            if (! FileUtils.isAbsolute(target)) {
                target =
                    FileUtils.appendSeparator(getParentComponent().getAbsoluteInstallLocation())
                        + target;
            }
        }

        // Check to make sure the source bean property exists on the 
        // filesystem. Otherwise throw a product exception. Handle a delete
        // as a special case.
        try {

            fs = (FileService) getService(FileService.NAME);

            if ((!fs.fileExists(source)) && (action != DELETE)) {

                throw new ProductException(
                    ProductException.PRODUCT_ACTION_INSTALL_FAILED,
                    Log.ERROR,
                    source + " does not exist on filesystem");

            }

            if ((!fs.fileExists(source)) && (action == DELETE)) {
                logEvent(
                    this,
                    Log.WARNING,
                    "File " + source + " can not be deleted since it does not exist on filesystem");
                return;
            }

        } catch (ServiceException se) {
            throw new ProductException(
                ProductException.PRODUCT_ACTION_INSTALL_FAILED,
                Log.ERROR,
                "Can not instantiate FileService");
        }

        // Do appropriate filesystem manipulation based on action
        if (action == COPY) {

            logEvent(this, Log.DBG, "Copying " + source + " to " + target);
            copy(source, target);

        } else if (action == DELETE) {

            logEvent(this, Log.DBG, "Deleting " + source);
            delete(source);

        } else if (action == MOVE) {

            logEvent(this, Log.DBG, "Moving " + source + " to " + target);
            copy(source, target);
            delete(source);

        } else {
            throw new ProductException(
                ProductException.PRODUCT_ACTION_INSTALL_FAILED,
                Log.ERROR,
                "Unkown FileSystemManipulation Action Specified");
        }

    }

    /**
     * Helper method to perform the file or directory copy process
     */
    private void copy(String source, String target) throws ProductException {
        try {

            if (fs.isDirectory(source))
                fs.copyDirectory(source, target, true, true);
            else
                fs.copyFile(source, target, true);
        } catch (ServiceException se) {
            throw new ProductException(
                ProductException.PRODUCT_ACTION_INSTALL_FAILED,
                Log.ERROR,
                "Unable to copy " + source + " to " + target);
        }
    }

    /**
     * Helper method to perform the file or directory delete process
     */
    private void delete(String source) throws ProductException {
        try {

            if (fs.isDirectory(source))
                fs.deleteDirectory(source, false, true);
            else
                fs.deleteFile(source);
        } catch (ServiceException se) {
            throw new ProductException(
                ProductException.PRODUCT_ACTION_INSTALL_FAILED,
                Log.ERROR,
                "Unable to delete " + source);
        }
    }

    /**
     * Reverse the action taken at install time. This involves deleting whatever
     * whatever was copied or moved.
     */
    public void uninstall(ProductActionSupport support) throws ProductException {

        if (runtime == UNINSTALL) {
            doAction(source, target, action);
        }

        // Cleanup from the action performed at Install time
        else if (action != DELETE) {
            target = resolveString(target);
            if (!FileUtils.isAbsolute(target)) {
                target =
                    FileUtils.appendSeparator(getParentComponent().getAbsoluteInstallLocation())
                        + target;
            }

            try {
                fs = (FileService) getService(FileService.NAME);
                if (fs.fileExists(target)) {
                    delete(target);
                }

            } catch (ServiceException se) {
                throw new ProductException(
                    ProductException.PRODUCT_ACTION_INSTALL_FAILED,
                    Log.ERROR,
                    "Can not delete " + target);
            }

        }

    }

    /**
     * Build in FileService and make sure that the source and target properties 
     * are not the empty string.
     */
    public void build(ProductBuilderSupport support) {

        // validate source property
        if (source.equals("")) {
            support.logEvent(this, Log.ERROR, "Source field can not be empty");
            support.setBuildCanceled(true);
        }

        // validate target property
        if (target.equals("")) {

            if ((action == MOVE) || (action == COPY)) {
                support.logEvent(
                    this,
                    Log.ERROR,
                    "Target field can not be empty with selected Action");
                support.setBuildCanceled(true);
            }
        }

        support.putRequiredService(FileService.NAME);
    }

    public String getTarget() {
        return target;
    }

    public void setTarget(String target) {
        this.target = target;
    }

    public String getSource() {
        return source;
    }

    public void setSource(String source) {
        this.source = source;
    }

    public void setAction(int action) {
        this.action = action;
    }

    public int getAction() {
        return action;
    }

    public int getRuntime() {
        return runtime;
    }

    public void setRuntime(int runtime) {
        this.runtime = runtime;
    }

}