/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor;

import java.io.IOException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.ParseException;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import sek.ase.auditor.collectors.SybAbstractCollector;
import sek.ase.auditor.utils.AseUtils;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.OpenSslAesUtil;
import sek.ase.auditor.utils.OpenSslAesUtil.DecryptionException;
import sek.ase.auditor.utils.StringUtil;

public class AppConfig
{
	private static Logger _logger = LogManager.getLogger();

	public static final String PROPKEY_dbms_server          = "dbms.server";
	public static final String DEFAULT_dbms_server          = null;
	
	public static final String PROPKEY_dbms_username        = "dbms.username";
	public static final String DEFAULT_dbms_username        = null;
	
	public static final String PROPKEY_dbms_password        = "dbms.password";
	public static final String DEFAULT_dbms_password        = null;

	//----------------------------------------------------------------
	// BEGIN: members
	//----------------------------------------------------------------
	public String _dbmsUsername;
	public String _dbmsPassword;
	public String _dbmsServer;
	public String _dbmsUrl;

	public String _configFile;
	public String _installAuditDir;
//	public String _logFile;
	//----------------------------------------------------------------
	// END: members
	//----------------------------------------------------------------


	//----------------------------------------------------------------
	// BEGIN: instance
	private static AppConfig _instance;

	public static AppConfig getInstance()
	{
		if (_instance == null)
		{
			throw new RuntimeException("AppConfig doesn't have an instance yet, please set with setInstance(instance).");
		}
		return _instance;
	}

	public static void setInstance(AppConfig instance)
	{
		_instance = instance;
	}

	public static boolean hasInstance()
	{
		return _instance != null;
	}
	// END: instance
	//----------------------------------------------------------------


	public AppConfig(CommandLine cmd, Configuration config)
	throws ParseException
	{
		setOptions(cmd, config);
		validate(); 
	}
	
	public void setOptions(CommandLine cmd, Configuration config)
	throws ParseException
	{
		_dbmsServer   = config.getProperty(PROPKEY_dbms_server  , DEFAULT_dbms_server);
		_dbmsUsername = config.getProperty(PROPKEY_dbms_username, DEFAULT_dbms_username);
		_dbmsPassword = config.getProperty(PROPKEY_dbms_password, DEFAULT_dbms_password);

		
		if ( StringUtil.isNullOrBlank(_dbmsServer  ) && cmd.hasOption('S') ) _dbmsServer      = cmd.getOptionValue('S');
		if ( StringUtil.isNullOrBlank(_dbmsUsername) && cmd.hasOption('U') ) _dbmsUsername    = cmd.getOptionValue('U');
		if ( StringUtil.isNullOrBlank(_dbmsPassword) && cmd.hasOption('P') ) _dbmsPassword    = cmd.getOptionValue('P');

//		if ( StringUtil.isNullOrBlank(_dbmsUsername) && cmd.hasOption('I') ) _installAuditDir = cmd.getOptionValue('I');
//		if ( StringUtil.isNullOrBlank(_dbmsUsername) && cmd.hasOption('C') ) _configFile      = cmd.getOptionValue('C');

		if (StringUtil.isNullOrBlank(_dbmsServer)) 
			throw new ParseException("Missing mandatory parameter '-S' or '--server' or in the config file as '" + PROPKEY_dbms_server   + "'.");

		if (StringUtil.isNullOrBlank(_dbmsUsername)) 
			throw new ParseException("Missing mandatory parameter '-U' or '--user' or in the config file as '"   + PROPKEY_dbms_username + "'.");

		//---------------------------------------------------
		// Check ServerName
		// -- Full URL
		// -- or HOST:PORT
		if (_dbmsServer.contains(":"))
		{
			// If this is a URL, no need to continue
			if (_dbmsServer.startsWith("jdbc:sybase:Tds:"))
			{
				_dbmsUrl    = _dbmsServer;
				_dbmsServer = StringUtils.substringBetween(_dbmsUrl, "jdbc:sybase:Tds:", ":");
			}
			// Is it host:port
			else if (StringUtil.count(":", _dbmsServer) == 1)
			{
				String[] sa = _dbmsServer.split(":");
				_dbmsUrl    = "jdbc:sybase:Tds:" + _dbmsServer; // set to: jdbc:sybase:Tds:HOSTNAME:PORT
				_dbmsServer = sa[0].trim();
			}
			else
			{
				throw new ParseException("The DBMS URL or ServerName format is not recognized. dbmsServer=|" + _dbmsServer + "|.");
			}
		}
		// Get host:port from interfaces file
		else
		{
			// Lookup the servername from the interfaces file
			String hostPortStr = AseUtils.getHostPortFromInterfacesFile(_dbmsServer);
			if (StringUtil.isNullOrBlank(hostPortStr))
			{
				_dbmsUrl = null;
				throw new RuntimeException("Can't find the server name '" + _dbmsServer + "' in the interfaces file. "
						+ "If you do not have a interfaces file, you can specify the URL "
						+ "as -S HOST:PORT "
						+ "or -S jdbc:sybase:Tds:HOST:PORT "
						+ "or in config file as: " + PROPKEY_dbms_server + " = jdbc:sybase:Tds:HOSTNAME:PORT");
			}
			_dbmsUrl = "jdbc:sybase:Tds:" + hostPortStr;
		}

		//---------------------------------------------------
		// Get password from ~/passwd.enc
		if (StringUtil.isNullOrBlank(_dbmsPassword))
		{
			try
			{
				_logger.info("Trying to get get password for user '" + _dbmsUsername + "', srvName='" + _dbmsServer + "' from file '" + OpenSslAesUtil.getPasswordFilename() + "'.");
				_dbmsPassword = OpenSslAesUtil.readPasswdFromFile(_dbmsUsername, _dbmsServer);
				
				if (StringUtil.hasValue(_dbmsPassword))
					_logger.info("Found password for user '" + _dbmsUsername + "', srvName='" + _dbmsServer + "' in file '" + OpenSslAesUtil.getPasswordFilename() + "'.");
				else
					_logger.info("No password for user '" + _dbmsUsername + "', srvName='" + _dbmsServer + "' was found in file '" + OpenSslAesUtil.getPasswordFilename() + "'.");
			}
			catch (IOException | DecryptionException ex)
			{
				_logger.warn("Problems to get password for user '" + _dbmsUsername + "', srvName='" + _dbmsServer + "' from file '" + OpenSslAesUtil.getPasswordFilename() + "'. Caught: " + ex);
			}
		}
		if (StringUtil.isNullOrBlank(_dbmsPassword)) 
			throw new ParseException("Missing mandatory parameter '-P' or '--passwd' or in the config file as '" + PROPKEY_dbms_password + "'.");


System.out.println(SybAbstractCollector.PROPKEY_dbms_server   + " = " +_dbmsUrl);
System.out.println(SybAbstractCollector.PROPKEY_dbms_username + " = " +_dbmsUsername);
System.out.println(SybAbstractCollector.PROPKEY_dbms_password + " = " +_dbmsPassword);
		//---------------------------------------------------
		// set DBMS configuration to ALL internal properties used
		config.setProperty( SybAbstractCollector.PROPKEY_dbms_server  , _dbmsUrl);
		config.setProperty( SybAbstractCollector.PROPKEY_dbms_username, _dbmsUsername);
		config.setProperty( SybAbstractCollector.PROPKEY_dbms_password, _dbmsPassword);

	}

	public void validate()
	throws ParseException
	{
	}

	public void printConfigToLogFile()
	{
		_logger.info("Configuration: ");
		_logger.info("    ...");
		_logger.info("    ...");
	}

//	@Override
//	public String toString()
//	{
//		PRINT CONFIG
//	}	
}
