/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.collectors;

import java.net.URI;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import sek.ase.auditor.Version;
import sek.ase.auditor.collectors.SybSysProcessesCache.SybSysProcessesEntry;
import sek.ase.auditor.utils.AseUtils;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.utils.TimeUtils;
import sek.ase.auditor.wqs.WriterQueueContainer;
import sek.ase.auditor.wqs.WriterQueueHandler;

/**
 * Abstract Collector that can be used as a base for other collectors
 * 
 * @author goran
 */
public abstract class SybAbstractCollector
implements Runnable
{
	private static Logger _logger = LogManager.getLogger();

	public static final String PROPKEY_skipLoginNameRegex   = "SybCollector.skip.loginName.regex";
	public static final String DEFAULT_skipLoginNameRegex   = null;
	
	public static final String PROPKEY_skipProgramNameRegex = "SybCollector.skip.programName.regex";
	public static final String DEFAULT_skipProgramNameRegex = "(AseTune.*)";
//	public static final String DEFAULT_skipProgramNameRegex = null;

	public static final String PROPKEY_sleepTimeInSeconds   = "SybCollector.sleepTime.seconds";
	public static final int    DEFAULT_sleepTimeInSeconds   = 1;

//	public static final String PROPKEY_NULL_REPLACE = "SybCollector.replace.null.with";
//	public static final String DEFAULT_NULL_REPLACE = "(NULL)";

//	public static final String NULL_REPLACE = Configuration.getCombinedConfiguration().getProperty(PROPKEY_NULL_REPLACE, DEFAULT_NULL_REPLACE);
	
	public static final String PROPKEY_statisticsMessagePeriodSeconds = "SybCollector.statistics.message.period.seconds";
	public static final int    DEFAULT_statisticsMessagePeriodSeconds = 300;

	public static final String PROPKEY_dbms_server          = "SybCollector.dbms.server";
	public static final String DEFAULT_dbms_server          = null;
	
//	public static final String PROPKEY_dbms_url             = "SybCollector.dbms.url";
//	public static final String DEFAULT_dbms_url             = null;
	
	public static final String PROPKEY_dbms_username        = "SybCollector.dbms.username";
	public static final String DEFAULT_dbms_username        = null;
	
	public static final String PROPKEY_dbms_password        = "SybCollector.dbms.password";
	public static final String DEFAULT_dbms_password        = null;

	public static final String PROPKEY_dbms_connectFaileRetrySleepTimeMs      = "SybCollector.dbms.connect.fail.retry.sleepTimeMs";
	public static final int    DEFAULT_dbms_connectFaileRetrySleepTimeMs      = 10_000;

	public static final String PROPKEY_dbms_connectFaileRetryMaxWaitTimeInSec = "SybCollector.dbms.connect.fail.retry.maxWaitTimeInSec";
	public static final int    DEFAULT_dbms_connectFaileRetryMaxWaitTimeInSec = 7200;
	
	public static final String PROPKEY_dbms_connectFaileRetryCount            = "SybCollector.dbms.connect.fail.retry.count";
	public static final int    DEFAULT_dbms_connectFaileRetryCount            = 100_000;
	
	public static final String  PROPKEY_dbms_exitIfFirstConnFails             = "SybCollector.dbms.exit.if.first.connect.fails";
	public static final boolean DEFAULT_dbms_exitIfFirstConnFails             = true;
	
	private String _skipLoginNameRegex   = DEFAULT_skipLoginNameRegex; 
	private String _skipProgramNameRegex = DEFAULT_skipProgramNameRegex; 

	private Pattern _skipLoginNamePattern   = _skipLoginNameRegex   == null ? null : Pattern.compile(_skipLoginNameRegex); 
	private Pattern _skipProgramNamePattern = _skipProgramNameRegex == null ? null : Pattern.compile(_skipProgramNameRegex);; 

	private Configuration _conf;

	private Thread     _thread;
	private Connection _conn;
	private boolean    _isNewConnection = false;
	private String     _aseServerName;
	private String     _aseHostName;

	private boolean _running;
	private int     _sleepTimeSec = 1;
	
	private ConcurrentHashMap<String, AtomicInteger> _skipLoginNameMapCounter   = new ConcurrentHashMap<>();
	private ConcurrentHashMap<String, AtomicInteger> _skipProgramNameMapCounter = new ConcurrentHashMap<>();

	// Incremented often
	private AtomicLong _totalQueueCount = new AtomicLong();
	private AtomicLong _totalSkipCount  = new AtomicLong();

	// Used to DIFF calculate how many in period
	private long _lastTotalQueueCount = 0;
	private long _lastTotalSkipCount  = 0;

	
	private long _statisticsMessageLastTime  = System.currentTimeMillis();
	private long _statisticsMessagePeriodSec = DEFAULT_statisticsMessagePeriodSeconds;

	public SybAbstractCollector()
	{
	}

	/**
	 * Initialize configurations etc...
	 * 
	 * @param config
	 * @throws Exception 
	 */
	public void init(Configuration config) 
	throws Exception
	{
		_conf = (config != null) ? config : Configuration.getCombinedConfiguration();
		
		// Initialize some values based on configuration
		_skipLoginNameRegex   = _conf.getProperty(PROPKEY_skipLoginNameRegex  , DEFAULT_skipLoginNameRegex); 
		_skipProgramNameRegex = _conf.getProperty(PROPKEY_skipProgramNameRegex, DEFAULT_skipProgramNameRegex);

		_skipLoginNamePattern   = StringUtil.isNullOrBlank(_skipLoginNameRegex)   ? null : Pattern.compile(_skipLoginNameRegex); 
		_skipProgramNamePattern = StringUtil.isNullOrBlank(_skipProgramNameRegex) ? null : Pattern.compile(_skipProgramNameRegex);; 

		// Sleep time
		setSleepTimeSec( getConfig().getIntProperty(PROPKEY_sleepTimeInSeconds, DEFAULT_sleepTimeInSeconds) );

		// Statistics Message Period
		_statisticsMessagePeriodSec = getConfig().getLongProperty(PROPKEY_statisticsMessagePeriodSeconds, DEFAULT_statisticsMessagePeriodSeconds);
		
		setDbmsUrl();
		setDbmsUsername();
		setDbmsPassword();
	}
	
	/**
	 * Print module configuration to the application log
	 */
	public abstract void printConfig();

	/**
	 * Print base configuration, usually called from 'printConfig()' in any subclass
	 * @param len
	 */
	public void printBaseConfig(int len)
	{
		_logger.info("   > Base Cfg for '" + getName() + "'");
		
		_logger.info("                   " + StringUtil.left(PROPKEY_skipLoginNameRegex                    , len) + " = " + _skipLoginNameRegex);
		_logger.info("                   " + StringUtil.left(PROPKEY_skipProgramNameRegex                  , len) + " = " + _skipProgramNameRegex);
		_logger.info("                   " + StringUtil.left(PROPKEY_statisticsMessagePeriodSeconds        , len) + " = " + _statisticsMessagePeriodSec);

		_logger.info("                   " + StringUtil.left(PROPKEY_dbms_exitIfFirstConnFails             , len) + " = " + getConfig().getBooleanProperty(PROPKEY_dbms_exitIfFirstConnFails             , DEFAULT_dbms_exitIfFirstConnFails));
		_logger.info("                   " + StringUtil.left(PROPKEY_dbms_connectFaileRetrySleepTimeMs     , len) + " = " + getConfig().getIntProperty    (PROPKEY_dbms_connectFaileRetrySleepTimeMs     , DEFAULT_dbms_connectFaileRetrySleepTimeMs));
		_logger.info("                   " + StringUtil.left(PROPKEY_dbms_connectFaileRetryMaxWaitTimeInSec, len) + " = " + getConfig().getIntProperty    (PROPKEY_dbms_connectFaileRetryMaxWaitTimeInSec, DEFAULT_dbms_connectFaileRetryMaxWaitTimeInSec));
		_logger.info("                   " + StringUtil.left(PROPKEY_dbms_connectFaileRetryCount           , len) + " = " + getConfig().getIntProperty    (PROPKEY_dbms_connectFaileRetryCount           , DEFAULT_dbms_connectFaileRetryCount));
	}
	
	/**
	 * Get the passed configuration object
	 * 
	 * @return
	 */
	public Configuration getConfig()
	{
		return _conf;
	}

	/**
	 * Get the name of thismodule
	 * 
	 * @return
	 */
	public String getName()
	{
		return this.getClass().getSimpleName();
	}

	/**
	 * Start this module
	 */
	public void start()
	{
		_thread = new Thread(this, getName());
		_thread.setDaemon(true);
		_thread.start();
	}
	
	/**
	 * Stop this module
	 */
	public void stop()
	{
		if (_thread != null)
		{
			setRunning(false);
			_thread.interrupt();
		}
	}

	/**
	 * Used to "mark"/notify the thread to stop...
	 * @param running
	 */
	public void setRunning(boolean running)
	{
		_running = running;
	}

	/**
	 * Check if this module is in a "running" state
	 * @return
	 */
	public boolean isRunning()
	{
		return _running;
	}

	/**
	 * Get how long we should sleep between samples
	 * @return
	 */
	public int getSleepTimeSec()
	{
		return _sleepTimeSec;
	}

	/**
	 * Set how long we should sleep between samples
	 * @return
	 */
	public void setSleepTimeSec(int sleepTimeSec)
	{
		_sleepTimeSec = sleepTimeSec;
	}

	/**
	 * Get data from any "child" classes that implements a Collector
	 * <p>
	 * Big picture
	 * <ul>
	 *   <li>call getData(), which is implemented by any subclass</li>
	 *   <li>Add the container to the WriterQueueHandler, which will write to all registered writers</li>
	 *   <li>Sleep a while before we do another "getData()"</li>
	 * </ul>
	 */
	@Override
	public void run()
	{
		_running = true;
		
		int loopCount = 0;

		_logger.info("Starting Thread for: " + getName());
		while(_running)
		{
			try
			{
				// Sleep for X milliseconds before fetching new records
				if (loopCount > 0)
					Thread.sleep(getSleepTimeSec() * 1000);
				loopCount++;

				// Is it time to print some statistical messages (this should be at the TOP of the loop, otherwise it wont be printed if no data is fetched.)
				if (TimeUtils.secondsDiffNow(_statisticsMessageLastTime) >= _statisticsMessagePeriodSec)
				{
					printStatisticsMessage();
				}

				// Get data
				WriterQueueContainer container = getData();

				// if no results, continue with next
				if ( container == null || (container != null && container.isEmpty()) )
					continue;
					
				// if we are about to STOP the service
				if ( ! isRunning() )
				{
					_logger.info("The service is about to stop, discarding a consume(AuditRecord) queue entry.");
					continue;
				}

				// Increment how many Entries we have "sent" to the WriterQueue
				_totalQueueCount.getAndAdd(container.size());
				
				// Post the data to the Writer Queue, which will write the container to each registered writer
				WriterQueueHandler.getInstance().add(container);
			}
			catch (InterruptedException ex) 
			{
				_logger.info("Received 'InterruptedException' in '" + getName() + "', time to stop...");
				_running = false;
			}
			catch(Throwable t)
			{
				_logger.error("Found some issue in '" + getName() + "', continuing anyway... Caught: " + t, t);
			}
		}
		_logger.info("Ending Thread for " + getName());
	}

	/**
	 * Print some statistical messages to the Application Log
	 */
	protected void printStatisticsMessage()
	{
		String statMsg = getStatisticsMessageAndReset();
		_logger.info("STATISTICS[" + getName() + "]: " + StringUtil.trim(statMsg));
	}

	/**
	 * Get a statistics String to be written to the Application Log
	 * @return
	 */
	protected String getStatisticsMessageAndReset()
	{
		// First reset the time
		_statisticsMessageLastTime = System.currentTimeMillis();

		long tmpTotalQueueCount = _totalQueueCount.get();
		long tmpTotalSkipCount  = _totalSkipCount .get();

		long diffTotalQueueCount = tmpTotalQueueCount - _lastTotalQueueCount;
		long diffTotalSkipCount  = tmpTotalSkipCount  - _lastTotalSkipCount;

		_lastTotalQueueCount = tmpTotalQueueCount;
		_lastTotalSkipCount  = tmpTotalSkipCount;
		
		return ""
				+   "TotalQueueCount="          + tmpTotalQueueCount + " (" + diffTotalQueueCount + ")"
				+ ", TotalSkipCount="           + tmpTotalSkipCount  + " (" + diffTotalSkipCount + ")"
				+ ", SkipLoginCounters="        + _skipLoginNameMapCounter    // a Map of <LoginName,   skipCnt>
				+ ", SkipProgramNameCounters="  + _skipProgramNameMapCounter  // a Map of <ProgramName, skipCnt>
				;
	}

	/**
	 * Implemented by any subclass that does the "collection" of data from the DBMS and produces a Container object with X number of records
	 * @return
	 * @throws SQLException
	 * @throws InterruptedException
	 */
	protected abstract WriterQueueContainer getData()
	throws SQLException, InterruptedException;
	

	
	/**
	 * Quick check if the LOGIN name should be included (this is executed <b>early</b> (when reading the ResultSet)
	 * <p>
	 * NOTE: In here you can NOT lookup SybSysProccessCache (since it may call the DBMS)
	 * 
	 * @param loginName
	 * 
	 * @return true if include, false if we want to discard this one
	 */
	public boolean includeLoginName(String loginName)
	{
		if (_skipLoginNamePattern != null && loginName != null)
		{
			boolean skipLogin = _skipLoginNamePattern.matcher(loginName).matches();
			if ( skipLogin )
			{
				_totalSkipCount.getAndIncrement();

				AtomicInteger counter = _skipLoginNameMapCounter.get(loginName);
				if (counter == null)
				{
					counter = new AtomicInteger(0);
					_skipLoginNameMapCounter.put(loginName, counter);
				}
				counter.getAndIncrement();
				
				if (_logger.isDebugEnabled())
					_logger.debug("Skipping login name '" + loginName + "', due to match in regex '" + _skipLoginNamePattern.pattern() + "'.");

				return false;
			}
			else
			{
				return true;
			}
		}

		return true;
	}

	/**
	 * Check if the Program Name should be included or discarded
	 * <p>
	 * This is executed <i>near end</i> so here we can lookup information from the SybSysProccessCache
	 * 
	 * @param conn    JDBC Connection (used to lookup/refresh information from sysprocesses if not found in cache)
	 * @param spid    Server Process ID
	 * @param kpid    Use -1 if KPID is not known (then it will grab last connected/known SPID)
	 * 
	 * @return true if include, false if we want to discard this one
	 */
	public boolean includeProgramName(Connection conn, int spid, int kpid)
	{
		// In here we could lookup stuff from SybSysProccessCache... like ApplicationName / HostName / IpAddress etc
		try
		{
			SybSysProcessesEntry sysProcEntry = SybSysProcessesCache.getInstance().getRecord(conn, spid, kpid);

			if (sysProcEntry != null)
			{
				String sysProcProgramName = sysProcEntry.getProgramName();
				if (sysProcProgramName != null)
				{
					if (Version.getAppName().equals(sysProcProgramName) )
					{
						if (_logger.isDebugEnabled())
							_logger.debug("Skipping program name '" + Version.getAppName() + "', since this is THIS program.");

						// Increment statistics
						privateIncrementSkipCounterForProgName(sysProcProgramName);

						// Skip this
						return false;
					}

					if (_skipProgramNamePattern != null && _skipProgramNamePattern.matcher(sysProcProgramName).matches())
					{
						if (_logger.isDebugEnabled())
							_logger.debug("Skipping program name '" + sysProcProgramName + "', due to match in regex '" + _skipProgramNamePattern.pattern() + "'.");

						// Increment statistics
						privateIncrementSkipCounterForProgName(sysProcProgramName);

						// Skip this
						return false;
					}
				}
			}

			return true;
		}
		catch (SQLException ex)
		{
			_logger.error("includeProgramName(spid=" + spid + ", kpid=" + kpid + "): Problems looking up information from SybSysProcessesCache. Action=The entry will be INCLUDED. " + AseUtils.sqlExceptionOrWarningAllToString(ex));
			return true;
		}
	}
	
	/** INTERNAL */
	private void privateIncrementSkipCounterForProgName(String progName)
	{
		_totalSkipCount.getAndIncrement();
		
		// Increment "skip" Counter, for a specific name
		AtomicInteger counter = _skipProgramNameMapCounter.get(progName);
		if (counter == null)
		{
			counter = new AtomicInteger(0);
			_skipProgramNameMapCounter.put(progName, counter);
		}
		counter.getAndIncrement();
	}
	

	/**
	 * Set Sybase properties 'clientname, clienthostname, clientapplname' in sysprocesses (for tracability)
	 * 
	 * @param conn
	 * @param clientname
	 * @param clienthostname
	 * @param clientapplname
	 * @throws SQLException
	 */
	private void setClientProps(Connection conn, String clientname, String clienthostname, String clientapplname)
	throws SQLException
	{
		String sql = "";
		
		if (clientname     != null) sql += "set clientname     '" + clientname     + "' \n";
		if (clienthostname != null) sql += "set clienthostname '" + clienthostname + "' \n";
		if (clientapplname != null) sql += "set clientapplname '" + clientapplname + "' \n";
		
		try (Statement stmnt = conn.createStatement())
		{
			stmnt.executeUpdate(sql);
		}
	}

	/**
	 * Simply execute 'use dbname'
	 * 
	 * @param conn
	 * @param dbname
	 * @throws SQLException
	 */
	private void setDbname(Connection conn, String dbname)
	throws SQLException
	{
		_logger.info(getName() + ": Changing database context to '" + dbname + "'.");
		String sql = "use " + dbname;

		try (Statement stmnt = conn.createStatement())
		{
			stmnt.executeUpdate(sql);
		}
	}

	/**
	 * Get DBMS Server info and store it in instance variable
	 * <p>
	 * What we typically get is:
	 * <ul>
	 *   <li>@@servername  -- which returns the name of the DBMS instance</li>
	 *   <li>asehostname() -- which returns on what hostname the DBMS is running on</li>
	 * </ul>
	 * 
	 * @param conn
	 */
	private void getServerInfo(Connection conn)
	{
		String sql = "select @@servername, asehostname()";
		
		try (Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql))
		{
			while (rs.next())
			{
				_aseServerName = rs.getString(1);
				_aseHostName   = rs.getString(2);
			}
		}
		catch(SQLException ex)
		{
			_logger.warn(getName() + ": Problems getting ASE Server information using SQL='" + sql + "'. " + AseUtils.sqlExceptionOrWarningAllToString(ex) + ". Lets try with just 'select @@servername' and getting hostname from the URL.");
			_aseServerName = "-UNKNOWN-";
			_aseHostName   = "-UNKNOWN-";
			
			//-------------------------------------------------------------------------------------------------
			// FALLBACK - We might not be allowed to do 'asehostname()'... so lets try with just '@@servername'
			//-------------------------------------------------------------------------------------------------
			sql = "select @@servername";
			try (Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql))
			{
				while (rs.next())
				{
					_aseServerName = rs.getString(1);
				}
				// Possibly we can parse the: URL from the connection to simulate 'asehostname()'
				// NOTE: The below has NOT yet been 100% tested
				String jdbcUrl = conn.getMetaData().getURL();
				if (jdbcUrl != null && jdbcUrl.startsWith("jdbc:sybase:Tds:"))
				{
					// note: URI needs '//' before the hostname, and only 1 "schema name" (so strip off 'jdbc:sybase:Tds:' and replace it with 'sybase:'
					String cleanURI = "sybase://" + jdbcUrl.substring("jdbc:sybase:Tds:".length()); 
					URI uri = URI.create(cleanURI);
					//System.out.println("jdbcUrl=|" + jdbcUrl + "|, cleanURI=|" + cleanURI + "|, uri.getHost()=|" + uri.getHost() + "|");
					_aseHostName = uri.getHost();
				}
			}
			catch(SQLException ex2)
			{
				_logger.warn(getName() + ": Problems getting ASE Server information using SQL='" + sql + "'. " + AseUtils.sqlExceptionOrWarningAllToString(ex2) + ". Hmmm... cant even do 'select @@servername', something must be OFF here...");
			}
		}
	}

	public String getAseServerName() { return _aseServerName; }
	public String getAseHostName()   { return _aseHostName; }

//	public static WriterQueueContainer createContainer()
//	{
//		HeaderInfo headerInfo = new HeaderInfo(new Timestamp(System.currentTimeMillis()), getAseServerName(), getAseHostName());
//		WriterQueueContainer container = new WriterQueueContainer(headerInfo);
//		
//	}

	/**
	 * If getConnection() made a nw connection.
	 * <p>
	 * NOTE: The status is reset every time you call getConnection()
	 * @return
	 */
	public boolean isNewConnection()
	{
		return _isNewConnection;
	}

	/**
	 * Get a previously used JDBC/DBMS Connection, or create a new connection (if none exists or it's not valid)
	 * 
	 * @param dbname    Name of the database we will "use"
	 * 
	 * @return The JDBC Connection
	 * @throws SQLException
	 * @throws InterruptedException
	 */
	public Connection getConnection(String dbname)
	throws SQLException, InterruptedException
	{
		return private_getConnection(dbname, false);
	}

	/**
	 * If we at init() want to test the connection, and possibly throw 
	 * 
	 * @param dbname    Name of the database we will "use"
	 * 
	 * @return true on OK
	 * @throws Exception on issues
	 */
	public boolean atInitTestConnection(String dbname)
	throws Exception
	{
		Connection conn = null;
		try
		{
			conn = private_getConnection(dbname, true);
			return true;
		}
		catch (SQLException | InterruptedException ex) 
		{
			boolean exitIfFirstConnFails = getConfig().getBooleanProperty(PROPKEY_dbms_exitIfFirstConnFails, DEFAULT_dbms_exitIfFirstConnFails);
			if (exitIfFirstConnFails)
			{
				throw new Exception("Not possible to Connect to the DBMS. url='" + getDbmsUrl() + "'. Can NOT continue... Possibly see the errorlog for more details. If you want the service and try new connections later. Enable config '" + PROPKEY_dbms_exitIfFirstConnFails + " = false'. Caught: " + ex, ex);
			}
			else
			{
				_logger.warn("Initial DBMS Connection-Test failed. Normally this would cause the service to NOT startup. But config '" + PROPKEY_dbms_exitIfFirstConnFails + " = false' so, lets continue to start, hopefully any following connection attempt(s) will be successfull.");
				return true;
			}
		}
		finally
		{
			if (conn != null)
			{
				try { conn.close(); }
				catch (SQLException ignore) {}
			}
		}
	}

	/**
	 * Get a previously used JDBC/DBMS Connection, or create a new connection (if none exists or it's not valid)
	 * 
	 * @param dbname    Name of the database we will "use"
	 * 
	 * @return The JDBC Connection
	 * @throws SQLException
	 * @throws InterruptedException
	 */
	private Connection private_getConnection(String dbname, boolean atInitTestConnection)
	throws SQLException, InterruptedException
	{
		_isNewConnection = false;

		// If we have a connection, check that it's valid (in a try block)
		boolean isConnectionValid = false;
		if (_conn != null)
		{
			try
			{
				isConnectionValid = _conn.isValid(1);
			}
			catch (SQLException ex)
			{
				isConnectionValid = false;
				_logger.info("Closing current DBMS Connection, since it wasn't valid anymore. Caught: ErrorCode=" + ex.getErrorCode() + ", SqlState=" + ex.getSQLState()+ ", Text=|" + ex.getMessage()+ "|, Exception: " + ex);

				// Close the connection, and set to null
				try 
				{
					if (_conn != null)
						_conn.close();
				}
				catch (SQLException ignore)
				{
					// ignore
				}
				finally 
				{
					_conn = null;
				}
			}
		}

		// Connect, if not a valid connection
		if ( ! isConnectionValid )
		{
//			if (_conn != null)
//				_conn.close();

			// Create a connection string URL
			String url = getDbmsUrl();

			// Set various Connection Properties
			Properties connProps = new Properties();
			connProps.put("user",     getDbmsUsername());
			connProps.put("password", getDbmsPassword());

			connProps.put("ENCRYPT_PASSWORD", "true");
			connProps.put("APPLICATIONNAME",  Version.getAppName());

			Properties strippedConnProps = new Properties();
			for (Entry<Object, Object> entry : connProps.entrySet())
				strippedConnProps.setProperty(entry.getKey().toString(), entry.getValue().toString());

			strippedConnProps.setProperty("password", "*******");

			
			// Grab a new connection, with retries
			SQLException finalException = null;
			int loopLimit        = getConfig().getIntProperty(PROPKEY_dbms_connectFaileRetryCount           , DEFAULT_dbms_connectFaileRetryCount);
			int sleepTime        = getConfig().getIntProperty(PROPKEY_dbms_connectFaileRetrySleepTimeMs     , DEFAULT_dbms_connectFaileRetrySleepTimeMs);
			int maxWaitTimeInSec = getConfig().getIntProperty(PROPKEY_dbms_connectFaileRetryMaxWaitTimeInSec, DEFAULT_dbms_connectFaileRetryMaxWaitTimeInSec);

			// For simpler print logic
			loopLimit = loopLimit + 1;
			long loopStartTime = System.currentTimeMillis();

			for (int i=1; i<loopLimit; i++)
			{
				try
				{
					finalException = null;;
					Connection conn = DriverManager.getConnection(url, connProps);

//new Exception("getConnection(): DUMMY").printStackTrace();
					_logger.info("Succeeded connecting to URL '" + url + "', Properties='" + strippedConnProps + "'.");

					// Use the correct database
					if (StringUtil.hasValue(dbname))
						setDbname(conn, dbname);

					// set some extra stuff in the DBMS
					setClientProps(conn, getName(), null, null);
					
					// Get some information on the server... @@servername and asehostname()
					getServerInfo(conn);

					// Do local checks/adjustments
					onConnect(conn);

					// Set Connection
					_conn = conn;
					_isNewConnection = true;
					
					// BREAK the loop on SUCCESS
					break;
				}
				catch (SQLException ex)
				{
					finalException = ex;
					
//					if (ex.getMessage().startsWith("ASE Configuration '"))
//					{
//						_logger.error("Problems Connection DBMS, Caught: " + ex);
//						break;
//					}
					
					// If we are testing the connection, and has issues... Get out of here
					if (atInitTestConnection)
					{
						_logger.error("Problems Connection DBMS. URL='" + url + "', Properties='" + strippedConnProps + "'. Exception: " + ex);
						throw finalException;
					}
					
					long secondsWaiting = TimeUtils.secondsDiffNow(loopStartTime);
					_logger.error("Problems Connection DBMS, atTry=" + i + ", tryLimit=" + loopLimit + ", maxWaitTimeInSec=" + maxWaitTimeInSec + ", secondsWaiting=" + secondsWaiting + ". URL='" + url + "', Properties='" + strippedConnProps + "'. Exception: " + ex);
					
					if (secondsWaiting > maxWaitTimeInSec)
					{
						throw new InterruptedException("Leaving Connection retry logic. Reached Connect Retry setting " + PROPKEY_dbms_connectFaileRetryMaxWaitTimeInSec + " = " + maxWaitTimeInSec + ". ");
					}

					// Sleep and try again
					Thread.sleep(sleepTime);
				}
			}
			
			if (finalException != null)
				throw finalException;
		}

		return _conn;
	}

	
	public void onConnect(Connection conn)
	throws SQLException
	{
	}

	private String _dbmsUrl = null;
	private String _dbmsUsername = null;
	private String _dbmsPassword = null;
//	private String _dbmsUrl      = "jdbc:sybase:Tds:gorans-ub3.home:1600";
//	private String _dbmsUsername = "audit_user";
//	private String _dbmsPassword = "sybase11";

	/** Get the DBMS URL to connect to */
	public String getDbmsUrl()
	{
		return _dbmsUrl;
	}
	/** Get the USERNAME used to connect to the DBMS */
	public String getDbmsUsername()
	{
		return _dbmsUsername;
	}
	/** Get the PASSWORD used to connect to the DBMS */
	public String getDbmsPassword()
	{
		return _dbmsPassword;
	}



	public void setDbmsUrl()
	{
		String srvName = _conf.getProperty(PROPKEY_dbms_server, DEFAULT_dbms_server);

		if (StringUtil.isNullOrBlank(srvName))
			throw new RuntimeException("Can't find the mandatory property '" + PROPKEY_dbms_server + "'.");

		// If this is a URL, no need to continue
		if (srvName.startsWith("jdbc:sybase:Tds:"))
		{
			_dbmsUrl = srvName;
			return;
		}

		// Lookup the servername from the interfaces file
		String hostPortStr = AseUtils.getHostPortFromInterfacesFile(srvName);
		if (StringUtil.isNullOrBlank(hostPortStr))
		{
			_dbmsUrl = null;
			throw new RuntimeException("Can't find the server name '" + srvName + "' in the interfaces file. If you do not have a interfaces file, you can specify the URL in property: " + PROPKEY_dbms_server + " = jdbc:sybase:Tds:HOSTNAME:PORT");
		}
		_dbmsUrl = "jdbc:sybase:Tds:" + hostPortStr;
	}

	public void setDbmsUsername()
	{
		String username = _conf.getProperty(PROPKEY_dbms_username, DEFAULT_dbms_username);

		if (StringUtil.isNullOrBlank(username))
			throw new RuntimeException("Can't find the mandatory property '" + PROPKEY_dbms_username + "'.");

		_dbmsUsername = username;
	}

	public void setDbmsPassword()
	{
		String password = _conf.getProperty(PROPKEY_dbms_password, DEFAULT_dbms_password);

		if (StringUtil.isNullOrBlank(password))
			throw new RuntimeException("Can't find the mandatory property '" + PROPKEY_dbms_password + "'.");

		_dbmsPassword = password;
	}


//	public void setDbmsUrl()
//	{
//		String srvName = _conf.getProperty(PROPKEY_dbms_server, DEFAULT_dbms_server);
//
//System.out.println("setDbmsUrl(): srvName='"+srvName+"'");
//		if (StringUtil.isNullOrBlank(srvName))
//			throw new RuntimeException("Can't find the mandatory property '" + PROPKEY_dbms_server + "'.");
//
//		// If this is a URL, no need to continue
//		if (srvName.startsWith("jdbc:sybase:Tds:"))
//		{
//			_dbmsUrl = srvName;
//			return;
//		}
//		
//		// Lookup the servername from the interfaces file
//		String hostPortStr = AseUtils.getHostPortFromInterfacesFile(srvName);
//		if (StringUtil.isNullOrBlank(hostPortStr))
//		{
//			_dbmsUrl = null;
//			throw new RuntimeException("Can't find the server name '" + srvName + "' in the interfaces file. If you do not have a interfaces file, you can specify the URL in property: " + PROPKEY_dbms_server + " = jdbc:sybase:Tds:HOSTNAME:PORT");
//		}
//		_dbmsUrl = "jdbc:sybase:Tds:" + hostPortStr;
//	}
//	public void setDbmsUsername()
//	{
//		String username = _conf.getProperty(PROPKEY_dbms_username, DEFAULT_dbms_username);
//
//		if (StringUtil.isNullOrBlank(username))
//			throw new RuntimeException("Can't find the mandatory property '" + PROPKEY_dbms_username + "'.");
//
//		_dbmsUsername = username;
//	}
//	public void setDbmsPassword()
//	{
//		String password = _conf.getProperty(PROPKEY_dbms_password, DEFAULT_dbms_password);
//
//		// Not found: Get it from the encrypted file: ~/passwd.enc
//		if (StringUtil.isNullOrBlank(password))
//		{
//			String username = getDbmsUsername();
//			String srvName = _conf.getProperty(PROPKEY_dbms_server, DEFAULT_dbms_server);
//
//			// If this is a URL, try to "sus" out the servername
//			if (srvName.startsWith("jdbc:sybase:Tds:"))
//			{
//				srvName = StringUtils.substringBetween(srvName, "jdbc:sybase:Tds:", ":");
//			}
//			
//			try
//			{
//				_logger.info("Trying to get get password for user '" + username + "', srvName='" + srvName + "' from file '" + OpenSslAesUtil.getPasswordFilename() + "'.");
//				password = OpenSslAesUtil.readPasswdFromFile(username, srvName);
//				
//				if (StringUtil.hasValue(password))
//					_logger.info("Found password for user '" + username + "', srvName='" + srvName + "' in file '" + OpenSslAesUtil.getPasswordFilename() + "'.");
//				else
//					_logger.info("No password for user '" + username + "', srvName='" + srvName + "' was found in file '" + OpenSslAesUtil.getPasswordFilename() + "'.");
//			}
//			catch (IOException | DecryptionException ex)
//			{
//				_logger.warn("Problems to get password for user '" + username + "', srvName='" + srvName + "' from file '" + OpenSslAesUtil.getPasswordFilename() + "'. Caught: " + ex);
//			}
//		}
//
//		if (StringUtil.isNullOrBlank(password))
//			throw new RuntimeException("Can't find the mandatory property '" + PROPKEY_dbms_password + "'.");
//
//		_dbmsPassword = password;
//	}
}
