/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.collectors;

import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonGenerator;

import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.utils.TimeUtils;

public class SybSysProcessesCache
{
	private static Logger _logger = LogManager.getLogger();

	public static final String PROPKEY_refreshThresholdInMs   = "SybSysProcessesCache.refreshThresholdInMs";
	public static final long   DEFAULT_refreshThresholdInMss  = 500;

	public static final String PROPKEY_cacheTimeToLiveInSec  = "SybSysProcessesCache.timeToLive.seconds";
	public static final long   DEFAULT_cacheTimeToLiveInSec  = 180;
	
	/*---------------------------------------------------
	** class members
	**---------------------------------------------------
	*/
	// implements singleton pattern
	private static SybSysProcessesCache _instance = null;

	// Whats the MAX time in seconds before the cache needs to be refreshed
	private long _cacheTtlInSec        = Configuration.getCombinedConfiguration().getLongProperty(PROPKEY_cacheTimeToLiveInSec, DEFAULT_cacheTimeToLiveInSec);

	private long _refreshThresholdInMs = Configuration.getCombinedConfiguration().getLongProperty(PROPKEY_refreshThresholdInMs, DEFAULT_refreshThresholdInMss);

	// What was a "refresh" last done
	private long _lastTotalRefreshTime  = -1;

	private long _totalRefreshCount  = 0;
//	private long _singleRefreshCount = 0;
	
	// What was a "refresh" last done
//	private long _minSecondsBetweenRefresh = 30_000; // 30 seconds
	
	// The cache Map<spid, Map<kpid, entry>>
	//          SPID         KPID
	private Map<Integer, Map<Integer, SybSysProcessesEntry>> _cache = new ConcurrentHashMap<>();

	
	//////////////////////////////////////////////
	//// Instance
	//////////////////////////////////////////////
	public static SybSysProcessesCache getInstance()
	{
		if (_instance == null)
			_instance = new SybSysProcessesCache();

		return _instance;
	}

//	public static boolean hasInstance()
//	{
//		return (_instance != null);
//	}
//
//	public static void setInstance(SybSysProcessesCache inst)
//	{
//		_instance = inst;
//	}


	/**
	 * Check if a SPID/KPID exists in the cache <br>
	 * Primarily looks up in the cache, but the cache is refreshed every X minute
	 * 
	 * @param conn
	 * @param spid
	 * @param kpid
	 * @return
	 */
	public boolean exists(Connection conn, int spid, int kpid)
	throws SQLException
	{
		// make sure that the cache is REFRESHED every X minute/second
		if (isCacheExpired())
		{
			refreshCache(conn);
		}

		return getCachedRecord(spid, kpid) != null;
	}

	/**
	 * Check if the Time To Live for the cache has expired
	 * @return
	 */
	public boolean isCacheExpired()
	{
		return TimeUtils.secondsDiffNow(_lastTotalRefreshTime) > _cacheTtlInSec;
	}

	/**
	 * Refresh the cache
	 * 
	 * @param conn
	 * @throws SQLException
	 */
	private synchronized void refreshCache(Connection conn)
	throws SQLException
	{
		// should we also get SRV LOGIN ROLES and add the "role names" to each sysprocesses entry ???
		Map<Integer, List<String>> suidToRoleNames = getSrvRolesMap(conn);
		
		String sql = ""
				+ "select \n"
				+ "     spid \n"
				+ "    ,kpid \n"
				
				+ "    ,suid \n"
				+ "    ,suser_name(suid) \n"
				+ "    ,hostname \n"
				+ "    ,program_name \n"
				+ "    ,hostprocess \n"
				+ "    ,dbid \n"
				+ "    ,db_name(dbid) \n"
				+ "    ,uid \n"
				+ "    ,gid \n"
				+ "    ,fid \n"
				+ "    ,origsuid \n"
				+ "    ,suser_name(origsuid) \n"
				+ "    ,clientname \n"
				+ "    ,clienthostname \n"
				+ "    ,clientapplname \n"
				+ "    ,loggedindatetime \n"
				+ "    ,ipaddr \n"
				+ "from master.dbo.sysprocesses \n"
				+ "where suid != 0 \n"
//				+ "  and spid = " + spid + " \n"
//				+ "  and kpid = " + kpid + " \n"
				+ "";

		long startTime = System.currentTimeMillis();
		int  refreshRecords = 0;

		int beforeRefreshSpidCount = _cache.size();
		
		// Should we clear the cache before getting new entries
		_cache.clear();

		// Get new records
		try (Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql))
		{
			while (rs.next())
			{
				refreshRecords++;
				SybSysProcessesEntry entry = new SybSysProcessesEntry();
				
				entry._spid              = rs.getInt      (1);
				entry._kpid              = rs.getInt      (2);
                                                          
				entry._suid              = rs.getInt      (3);
				entry._login_name        = rs.getString   (4);
				entry._hostname          = rs.getString   (5);
				entry._program_name      = rs.getString   (6);
				entry._hostprocess       = rs.getString   (7);
				entry._dbid              = rs.getInt      (8);
				entry._dbname            = rs.getString   (9);
				entry._uid               = rs.getInt      (10);
				entry._gid               = rs.getInt      (11);
				entry._fid               = rs.getInt      (12);
				entry._origsuid          = rs.getInt      (13);
				entry._origin_login_name = rs.getString   (14);
				entry._clientname        = rs.getString   (15);
				entry._clienthostname    = rs.getString   (16);
				entry._clientapplname    = rs.getString   (17);
				entry._loggedindatetime  = rs.getTimestamp(18);
				entry._ipaddr            = rs.getString   (19);

				entry._roleNames         = StringUtil.toCommaStr( suidToRoleNames.get(entry._suid) );
				
				// Handle Strings with null values (assign empty string)
				if (entry._login_name        == null) entry._login_name        = "";
				if (entry._hostname          == null) entry._hostname          = "";
				if (entry._program_name      == null) entry._program_name      = "";
				if (entry._hostprocess       == null) entry._hostprocess       = "";
				if (entry._dbname            == null) entry._dbname            = "";
				if (entry._origin_login_name == null) entry._origin_login_name = "";
				if (entry._clientname        == null) entry._clientname        = "";
				if (entry._clienthostname    == null) entry._clienthostname    = "";
				if (entry._clientapplname    == null) entry._clientapplname    = "";
				if (entry._ipaddr            == null) entry._ipaddr            = "";

				// Add it to the cache
				Map<Integer, SybSysProcessesEntry> newKpidEntry = new HashMap<>();
				newKpidEntry.put(entry._kpid, entry);
				
				_cache.put(entry._spid, newKpidEntry);
			}
		}
		
		// If you want to debug what was added to the _cache
		boolean debugPrint = false;
		if (debugPrint)
		{
			for (Entry<Integer, Map<Integer, SybSysProcessesEntry>> eSet1 : _cache.entrySet())
			{
				for (Entry<Integer, SybSysProcessesEntry> eSet2 : eSet1.getValue().entrySet())
				{
					System.out.println("############# " + this.getClass().getSimpleName() + " ################# SPID=" + eSet1.getKey() + ", KPID=" + eSet2.getKey() + ": " + eSet2.getValue());
				}
			}
		}
		
		long execTimeMs = TimeUtils.msDiffNow(startTime);
		String msSinceLastRefreshStr = _lastTotalRefreshTime < 0 ? "-initial-refresh-" : TimeUtils.msToTimeStrShort(TimeUtils.msDiffNow(_lastTotalRefreshTime));

		int afterRefreshSpidCount     = _cache.size();
		int afterRefreshSpidCountDiff = afterRefreshSpidCount - beforeRefreshSpidCount;
		
		_logger.info("Refreshed 'sysprocesses' cache (total refresh) with " + refreshRecords + " entries, which took " + execTimeMs + " ms. "
				+ "Last refresh was done '" + msSinceLastRefreshStr + "' (MM:SS.ms) ago. "
				+ "isCacheExpired=" + isCacheExpired() + ", totalRefreshCount=" + _totalRefreshCount + ", spidCountBeforeRefresh=" + beforeRefreshSpidCount + ", spidCountAfterRefresh=" + afterRefreshSpidCount + ", diff=" + afterRefreshSpidCountDiff);

		_totalRefreshCount++;
		_lastTotalRefreshTime  = System.currentTimeMillis();
	}

//	/**
//	 * Get a cached record! <br>
//	 * If the record is not found in the cache it will try to get it from the 'master.dbo.sysprocess' table
//	 * 
//	 * @param conn
//	 * @param spid
//	 * @param kpid
//	 * @return
//	 */
//	public SybSysProcessesEntry getRecord(Connection conn, int spid, int kpid)
//	throws SQLException
//	{
//		// Get cached: SPID, KPID 
//		// If in cache return the entry
//		SybSysProcessesEntry cachedRecord = getCachedRecord(spid, kpid);
//		if (cachedRecord != null)
//			return cachedRecord;
//
//		// Do not refresh to often
//		long msSinceLastRefresh = TimeUtils.msDiffNow(_lastTotalRefreshTime);
//		if (msSinceLastRefresh < _refreshThresholdInMs)
//		{
//			_logger.info("Skipping refresh of " + this.getClass().getSimpleName() + ", (spid=" + spid + ", kpid=" + kpid + ") due to last refresh was done " + msSinceLastRefresh + " ms ago, and the threshold is " + _refreshThresholdInMs +". This to not overload when there are SPID's not logged in. This can be changed with Property: " + PROPKEY_refreshThresholdInMs + " = ###");
//			return null;
//		}
//		// Or we can remember "last SPID/KPID" that were a "miss" and clear that when we have a "hit"
//
//		
//		// NOT IN CACHE -- Get SPID,KPID entry from DBMS or just refresh ALL entries
//		//                 In later versions we might get a single spid/kpid... But lets do that later
//		synchronized (this)
//		{
//			// Should we check again if we can find it in the cache (while we were sleeping on the synchronized section, it might already be added by the previous
//			cachedRecord = getCachedRecord(spid, kpid);
//			if (cachedRecord != null)
//				return cachedRecord;
//			
//			// should we also get SRV LOGIN ROLES and add the "role names" to each sysprocesses entry ???
//			Map<Integer, List<String>> suidToRoleNames = getSrvRolesMap(conn);
//			
//			String sql = ""
//					+ "select \n"
//					+ "     spid \n"
//					+ "    ,kpid \n"
//					
//					+ "    ,suid \n"
//					+ "    ,suser_name(suid) \n"
//					+ "    ,hostname \n"
//					+ "    ,program_name \n"
//					+ "    ,hostprocess \n"
//					+ "    ,dbid \n"
//					+ "    ,db_name(dbid) \n"
//					+ "    ,uid \n"
//					+ "    ,gid \n"
//					+ "    ,fid \n"
//					+ "    ,origsuid \n"
//					+ "    ,suser_name(origsuid) \n"
//					+ "    ,clientname \n"
//					+ "    ,clienthostname \n"
//					+ "    ,clientapplname \n"
//					+ "    ,loggedindatetime \n"
//					+ "    ,ipaddr \n"
//					+ "from master.dbo.sysprocesses \n"
//					+ "where suid != 0 \n"
////					+ "  and spid = " + spid + " \n"
////					+ "  and kpid = " + kpid + " \n"
//					+ "";
//
//			SybSysProcessesEntry returnEntry = null;
//
//			long startTime = System.currentTimeMillis();
//			int  refreshRecords = 0;
//			
//			int beforeRefreshSpidCount = _cache.size();
//					
//			try (Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql))
//			{
//				while (rs.next())
//				{
//					refreshRecords++;
//					SybSysProcessesEntry entry = new SybSysProcessesEntry();
//					
//					entry._spid              = rs.getInt      (1);
//					entry._kpid              = rs.getInt      (2);
//                                                              
//					entry._suid              = rs.getInt      (3);
//					entry._login_name        = rs.getString   (4);
//					entry._hostname          = rs.getString   (5);
//					entry._program_name      = rs.getString   (6);
//					entry._hostprocess       = rs.getString   (7);
//					entry._dbid              = rs.getInt      (8);
//					entry._dbname            = rs.getString   (9);
//					entry._uid               = rs.getInt      (10);
//					entry._gid               = rs.getInt      (11);
//					entry._fid               = rs.getInt      (12);
//					entry._origsuid          = rs.getInt      (13);
//					entry._origin_login_name = rs.getString   (14);
//					entry._clientname        = rs.getString   (15);
//					entry._clienthostname    = rs.getString   (16);
//					entry._clientapplname    = rs.getString   (17);
//					entry._loggedindatetime  = rs.getTimestamp(18);
//					entry._ipaddr            = rs.getString   (19);
//
//					entry._roleNames         = StringUtil.toCommaStr( suidToRoleNames.get(entry._suid) );
//					
//					if (entry._spid == spid && entry._kpid == kpid)
//						returnEntry = entry;
//
//					// Add it to the cache
//					Map<Integer, SybSysProcessesEntry> newKpidEntry = new HashMap<>();
//					newKpidEntry.put(entry._kpid, entry);
//					
//					_cache.put(entry._spid, newKpidEntry);
//
////					_singleRefreshCount++;
//				}
//			}
//			
//			// If you want to debug what was added to the _cache
//			boolean debugPrint = false;
//			if (debugPrint)
//			{
//				for (Entry<Integer, Map<Integer, SybSysProcessesEntry>> eSet1 : _cache.entrySet())
//				{
//					for (Entry<Integer, SybSysProcessesEntry> eSet2 : eSet1.getValue().entrySet())
//					{
//						System.out.println("############# " + this.getClass().getSimpleName() + " ################# SPID=" + eSet1.getKey() + ", KPID=" + eSet2.getKey() + ": " + eSet2.getValue());
//					}
//				}
//			}
//			
//			long execTimeMs = TimeUtils.msDiffNow(startTime);
//			String msSinceLastRefreshStr = _lastTotalRefreshTime < 0 ? "-initial-refresh-" : TimeUtils.msToTimeStrShort(TimeUtils.msDiffNow(_lastTotalRefreshTime));
//
//			int afterRefreshSpidCount     = _cache.size();
//			int afterRefreshSpidCountDiff = afterRefreshSpidCount - beforeRefreshSpidCount;
//			
//			_logger.info("Refreshed 'sysprocesses' cache (total refresh) with " + refreshRecords + " entries, which took " + execTimeMs + " ms. "
//					+ "The requested spid=" + spid + ", kpid=" + kpid + ". "
//					+ "Last refresh was done '" + msSinceLastRefreshStr + "' (MM:SS.ms) ago. "
//					+ "isCacheExpired=" + isCacheExpired() + ", spidCountBeforeRefresh=" + beforeRefreshSpidCount + ", spidCountAfterRefresh=" + afterRefreshSpidCount + ", diff=" + afterRefreshSpidCountDiff);
//
//			_totalRefreshCount++;
//			_lastTotalRefreshTime  = System.currentTimeMillis();
//			
//			return returnEntry;
//		}
//	}

	/**
	 * Get any cached record
	 * 
	 * @param spid
	 * @param kpid
	 * @return
	 */
	private SybSysProcessesEntry getCachedRecord(int spid, int kpid)
	{
		Map<Integer, SybSysProcessesEntry> kpidEntry = _cache.get(spid);
		if (kpidEntry != null)
		{
			// if KPID is not specified
			if (kpid == -1)
			{
				if (kpidEntry.size() == 1)
				{
					// Just grab first one
					for (SybSysProcessesEntry entry : kpidEntry.values())
						return entry;
				}
				else
				{
					// Grab the one with the latest login
					long loginTs = 0;
					SybSysProcessesEntry returnEntry = null;
					for (SybSysProcessesEntry entry : kpidEntry.values())
					{
						if (entry._loggedindatetime.getTime() > loginTs)
						{
							loginTs = entry._loggedindatetime.getTime();
							returnEntry = entry;
						}
					}
					return returnEntry;
				}
			}
			else
			{
				SybSysProcessesEntry entry = kpidEntry.get(kpid);
				if (entry != null)
				{
					return entry;
				}
			}
		}
		return null;
	}
	
	/**
	 * Get a cached record! <br>
	 * If the record is not found in the cache it will try to get it from the 'master.dbo.sysprocess' table
	 * 
	 * @param conn
	 * @param spid
	 * @param kpid
	 * @return
	 */
	public SybSysProcessesEntry getRecord(Connection conn, int spid, int kpid)
	throws SQLException
	{
		// Get cached: SPID, KPID 
		// If in cache return the entry
		SybSysProcessesEntry cachedRecord = getCachedRecord(spid, kpid);
		if (cachedRecord != null)
			return cachedRecord;

		// Do not refresh to often
		long msSinceLastRefresh = TimeUtils.msDiffNow(_lastTotalRefreshTime);
		if (msSinceLastRefresh < _refreshThresholdInMs)
		{
			_logger.info("Skipping refresh of " + this.getClass().getSimpleName() + ", (spid=" + spid + ", kpid=" + kpid + ") due to last refresh was done " + msSinceLastRefresh + " ms ago, and the threshold is " + _refreshThresholdInMs +". This to not overload when there are SPID's not logged in. This can be changed with Property: " + PROPKEY_refreshThresholdInMs + " = ###");
			return null;
		}
		// Or we can remember "last SPID/KPID" that were a "miss" and clear that when we have a "hit"

		
		// NOT IN CACHE -- Get SPID,KPID entry from DBMS or just refresh ALL entries
		//                 In later versions we might get a single spid/kpid... But lets do that later
		synchronized (this)
		{
			// Should we check again if we can find it in the cache (while we were sleeping on the synchronized section, it might already be added by the previous
			cachedRecord = getCachedRecord(spid, kpid);
			if (cachedRecord != null)
				return cachedRecord;

			refreshCache(conn);
			
			return getCachedRecord(spid, kpid);
		}
	}
	
	private Map<Integer, List<String>> getSrvRolesMap(Connection conn)
	{
		String sql = "select suid, role_name(srid) from master.dbo.sysloginroles";

		//  suid          roleName
		Map<Integer, List<String>> map = new HashMap<>();
		
		try (Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql))
		{
			while(rs.next())
			{
				int suid        = rs.getInt   (1);
				String roleName = rs.getString(2);
				
				List<String> roleNames = map.get(suid);
				if (roleNames == null)
				{
					roleNames = new ArrayList<>();

					// Create the mapping SPID->RoleNames
					map.put(suid, roleNames);
				}

				// Add role name to the list
				roleNames.add(roleName);
			}
		}
		catch(SQLException ex)
		{
			_logger.warn("Problems getting ASE Role information using SQL='" + sql + "'. Error=" + ex.getErrorCode() + ", Msg='" + ex.getMessage() + "'.");
		}
		
		return map;
	}
	
	
//	@JsonPropertyOrder(value = {"spid", "kpid", ...}, alphabetic = true)
	public static class SybSysProcessesEntry
	{
		int       _spid;
		int       _kpid;

		int       _suid;
		String    _login_name;
		String    _hostname;
		String    _program_name;
		String    _hostprocess;
		int       _dbid;
		String    _dbname;
		int       _uid;
		int       _gid;
		int       _fid;
		int       _origsuid;
		String    _origin_login_name;
		String    _clientname;
		String    _clienthostname;
		String    _clientapplname;
		Timestamp _loggedindatetime;
		String    _ipaddr;
		String    _roleNames;

		public int       getSpid             () { return _spid; }
		public int       getKpid             () { return _kpid; }
		public int       getSuid             () { return _suid; }
		public String    getLoginName        () { return _login_name; }
		public String    getHostName         () { return _hostname; }
		public String    getProgramName      () { return _program_name; }
		public String    getHostProcess      () { return _hostprocess; }
		public int       getDbid             () { return _dbid; }
		public String    getDbname           () { return _dbname; }
		public int       getUid              () { return _uid; }
		public int       getGid              () { return _gid; }
		public int       getFid              () { return _fid; }
		public int       getOrigSuid         () { return _origsuid; }
		public String    getOriginLoginName  () { return _origin_login_name; }
		public String    getClientName       () { return _clientname; }
		public String    getClientHostName   () { return _clienthostname; }
		public String    getClientApplName   () { return _clientapplname; }
		public Timestamp getLoggedinDatetime () { return _loggedindatetime; }
		public String    getIpaddr           () { return _ipaddr; }
		public String    getRoleNames        () { return _roleNames; }
		
		public void createJsonForRecord(JsonGenerator gen) 
		throws IOException
		{
			gen.writeFieldName("sysProcessesInfo");
			gen.writeStartObject();
				gen.writeNumberField("spid"             , _spid);
				gen.writeNumberField("kpid"             , _kpid);
				gen.writeNumberField("suid"             , _suid);
				gen.writeStringField("loginName"        , _login_name);
				gen.writeStringField("hostName"         , _hostname);
				gen.writeStringField("programName"      , _program_name);
				gen.writeStringField("hostProcess"      , _hostprocess);
				gen.writeNumberField("dbid"             , _dbid);
				gen.writeStringField("dbname"           , _dbname);
				gen.writeNumberField("uid"              , _uid);
				gen.writeNumberField("gid"              , _gid);
				gen.writeNumberField("fid"              , _fid);
				gen.writeNumberField("origSuid"         , _origsuid);
				gen.writeStringField("originLoginName"  , _origin_login_name);
				gen.writeStringField("clientName"       , _clientname);
				gen.writeStringField("clientHostName"   , _clienthostname);
				gen.writeStringField("clientApplName"   , _clientapplname);
				gen.writeStringField("loggedinDatetime" , TimeUtils.toStringIso8601(_loggedindatetime));
				gen.writeStringField("ipaddr"           , _ipaddr);
				gen.writeStringField("roleNames"        , _roleNames);
			gen.writeEndObject();
		}
		
		@Override
		public int hashCode()
		{
			return Objects.hash(_kpid, _spid);
		}

		@Override
		public boolean equals(Object obj)
		{
			if ( this == obj )
				return true;
			if ( obj == null )
				return false;
			if ( getClass() != obj.getClass() )
				return false;
			SybSysProcessesEntry other = (SybSysProcessesEntry) obj;
			return _kpid == other._kpid && _spid == other._spid;
		}

		@Override
		public String toString()
		{
			return "SybSysProcessesEntry [" 
					+ "spid="                 + _spid 
					+ ", kpid="               + _kpid 
					+ ", suid="               + _suid 
					+ ", login_name='"        + _login_name        + "'"
					+ ", hostname='"          + _hostname          + "'"
					+ ", program_name='"      + _program_name      + "'"
					+ ", hostprocess='"       + _hostprocess       + "'"
					+ ", dbid="               + _dbid 
					+ ", dbname='"            + _dbname            + "'"
					+ ", uid="                + _uid 
					+ ", gid="                + _gid 
					+ ", fid="                + _fid 
					+ ", origsuid="           + _origsuid 
					+ ", origin_login_name='" + _origin_login_name + "'" 
					+ ", clientname='"        + _clientname        + "'"
					+ ", clienthostname='"    + _clienthostname    + "'"
			        + ", clientapplname='"    + _clientapplname    + "'"
			        + ", loggedindatetime='"  + _loggedindatetime  + "'"
			        + ", ipaddr="             + _ipaddr 
			        + "]";
		}
		
		
	}
}
