/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.collectors.records;

import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonGenerator;

import sek.ase.auditor.collectors.SybStmntCollector;
import sek.ase.auditor.collectors.SybSysProcessesCache.SybSysProcessesEntry;
import sek.ase.auditor.utils.AseErrorMessageDictionary;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.TimeUtils;

public class AuditRecordSybStmnt
extends AuditRecord
{
	private static Logger _logger = LogManager.getLogger();
	
	private SybSysProcessesEntry  _sybSysProcessEntry;
	private SqlTextEntry          _sqlTextEntry;

	public AuditRecordSybStmnt()
	{
	}

	public void setSybSysProcessesEntry(SybSysProcessesEntry sybSysProcessEntry)
	{
		_sybSysProcessEntry = sybSysProcessEntry;
	}

	public void setSqlTextEntry(SqlTextEntry sqlTextEntry)
	{
		_sqlTextEntry = sqlTextEntry;
	}

	public void setSqlTextEntry(int spid, int kpid, int batchId, String srvLogin, String sqlText, List<MonSqlStatementEntry> mdaStatementRecords)
	{
		_sqlTextEntry = new SqlTextEntry(spid, kpid, batchId, srvLogin, sqlText, mdaStatementRecords);
	}

	
	@Override
	public String toString()
	{
		return super.toString() + "\n"
				+ "      -- " + _sqlTextEntry       + ", \n"
				+ "      -- " + _sybSysProcessEntry + ".";
	}

	@Override
	public void createJsonForRecord(JsonGenerator gen) 
	throws IOException
	{
		if (_sqlTextEntry != null)
		{
			gen.writeFieldName("sqlText");
			gen.writeStartObject();
				gen.writeNumberField("spid"      , _sqlTextEntry._spid);
				gen.writeNumberField("kpid"      , _sqlTextEntry._kpid);
				gen.writeNumberField("batchId"   , _sqlTextEntry._batchId);
				gen.writeStringField("srvLogin"  , _sqlTextEntry._srvLogin);
				gen.writeStringField("sqlText"   , _sqlTextEntry._sqlText);
				
				if (_sqlTextEntry._mdaStatementRecords != null && !_sqlTextEntry._mdaStatementRecords.isEmpty())
				{
					gen.writeFieldName("statementInfo");
					gen.writeStartArray();

					int stmntSeqNum = 0;
					for (MonSqlStatementEntry stmntEntry : _sqlTextEntry._mdaStatementRecords)
					{
						stmntSeqNum++; // Comment this out if you do NOT want 'stmntSeqNum' in the JSON (stmntSeqNum must be above 0 to be included)
						stmntEntry.createJsonForRecord(gen, stmntSeqNum);
					}
					gen.writeEndArray();
				}
			gen.writeEndObject();
		}

		if (_sybSysProcessEntry != null)
		{
			_sybSysProcessEntry.createJsonForRecord(gen);
		}
	}
	




	public static class SqlTextEntry
	{
		int    _spid;
		int    _kpid;
		int    _batchId;
		String _srvLogin;
		String _sqlText;
		List<MonSqlStatementEntry> _mdaStatementRecords;

		public SqlTextEntry()
		{
		}

		public SqlTextEntry(int spid, int kpid, int batchId, String srvLogin, String sqlText, List<MonSqlStatementEntry> mdaStatementRecords)
		{
			_spid     = spid    ;
			_kpid     = kpid    ;
			_batchId  = batchId ;
			_srvLogin = srvLogin;
			_sqlText  = sqlText ;
			_mdaStatementRecords = mdaStatementRecords;
		}
		
		@Override
		public String toString()
		{
			return "SqlTextEntry [spid=" + _spid + ", kpid=" + _kpid + ", batchId=" + _batchId + ", srvLogin='" + _srvLogin + "', sqlText=|" + _sqlText + "|, mdaStatementRecords.size=" + (_mdaStatementRecords == null ? "null" : _mdaStatementRecords.size()) + "]";
		}
	}




	public static class MonSqlStatementEntry
	{
		public int       SPID                  ;
		public int       KPID                  ;
		public int       BatchID               ;
		
		public Timestamp StartTime             ;
		public Timestamp EndTime               ;
		public int       Elapsed_ms            ;
		public Timestamp sampleTime            ;

		public int       DBID                  ;
		public String    DBName                ;
		public int       RowsAffected          ;
		public int       ErrorStatus           ;
		public int       HashKey               ;
		public int       SsqlId                ;
		public int       ObjOwnerID            ;
		public int       ProcedureID           ;
		public int       PlanID                ;
		public int       ContextID             ;
		public String    ProcName              ;
		public int       ProcNestLevel         ;
		public int       StatementNumber       ;
		public int       LineNumber            ;

		public long      QueryOptimizationTime ;
		public int       PlansAltered          ;
		public int       CpuTime               ;
		public int       WaitTime              ;
		public int       MemUsageKB            ;
		public long      PhysicalReads         ;
		public long      LogicalReads          ;
		public long      PagesModified         ;
		public int       PacketsSent           ;
		public int       PacketsReceived       ;
		public int       NetworkPacketSize     ;

		public MonSqlStatementEntry(ResultSet rs)
		throws SQLException
		{
			int colNum = 1;

			SPID                  = rs.getInt      (colNum++);
			KPID                  = rs.getInt      (colNum++);
			BatchID               = rs.getInt      (colNum++);

			StartTime             = rs.getTimestamp(colNum++);
			EndTime               = rs.getTimestamp(colNum++);
			Elapsed_ms            = rs.getInt      (colNum++);
			sampleTime            = rs.getTimestamp(colNum++);

			DBID                  = rs.getInt      (colNum++);
			DBName                = rs.getString   (colNum++);
			RowsAffected          = rs.getInt      (colNum++);
			ErrorStatus           = rs.getInt      (colNum++);
			HashKey               = rs.getInt      (colNum++);
			SsqlId                = rs.getInt      (colNum++);
			ObjOwnerID            = rs.getInt      (colNum++);
			ProcedureID           = rs.getInt      (colNum++);
			PlanID                = rs.getInt      (colNum++);
			ContextID             = rs.getInt      (colNum++);
			ProcName              = rs.getString   (colNum++);
			ProcNestLevel         = rs.getInt      (colNum++);
			StatementNumber       = rs.getInt      (colNum++);
			LineNumber            = rs.getInt      (colNum++);

			QueryOptimizationTime = rs.getLong     (colNum++);
			PlansAltered          = rs.getInt      (colNum++);
			CpuTime               = rs.getInt      (colNum++);
			WaitTime              = rs.getInt      (colNum++);
			MemUsageKB            = rs.getInt      (colNum++);
			try { PhysicalReads   = rs.getLong     (colNum++); } catch (SQLException ex) { _logger.warn("Problems reading column 'PhysicalReads', strVal='" + rs.getString(colNum-1) + "', setting this to -1 and continuing. Caught: " + ex); }
			LogicalReads          = rs.getLong     (colNum++);
			PagesModified         = rs.getLong     (colNum++);
			PacketsSent           = rs.getInt      (colNum++);
			PacketsReceived       = rs.getInt      (colNum++);
			NetworkPacketSize     = rs.getInt      (colNum++);
			
			// Handle Strings with null values (assign empty string)
			if (DBName   == null) DBName   = "";
			if (ProcName == null) ProcName = "";
		}

		private static String _sql;
		public static String getSql()
		{
			if (_sql != null)
				return _sql;

			boolean collectProcName = Configuration.getCombinedConfiguration().getBooleanProperty(SybStmntCollector.PROPKEY_sampleStatementProcName, SybStmntCollector.DEFAULT_sampleStatementProcName);
			
			String ObjOwnerID = null;
			String ProcName   = null;
			String QueryOptimizationTime = "    ,QueryOptimizationTime = -1 \n";  // Commented out for SEK
//			String QueryOptimizationTime = "    ,QueryOptimizationTime \n";       // added in 16.0 SP3 <<<<<< NOTE: This might be to high for SEK
			

			// Do we want to collect 'ObjOwnerID' and 'ProcName'
			//   * 'object_owner_id(ProcedureID, DBID)' 
			//   * 'object_name(ProcedureID,DBID)' 
			// The above causes error 10351, Desc="Server user id 25 is not a valid user in database 'dummy1'."
			// So the user that does the below select needs 'sa_role' or being a user in each of the databases.
			if (collectProcName)
			{
				ObjOwnerID = "    ,ObjOwnerID = CASE WHEN SsqlId > 0 THEN 0 ELSE object_owner_id(ProcedureID, DBID) END \n";   // added in 15.0.3
				
				ProcName   = "    ,ProcName   = CASE WHEN SsqlId > 0 THEN isnull(object_name(SsqlId,2), '*##'+right('0000000000'+convert(varchar(20),SsqlId),10)+'_'+right('0000000000'+convert(varchar(20),HashKey),10)+'##*') \n"
				           + "                       ELSE coalesce(object_name(ProcedureID,DBID), object_name(ProcedureID,2), object_name(ProcedureID,db_id('sybsystemprocs'))) \n"
				           + "                  END \n";
			}
			else
			{
				ObjOwnerID = "    ,ObjOwnerID = -1 \n";   // added in 15.0.3
				ProcName   = "    ,ProcName   = '" + SybStmntCollector.PROPKEY_sampleStatementProcName + " = false'";

				_logger.warn("When initializing SQL query for '" + SybStmntCollector.PROPKEY_sampleStatementDetatils + "' we will SKIP the columns 'ProcName' and 'ObjOwnerID'. If you want to enable them set '" + SybStmntCollector.PROPKEY_sampleStatementProcName + " = true'. NOTE: The Login that collects data needs to have 'sa_role' or access to all databases (sp_adduser -in-all-databases-).");
			}
			
			_sql = ""
				+ "select \n"
				+ "     SPID \n"
				+ "    ,KPID \n"
				+ "    ,BatchID \n"

				+ "    ,StartTime \n"
				+ "    ,EndTime \n"
				+ "    ,Elapsed_ms = CASE WHEN datediff(day, StartTime, EndTime) >= 24 THEN -1 ELSE  datediff(ms, StartTime, EndTime) END \n"
				+ "    ,sampleTime = getdate() \n"

				+ "    ,DBID \n"
				+ "    ,DBName \n"    // added in 15.0.2 ESD#3
				+ "    ,RowsAffected \n"   // added in 15.0.0 ESD#2
				+ "    ,ErrorStatus \n"    // added in 15.0.0 ESD#2
				+ "    ,HashKey \n"   // added in 15.0.2
				+ "    ,SsqlId \n"    // added in 15.0.2
				+      ObjOwnerID
				+ "    ,ProcedureID \n"
				+ "    ,PlanID \n"
				+ "    ,ContextID \n"
				+      ProcName
				+ "    ,ProcNestLevel \n" // added in 15.0.3
				+ "    ,StatementNumber \n" // added in 15.0.3
				+ "    ,LineNumber \n"

				+      QueryOptimizationTime // added in 16.0 SP3 <<<<<< NOTE: This might be to high for SEK
				+ "    ,PlansAltered \n"
				+ "    ,CpuTime \n"
				+ "    ,WaitTime \n"
				+ "    ,MemUsageKB \n"
				+ "    ,PhysicalReads \n"
				+ "    ,LogicalReads \n"
				+ "    ,PagesModified \n"
				+ "    ,PacketsSent \n"
				+ "    ,PacketsReceived \n"
				+ "    ,NetworkPacketSize \n"
				+ "from master.dbo.monSysStatement \n"
				+ "";
		
			return _sql;
		}

		public void createJsonForRecord(JsonGenerator gen, int stmntSeqNum)
		throws IOException
		{
			String errorMsg = null;
			if (ErrorStatus != 0)
				errorMsg = AseErrorMessageDictionary.getInstance().getDescription(ErrorStatus);
			
			
			gen.writeStartObject(); // ---- START OBJECT

			if (stmntSeqNum > 0)
				gen.writeNumberField("stmntSeqNum"       ,                           stmntSeqNum           );

			gen.writeNumberField("spid"                  ,                           SPID                  );
			gen.writeNumberField("kpid"                  ,                           KPID                  );
			gen.writeNumberField("batchId"               ,                           BatchID               );
			
			gen.writeStringField("startTime"             , TimeUtils.toStringIso8601(StartTime)            );
			gen.writeStringField("endTime"               , TimeUtils.toStringIso8601(EndTime)              );
			gen.writeNumberField("elapsedMs"             ,                           Elapsed_ms            );
			gen.writeStringField("sampleTime"            , TimeUtils.toStringIso8601(sampleTime)           );

			gen.writeNumberField("dbid"                  ,                           DBID                  );
			gen.writeStringField("dbname"                ,                           DBName                );
			gen.writeNumberField("rowsAffected"          ,                           RowsAffected          );
			gen.writeNumberField("errorStatus"           ,                           ErrorStatus           );
			if (errorMsg != null)
				gen.writeStringField("errorMsg"          ,                           errorMsg              ); // If ErrorStatus != ... fill in the Error Text Description
			gen.writeNumberField("hashKey"               ,                           HashKey               );
			gen.writeNumberField("ssqlId"                ,                           SsqlId                );
			gen.writeNumberField("objOwnerId"            ,                           ObjOwnerID            );
			gen.writeNumberField("procedureId"           ,                           ProcedureID           );
			gen.writeNumberField("planId"                ,                           PlanID                );
			gen.writeNumberField("contextId"             ,                           ContextID             );
			gen.writeStringField("procName"              ,                           ProcName              );
			gen.writeNumberField("procNestLevel"         ,                           ProcNestLevel         );
			gen.writeNumberField("statementNumber"       ,                           StatementNumber       );
			gen.writeNumberField("lineNumber"            ,                           LineNumber            );

			gen.writeNumberField("queryOptimizationTime" ,                           QueryOptimizationTime );
			gen.writeNumberField("plansAltered"          ,                           PlansAltered          );
			gen.writeNumberField("cpuTime"               ,                           CpuTime               );
			gen.writeNumberField("waitTime"              ,                           WaitTime              );
			gen.writeNumberField("memUsageKb"            ,                           MemUsageKB            );
			gen.writeNumberField("physicalReads"         ,                           PhysicalReads         );
			gen.writeNumberField("logicalReads"          ,                           LogicalReads          );
			gen.writeNumberField("pagesModified"         ,                           PagesModified         );
			gen.writeNumberField("packetsSent"           ,                           PacketsSent           );
			gen.writeNumberField("packetsReceived"       ,                           PacketsReceived       );
			gen.writeNumberField("networkPacketSize"     ,                           NetworkPacketSize     );

			gen.writeEndObject(); // ---- END OBJECT
		}
	}
}


// 1> select * from master..monSysStatement
// RS> Col# Label                 JDBC Type Name           Guessed DBMS type Source Table              
// RS> ---- --------------------- ------------------------ ----------------- --------------------------
// RS> 1    SPID                  java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 2    InstanceID            java.sql.Types.TINYINT   tinyint           master.dbo.monSysStatement
// RS> 3    KPID                  java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 4    DBID                  java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 5    ProcedureID           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 6    PlanID                java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 7    BatchID               java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 8    ContextID             java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 9    LineNumber            java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 10   CpuTime               java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 11   WaitTime              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 12   MemUsageKB            java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 13   PhysicalReads         java.sql.Types.BIGINT    bigint            master.dbo.monSysStatement
// RS> 14   LogicalReads          java.sql.Types.BIGINT    bigint            master.dbo.monSysStatement
// RS> 15   PagesModified         java.sql.Types.BIGINT    bigint            master.dbo.monSysStatement
// RS> 16   PacketsSent           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 17   PacketsReceived       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 18   NetworkPacketSize     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 19   PlansAltered          java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 20   RowsAffected          java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 21   ErrorStatus           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 22   HashKey               java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 23   SsqlId                java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 24   ProcNestLevel         java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 25   StatementNumber       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 26   SnapCodegenTime       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 27   SnapJITTime           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 28   SnapExecutionTime     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 29   SnapExecutionCount    java.sql.Types.INTEGER   int               master.dbo.monSysStatement
// RS> 30   QueryOptimizationTime java.sql.Types.BIGINT    bigint            master.dbo.monSysStatement
// RS> 31   DBName                java.sql.Types.VARCHAR   varchar(30)       master.dbo.monSysStatement
// RS> 32   StartTime             java.sql.Types.TIMESTAMP datetime          master.dbo.monSysStatement
// RS> 33   EndTime               java.sql.Types.TIMESTAMP datetime          master.dbo.monSysStatement
// +----+----------+----------+----+-------------+---------+-------+---------+----------+-------+--------+----------+-------------+------------+-------------+-----------+---------------+-----------------+------------+------------+-----------+-------------+-------------+-------------+---------------+---------------+-----------+-----------------+------------------+---------------------+------+-----------------------+-----------------------+
// |SPID|InstanceID|KPID      |DBID|ProcedureID  |PlanID   |BatchID|ContextID|LineNumber|CpuTime|WaitTime|MemUsageKB|PhysicalReads|LogicalReads|PagesModified|PacketsSent|PacketsReceived|NetworkPacketSize|PlansAltered|RowsAffected|ErrorStatus|HashKey      |SsqlId       |ProcNestLevel|StatementNumber|SnapCodegenTime|SnapJITTime|SnapExecutionTime|SnapExecutionCount|QueryOptimizationTime|DBName|StartTime              |EndTime                |
// +----+----------+----------+----+-------------+---------+-------+---------+----------+-------+--------+----------+-------------+------------+-------------+-----------+---------------+-----------------+------------+------------+-----------+-------------+-------------+-------------+---------------+---------------+-----------+-----------------+------------------+---------------------+------+-----------------------+-----------------------+
// |  72|         0|26542144|   2|  656888237|1110501|    236|        1|         1|      0|       0|        18|            0|           0|            0|          6|              0|            2048|           0|          67|          0|  352741164|  656888237|            1|              1|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.093|2023-04-28 22:57:02.093|
// |  70|         0|26148921|   1|2014067330|       81|364089|        1|         0|      0|       0|        42|            0|           0|            0|          0|              0|            2048|           0|           0|          0|1265659492|2014067330|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  70|         0|26148921|   1|2014067330|       81|364089|        1|         1|      0|       0|        20|            0|           5|            0|          0|              0|            2048|           0|           1|          0|1265659492|2014067330|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  72|         0|26542144|   2|            0|        0|    237|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  70|         0|26148921|   1|2030067387|       83|364090|        1|         1|      0|       0|        18|            0|           0|            0|          0|              0|            2048|           0|           0|          0|  276247661|2030067387|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  72|         0|26542144|   2|  686622458|  324380|    238|        1|         5|      0|       0|        16|            0|           0|            0|          0|              0|            2048|           0|           0|          0|            0|            0|            1|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  72|         0|26542144|   2|  686622458|  324380|    238|        1|         7|      0|       0|         0|            0|           0|            0|          0|              0|            2048|           0|           0|          0|            0|            0|            1|              1|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  72|         0|26542144|   2|  686622458|  324380|    238|        1|        10|      0|       0|         0|            0|           0|            0|          0|              0|            2048|           0|           0|          0|            0|            0|            1|              2|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  72|         0|26542144|   2|  686622458|  324380|    238|        1|        15|      0|       0|         0|            0|           0|            0|          0|              0|            2048|           0|           0|          0|            0|            0|            1|              3|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  72|         0|26542144|   2|  686622458|  324380|    238|        1|        17|      0|       0|         0|            0|           1|            0|          0|              0|            2048|           0|          17|          0|            0|            0|            1|              4|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  70|         0|26148921|   1|2046067444|       85|364091|        1|         0|      0|       0|        26|            0|           0|            0|          0|              0|            2048|           0|           0|          0|1193945116|2046067444|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  70|         0|26148921|   1|2046067444|       85|364091|        1|         1|      0|       0|        14|            0|           0|            0|          0|              0|            2048|           0|           7|          0|1193945116|2046067444|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.123|2023-04-28 22:57:02.123|
// |  70|         0|26148921|   1|2078067558|       89|364092|        1|         0|      0|       0|        78|            0|           0|            0|          0|              0|            2048|           0|           0|          0|  152176451|2078067558|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.126|2023-04-28 22:57:02.126|
// |  70|         0|26148921|   1|2078067558|       89|364092|        1|         2|      1|       0|        24|            0|          21|            0|          1|              0|            2048|           0|           9|          0|  152176451|2078067558|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.126|2023-04-28 22:57:02.126|
// |  70|         0|26148921|   1|2094067615|       91|364093|        1|         0|      0|       0|        34|            0|           0|            0|          0|              0|            2048|           0|           0|          0|1194815523|2094067615|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.13 |2023-04-28 22:57:02.13 |
// |  70|         0|26148921|   1|2094067615|       91|364093|        1|         2|      0|       0|        16|            0|          21|            0|          0|              0|            2048|           0|           4|          0|1194815523|2094067615|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:02.13 |2023-04-28 22:57:02.13 |
// |  72|         0|26542144|   2|            0|        0|    239|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.14 |2023-04-28 22:57:02.14 |
// |  72|         0|26542144|   2|            0|        0|    240|        0|         1|      0|       0|        44|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.14 |2023-04-28 22:57:02.14 |
// |  72|         0|26542144|   2|            0|        0|    241|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.14 |2023-04-28 22:57:02.14 |
// |  72|         0|26542144|   2|   76898193|  324382|    242|        1|         1|      0|       0|        24|            0|           5|            0|          0|              0|            2048|           0|           0|          0|  427372835|   76898193|            1|              1|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.14 |2023-04-28 22:57:02.14 |
// |  72|         0|26542144|   2|            0|        0|    243|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.15 |2023-04-28 22:57:02.15 |
// |  72|         0|26542144|   2|            0|        0|    244|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:02.15 |2023-04-28 22:57:02.15 |
// |  70|         0|26148921|   1|            0|        0|364094|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.13 |2023-04-28 22:57:03.13 |
// |  70|         0|26148921|   1|2014067330|       81|364095|        1|         0|      0|       0|        42|            0|           0|            0|          0|              0|            2048|           0|           0|          0|1265659492|2014067330|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.13 |2023-04-28 22:57:03.13 |
// |  70|         0|26148921|   1|2014067330|       81|364095|        1|         1|      0|       0|        20|            0|           6|            0|          0|              0|            2048|           0|           0|          0|1265659492|2014067330|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.133|2023-04-28 22:57:03.133|
// |  70|         0|26148921|   1|2078067558|       89|364096|        1|         0|      0|       0|        78|            0|           0|            0|          0|              0|            2048|           0|           0|          0|  152176451|2078067558|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.133|2023-04-28 22:57:03.133|
// |  70|         0|26148921|   1|2078067558|       89|364096|        1|         2|      1|       0|        24|            0|          21|            0|          1|              0|            2048|           0|           6|          0|  152176451|2078067558|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.133|2023-04-28 22:57:03.133|
// |  70|         0|26148921|   1|2094067615|       91|364097|        1|         0|      0|       0|        34|            0|           0|            0|          0|              0|            2048|           0|           0|          0|1194815523|2094067615|            1|              1|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.136|2023-04-28 22:57:03.136|
// |  70|         0|26148921|   1|2094067615|       91|364097|        1|         2|      0|       0|        16|            0|          21|            0|          0|              0|            2048|           0|           6|          0|1194815523|2094067615|            1|              2|              0|          0|                0|                 0|                    0|master|2023-04-28 22:57:03.136|2023-04-28 22:57:03.136|
// |  72|         0|26542144|   2|            0|        0|    245|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:03.51 |2023-04-28 22:57:03.51 |
// |  72|         0|26542144|   2|            0|        0|    246|        0|         1|      0|       0|        30|            0|           0|            0|          0|              0|            2048|           0|           1|          0|            0|            0|            0|              0|              0|          0|                0|                 0|                    0|tempdb|2023-04-28 22:57:03.626|2023-04-28 22:57:03.626|
// +----+----------+----------+----+-------------+---------+-------+---------+----------+-------+--------+----------+-------------+------------+-------------+-----------+---------------+-----------------+------------+------------+-----------+-------------+-------------+-------------+---------------+---------------+-----------+-----------------+------------------+---------------------+------+-----------------------+-----------------------+
// Rows 31
// Client Exec Time: 00:00.001 (sqlExec=00:00.000, readResults=00:00.001, other=00:00.000), at '2023-04-28 22:57:03.798', for SQL starting at Line 13152




// 1> select * from master..monSysSQLText
// RS> Col# Label           JDBC Type Name         Guessed DBMS type Source Table            
// RS> ---- --------------- ---------------------- ----------------- ------------------------
// RS> 1    SPID            java.sql.Types.INTEGER int               master.dbo.monSysSQLText
// RS> 2    InstanceID      java.sql.Types.TINYINT tinyint           master.dbo.monSysSQLText
// RS> 3    KPID            java.sql.Types.INTEGER int               master.dbo.monSysSQLText
// RS> 4    ServerUserID    java.sql.Types.INTEGER int               master.dbo.monSysSQLText
// RS> 5    BatchID         java.sql.Types.INTEGER int               master.dbo.monSysSQLText
// RS> 6    SequenceInBatch java.sql.Types.INTEGER int               master.dbo.monSysSQLText
// RS> 7    SQLText         java.sql.Types.VARCHAR varchar(255)      master.dbo.monSysSQLText
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |SPID|InstanceID|KPID      |ServerUserID|BatchID|SequenceInBatch|SQLText                                                                                                                                                                                                             |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    282|              1|select 1                                                                                                                                                                                                            |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    284|              1|select @@version                                                                                                                                                                                                    |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    285|              1|select dbname=db_name(), spid=@@spid, username = user_name(), susername =suser_name(), trancount=@@trancount, tranchained=@@tranchained, transtate=@@transtate, @@hadr_mode, hadr_mode(), @@hadr_state, hadr_state()|
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    286|              1|select @@version                                                                                                                                                                                                    |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    287|              1|select dbname=db_name(dbid), table_name=object_name(id, dbid), lock_type=type, lock_count=count(*)  from master.dbo.syslocks  where spid = @@spid         group by dbid, id, type                                   |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    288|              1|select 1                                                                                                                                                                                                            |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    289|              1|select @@tranchained                                                                                                                                                                                                |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  70|         0|26148921|           1|364236|              1|select 1                                                                                                                                                                                                            |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364237 |1              |select                                                                                                                                                                                                              |
// |    |          |          |            |       |               |     sampleTime=getdate()                                                                                                                                                                                           |
// |    |          |          |            |       |               |    ,p.SPID                                                                                                                                                                                                         |
// |    |          |          |            |       |               |    ,p.KPID                                                                                                                                                                                                         |
// |    |          |          |            |       |               |    ,p.BatchID                                                                                                                                                                                                      |
// |    |          |          |            |       |               |    ,p.ContextID                                                                                                                                                                                                    |
// |    |          |          |            |       |               |    ,p.LineNumber                                                                                                                                                                                                   |
// |    |          |          |            |       |               |    ,p.SecondsConnected                                                                                                                                                                                             |
// |    |          |          |            |       |               |    ,p.Command                                                                                                                                                                                                      |
// |    |          |          |            |       |               |    ,p.SecondsWaiting                                                                                                                                                                                               |
// |    |          |          |            |       |               |    ,p.WaitEventID                                                                                                                                                                                                  |
// |    |          |          |            |       |               |    ,p.BlockingSPID                                                                                                                                                                                                 |
// |    |          |          |            |       |               |    ,BlockingKPID    = null                                                                                                                                                                                         |
// |    |          |          |            |       |               |    ,Bl                                                                                                                                                                                                             |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364237 |2              |ockingBatchID = null                                                                                                                                                                                                |
// |    |          |          |            |       |               |    ,p.BlockingXLOID                                                                                                                                                                                                |
// |    |          |          |            |       |               |    ,p.NumChildren                                                                                                                                                                                                  |
// |    |          |          |            |       |               |    ,p.Login                                                                                                                                                                                                        |
// |    |          |          |            |       |               |    ,p.DBName                                                                                                                                                                                                       |
// |    |          |          |            |       |               |    ,p.Application                                                                                                                                                                                                  |
// |    |          |          |            |       |               |    ,p.HostName                                                                                                                                                                                                     |
// |    |          |          |            |       |               |    ,p.MasterTransactionID                                                                                                                                                                                          |
// |    |          |          |            |       |               |from master.dbo.monProcess p                                                                                                                                                                                        |
// |    |          |          |            |       |               |where 1=1                                                                                                                                                                                                           |
// |    |          |          |            |       |               |  and p.SPID != @@spid                                                                                                                                                                                              |
// |    |          |          |            |       |               |  and p.Login is not null                                                                                                                                                                                           |
// |    |          |          |            |       |               |  and                                                                                                                                                                                                               |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364237 |3              | p.Login != 'probe'                                                                                                                                                                                                 |
// |    |          |          |            |       |               |  and (p.BlockingSPID is not null or p.WaitEventID != 250)                                                                                                                                                          |
// |    |          |          |            |       |               |   or (p.WaitEventID = 250 and p.SPID in (select spid from master.dbo.syslocks))                                                                                                                                    |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |1              |/* AseTune:SqlCapture */                                                                                                                                                                                            |
// |    |          |          |            |       |               |select                                                                                                                                                                                                              |
// |    |          |          |            |       |               |    getdate() as sampleTime,                                                                                                                                                                                        |
// |    |          |          |            |       |               |    InstanceID,                                                                                                                                                                                                     |
// |    |          |          |            |       |               |    SPID,                                                                                                                                                                                                           |
// |    |          |          |            |       |               |    KPID,                                                                                                                                                                                                           |
// |    |          |          |            |       |               |    DBID,                                                                                                                                                                                                           |
// |    |          |          |            |       |               |    ProcedureID,                                                                                                                                                                                                    |
// |    |          |          |            |       |               |    PlanID,                                                                                                                                                                                                         |
// |    |          |          |            |       |               |    BatchID,                                                                                                                                                                                                        |
// |    |          |          |            |       |               |    ContextID,                                                                                                                                                                                                      |
// |    |          |          |            |       |               |    LineNumber,                                                                                                                                                                                                     |
// |    |          |          |            |       |               |    ObjOwnerID = CASE WHEN SsqlId > 0 THEN 0 ELSE object_owner_                                                                                                                                                     |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |2              |id(ProcedureID, DBID) END,                                                                                                                                                                                          |
// |    |          |          |            |       |               |    DBName,                                                                                                                                                                                                         |
// |    |          |          |            |       |               |    HashKey,                                                                                                                                                                                                        |
// |    |          |          |            |       |               |    SsqlId,                                                                                                                                                                                                         |
// |    |          |          |            |       |               |    ProcName = CASE WHEN SsqlId > 0 THEN isnull(object_name(SsqlId,2), '*##'+right('0000000000'+convert(varchar(20),SsqlId),10)+'_'+right('0000000000'+convert(varchar(20),HashKey),10)+                            |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |3              |'##*')                                                                                                                                                                                                              |
// |    |          |          |            |       |               |                    ELSE coalesce(object_name(ProcedureID,DBID), object_name(ProcedureID,2), object_name(ProcedureID,db_id('sybsystemprocs')))                                                                      |
// |    |          |          |            |       |               |                END,                                                                                                                                                                                                |
// |    |          |          |            |       |               |    CASE WHEN datediff(day, StartTime, EndTime) >= 24 THEN -1 ELSE  datediff(ms,                                                                                                                                    |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |4              | StartTime, EndTime) END as Elapsed_ms,                                                                                                                                                                             |
// |    |          |          |            |       |               |    CpuTime,                                                                                                                                                                                                        |
// |    |          |          |            |       |               |    WaitTime,                                                                                                                                                                                                       |
// |    |          |          |            |       |               |    MemUsageKB,                                                                                                                                                                                                     |
// |    |          |          |            |       |               |    PhysicalReads,                                                                                                                                                                                                  |
// |    |          |          |            |       |               |    LogicalReads,                                                                                                                                                                                                   |
// |    |          |          |            |       |               |    RowsAffected,                                                                                                                                                                                                   |
// |    |          |          |            |       |               |    ErrorStatus,                                                                                                                                                                                                    |
// |    |          |          |            |       |               |    ProcNestLevel,                                                                                                                                                                                                  |
// |    |          |          |            |       |               |    StatementNumber,                                                                                                                                                                                                |
// |    |          |          |            |       |               |    QueryOptimizationTime,                                                                                                                                                                                          |
// |    |          |          |            |       |               |    PagesModified                                                                                                                                                                                                   |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |5              |,                                                                                                                                                                                                                   |
// |    |          |          |            |       |               |    PacketsSent,                                                                                                                                                                                                    |
// |    |          |          |            |       |               |    PacketsReceived,                                                                                                                                                                                                |
// |    |          |          |            |       |               |    NetworkPacketSize,                                                                                                                                                                                              |
// |    |          |          |            |       |               |    PlansAltered,                                                                                                                                                                                                   |
// |    |          |          |            |       |               |    StartTime,                                                                                                                                                                                                      |
// |    |          |          |            |       |               |    EndTime,                                                                                                                                                                                                        |
// |    |          |          |            |       |               |    convert(varchar(10), NULL) as ServerLogin,                                                                                                                                                                      |
// |    |          |          |            |       |               |    convert(int,         -1  ) as AddStatus,                                                                                                                                                                        |
// |    |          |          |            |       |               |    convert(int,         -1  ) as JavaSqlLeng                                                                                                                                                                       |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |6              |th,                                                                                                                                                                                                                 |
// |    |          |          |            |       |               |    convert(int,         -1  ) as JavaSqlLengthShort,                                                                                                                                                               |
// |    |          |          |            |       |               |    convert(int,         -1  ) as NormJavaSqlLength,                                                                                                                                                                |
// |    |          |          |            |       |               |    convert(int,         -1  ) as JavaSqlHashCode,                                                                                                                                                                  |
// |    |          |          |            |       |               |    convert(int,         -1  ) as JavaSqlHashCodeShort,                                                                                                                                                             |
// |    |          |          |            |       |               |    convert(int,         -1  ) a                                                                                                                                                                                    |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |7              |s NormJavaSqlHashCode,                                                                                                                                                                                              |
// |    |          |          |            |       |               |    convert(varchar(10), NULL) as SQLText,                                                                                                                                                                          |
// |    |          |          |            |       |               |    convert(varchar(10), NULL) as NormSQLText,                                                                                                                                                                      |
// |    |          |          |            |       |               |    convert(varchar(10), NULL) as PlanText,                                                                                                                                                                         |
// |    |          |          |            |       |               |    convert(varchar(10), NULL) as BlockedBySqlText                                                                                                                                                                  |
// |    |          |          |            |       |               |from master.dbo.monSysStatement                                                                                                                                                                                     |
// |    |          |          |            |       |               |where 1 =                                                                                                                                                                                                           |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364238 |8              | 1                                                                                                                                                                                                                  |
// |    |          |          |            |       |               |  and SPID != @@spid                                                                                                                                                                                                |
// |    |          |          |            |       |               |  and SPID not in (select spid from master.dbo.sysprocesses where program_name like 'AseTune%')                                                                                                                     |
// |    |          |          |            |       |               |  and 1 = 1                                                                                                                                                                                                         |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364239 |1              |/* AseTune:SqlCapture */                                                                                                                                                                                            |
// |    |          |          |            |       |               |select getdate() as sampleTime,                                                                                                                                                                                     |
// |    |          |          |            |       |               |    InstanceID,                                                                                                                                                                                                     |
// |    |          |          |            |       |               |    SPID,                                                                                                                                                                                                           |
// |    |          |          |            |       |               |    KPID,                                                                                                                                                                                                           |
// |    |          |          |            |       |               |    BatchID,                                                                                                                                                                                                        |
// |    |          |          |            |       |               |    SequenceInBatch,                                                                                                                                                                                                |
// |    |          |          |            |       |               |    ServerLogin = suser_name(ServerUserID),                                                                                                                                                                         |
// |    |          |          |            |       |               |    SQLText                                                                                                                                                                                                         |
// |    |          |          |            |       |               |from master.dbo.monSysSQLText                                                                                                                                                                                       |
// |    |          |          |            |       |               |where 1 = 1                                                                                                                                                                                                         |
// |    |          |          |            |       |               |  and SPID != @@spi                                                                                                                                                                                                 |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |70  |0         |26148921  |1           |364239 |2              |d                                                                                                                                                                                                                   |
// |    |          |          |            |       |               |  and SPID not in (select spid from master.dbo.sysprocesses where program_name like 'AseTune%')                                                                                                                     |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    290|              1|select 1                                                                                                                                                                                                            |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    291|              1|select @@tranchained                                                                                                                                                                                                |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// |  72|         0|26542144|           1|    292|              1|select * from master..monSysSQLText                                                                                                                                                                                 |
// +----+----------+----------+------------+-------+---------------+--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
// Rows 24
// Client Exec Time: 00:00.002 (sqlExec=00:00.001, readResults=00:00.001, other=00:00.000), at '2023-04-28 22:57:43.873', for SQL starting at Line 13153





// 1> select * from master.dbo.monProcess
// RS> Col# Label               JDBC Type Name          Guessed DBMS type Source Table         
// RS> ---- ------------------- ----------------------- ----------------- ---------------------
// RS> 1    SPID                java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 2    InstanceID          java.sql.Types.TINYINT  tinyint           master.dbo.monProcess
// RS> 3    KPID                java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 4    ServerUserID        java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 5    OrigServerUserID    java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 6    BatchID             java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 7    ContextID           java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 8    LineNumber          java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 9    SecondsConnected    java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 10   DBID                java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 11   EngineNumber        java.sql.Types.SMALLINT smallint          master.dbo.monProcess
// RS> 12   Priority            java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 13   FamilyID            java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 14   Login               java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 15   Application         java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 16   Command             java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 17   NumChildren         java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 18   SecondsWaiting      java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 19   WaitEventID         java.sql.Types.SMALLINT smallint          master.dbo.monProcess
// RS> 20   BlockingSPID        java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 21   BlockingXLOID       java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 22   DBName              java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 23   EngineGroupName     java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 24   ExecutionClass      java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 25   MasterTransactionID java.sql.Types.VARCHAR  varchar(255)      master.dbo.monProcess
// RS> 26   HostName            java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 27   ClientName          java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 28   ClientHostName      java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 29   ClientApplName      java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 30   ClientDriverVersion java.sql.Types.VARCHAR  varchar(16)       master.dbo.monProcess
// RS> 31   ClientPort          java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 32   NetType             java.sql.Types.VARCHAR  varchar(30)       master.dbo.monProcess
// RS> 33   ListenerAddress     java.sql.Types.VARCHAR  varchar(64)       master.dbo.monProcess
// RS> 34   ListenerPort        java.sql.Types.INTEGER  int               master.dbo.monProcess
// RS> 35   SSLCommonName       java.sql.Types.VARCHAR  varchar(255)      master.dbo.monProcess
// RS> 36   SSLCipherName       java.sql.Types.VARCHAR  varchar(64)       master.dbo.monProcess
// RS> 37   SSLVersionName      java.sql.Types.VARCHAR  varchar(16)       master.dbo.monProcess
// +----+----------+----------+------------+----------------+-------+---------+----------+----------------+------+------------+--------+--------+------+------------------------+-----------------+-----------+--------------+-----------+------------+-------------+-----------+----------------+--------------+-------------------+------------+---------------+--------------+-----------------------------+-------------------+----------+-------+---------------+------------+-------------+-------------+--------------+
// |SPID|InstanceID|KPID      |ServerUserID|OrigServerUserID|BatchID|ContextID|LineNumber|SecondsConnected|DBID  |EngineNumber|Priority|FamilyID|Login |Application             |Command          |NumChildren|SecondsWaiting|WaitEventID|BlockingSPID|BlockingXLOID|DBName     |EngineGroupName |ExecutionClass|MasterTransactionID|HostName    |ClientName     |ClientHostName|ClientApplName               |ClientDriverVersion|ClientPort|NetType|ListenerAddress|ListenerPort|SSLCommonName|SSLCipherName|SSLVersionName|
// +----+----------+----------+------------+----------------+-------+---------+----------+----------------+------+------------+--------+--------+------+------------------------+-----------------+-----------+--------------+-----------+------------+-------------+-----------+----------------+--------------+-------------------+------------+---------------+--------------+-----------------------------+-------------------+----------+-------+---------------+------------+-------------+-------------+--------------+
// |   1|         0|   327683|           0|               0|      0|        0|         0|               0|     0|           0|       0|(NULL)  |(NULL)|(NULL)                  |MEMORY TUNE      |(NULL)     |           793|        780|(NULL)      |(NULL)       |(NULL)     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   3|         0|   458756|           0|               0|      0|        0|         0|       4770088|     1|           0|       2|(NULL)  |(NULL)|(NULL)                  |DEADLOCK TUNE    |(NULL)     |     4770087|        151|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   4|         0|   589829|           0|               0|      0|        0|         0|       4770088|     1|           0|       5|(NULL)  |(NULL)|(NULL)                  |SHUTDOWN HANDLER |(NULL)     |     4770087|        169|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   5|         0|   720902|           0|               0|      0|        0|         0|       4770088|     1|           0|       6|(NULL)  |(NULL)|(NULL)                  |KPP HANDLER      |(NULL)     |             0|          1|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   6|         0|   851975|           0|               0|      0|        0|         0|       4770088|     1|           0|       5|(NULL)  |probe |<astc>                  |ASTC HANDLER     |(NULL)     |            26|         19|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |GORAN_UB3_DS|(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   7|         0|   983048|           0|               0|      0|        0|         0|       4770088|     1|           0|       5|(NULL)  |(NULL)|(NULL)                  |CHECKPOINT SLEEP |(NULL)     |            26|         57|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   8|         0| 1114121|           0|               0|      0|        0|         0|       4770088|     1|           0|       7|(NULL)  |(NULL)|(NULL)                  |HK WASH          |(NULL)     |             0|         61|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |   9|         0| 1245194|           0|               0|      0|        0|         0|       4770088|     1|           0|       5|(NULL)  |(NULL)|(NULL)                  |HK GC            |(NULL)     |             3|         61|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |  10|         0| 1376267|           0|               0|      0|        0|         0|       4770088|     1|           0|       7|(NULL)  |(NULL)|(NULL)                  |HK CHORES        |(NULL)     |             2|         61|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |  11|         0| 1507340|           0|               0|      0|        0|         0|       4770088|31515|           0|       5|(NULL)  |(NULL)|(NULL)                  |AUDIT PROCESS    |(NULL)     |       104150|        169|(NULL)      |(NULL)       |sybsecurity|(NULL)          |(NULL)        |$audit_xact        |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |  12|         0| 1638413|           0|               0|      0|        0|         0|       4770088|     1|           0|       0|(NULL)  |(NULL)|(NULL)                  |PORT MANAGER     |(NULL)     |     4770087|        369|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |  13|         0| 1769486|           0|               0|      0|        0|         0|       4770088|     0|           0|       0|(NULL)  |(NULL)|(NULL)                  |NETWORK HANDLER  |(NULL)     |           148|        178|(NULL)      |(NULL)       |(NULL)     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |  16|         0| 2162705|           0|               0|      0|        0|         0|       4770087|     1|           0|       5|(NULL)  |(NULL)|(NULL)                  |LICENSE HEARTBEAT|(NULL)     |            88|        596|(NULL)      |(NULL)       |master     |(NULL)          |(NULL)        |(NULL)             |(NULL)      |(NULL)         |(NULL)        |(NULL)                       |0.0.0.0            |(NULL)    |(NULL) |(NULL)         |(NULL)      |(NULL)       |(NULL)       |(NULL)        |
// |  68|         0|26017850|           1|               0|141541|        0|         3|         104380|     1|           0|       5|(NULL)  |sa    |AseTune-nogui           |AWAITING COMMAND |(NULL)     |            25|        250|(NULL)      |(NULL)       |master     |syb_default_pool|EC2           |(NULL)             |gorans-ub3  |AseTune-nogui  |gorans-ub3    |AseTune-nogui - 4.5.0.8.dev  |16.0.4.0           |    58878|tcp    |192.168.0.164  |       1600|(NULL)       |(NULL)       |(NULL)        |
// |  70|         0|26148921|           1|               0|364351|        0|         2|         104379|     1|           0|       5|(NULL)  |sa    |AseTune-SqlCaptureBroker|AWAITING COMMAND |(NULL)     |             0|        250|(NULL)      |(NULL)       |master     |syb_default_pool|EC2           |(NULL)             |gorans-ub3  |(NULL)         |(NULL)        |(NULL)                       |16.0.4.0           |    58892|tcp    |192.168.0.164  |       1600|(NULL)       |(NULL)       |(NULL)        |
// |  72|         0|26542144|           1|               0|    315|        1|         1|             230|     2|           0|       5|(NULL)  |sa    |SqlWindow               |SELECT           |(NULL)     |(NULL)        |          0|(NULL)      |(NULL)       |tempdb     |syb_default_pool|EC2           |(NULL)             |gorans2     |SqlWindow      |gorans2       |SqlWindow - 4.5.0.8.dev      |16.0.4.0           |    61827|tcp    |192.168.0.164  |       1600|(NULL)       |(NULL)       |(NULL)        |
// |  74|         0|26673213|           1|               0|    534|        0|         1|             148|     2|           0|       5|(NULL)  |sa    |SqlWindow-Compl         |AWAITING COMMAND |(NULL)     |            94|        250|(NULL)      |(NULL)       |tempdb     |syb_default_pool|EC2           |(NULL)             |gorans2     |SqlWindow-Compl|gorans2       |SqlWindow-Compl - 4.5.0.8.dev|16.0.4.0           |    61846|tcp    |192.168.0.164  |       1600|(NULL)       |(NULL)       |(NULL)        |
// |  75|         0|26411070|           1|               0| 10570|        0|         0|         104375|     1|           0|       5|(NULL)  |sa    |AseTune-ObjInfoLookup   |AWAITING COMMAND |(NULL)     |            25|        250|(NULL)      |(NULL)       |master     |syb_default_pool|EC2           |(NULL)             |gorans-ub3  |(NULL)         |(NULL)        |(NULL)                       |16.0.4.0           |    58906|tcp    |192.168.0.164  |       1600|(NULL)       |(NULL)       |(NULL)        |
// +----+----------+----------+------------+----------------+-------+---------+----------+----------------+------+------------+--------+--------+------+------------------------+-----------------+-----------+--------------+-----------+------------+-------------+-----------+----------------+--------------+-------------------+------------+---------------+--------------+-----------------------------+-------------------+----------+-------+---------------+------------+-------------+-------------+--------------+
// Rows 18
// Client Exec Time: 00:00.002 (sqlExec=00:00.001, readResults=00:00.001, other=00:00.000), at '2023-04-28 22:58:16.355', for SQL starting at Line 13155
// 