/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.wqs;

import java.sql.Timestamp;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import sek.ase.auditor.collectors.records.AuditRecord;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.Memory;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.utils.TimeUtils;
import sek.ase.auditor.wqs.consumers.IWriterConsumer;


public class WriterQueueHandler 
implements Runnable
{
	private static Logger _logger = LogManager.getLogger();

	
	/*---------------------------------------------------
	** Constants
	**---------------------------------------------------
	*/
	public static final String  PROPKEY_warnQueueSizeThresh                      = "WriterQueueHandler.warnQueueSizeThresh";
	public static final int     DEFAULT_warnQueueSizeThresh                      = 3;

	public static final String  PROPKEY_writeInfoOnEachConsumerWrite             = "WriterQueueHandler.writeInfoOnEachConsumeWrite";
	public static final boolean DEFAULT_writeInfoOnEachConsumerWrite             = false;

	public static final String PROPKEY_statisticsMessagePeriodSeconds            = "WriterQueueHandler.statistics.message.period.seconds";
	public static final int    DEFAULT_statisticsMessagePeriodSeconds            = 300;


	public static final String  PROPKEY_WriterClass                              = "WriterQueueHandler.WriterClass";
	public static final String  DEFAULT_WriterClass                              = null; // no default
//	public static final String  DEFAULT_WriterClass                              = "sek.ase.auditor.wqs.consumers.WriterToSplunk"; // Used during development
//	public static final String  DEFAULT_WriterClass                              = "sek.ase.auditor.wqs.consumers.WriterToFile"; // Used during development
//	public static final String  DEFAULT_WriterClass                              = "sek.ase.auditor.wqs.consumers.WriterToFile, sek.ase.auditor.wqs.consumers.WriterToSplunk";

	
	/*---------------------------------------------------
	** class members
	**---------------------------------------------------
	*/
	// implements singleton pattern
	private static WriterQueueHandler _instance = null;

	private boolean  _initialized = false;
	private boolean  _running     = false;

	private Thread   _thread           = null;


	/** Configuration we were initialized with */
	private Configuration _conf;
	
	/** a list of installed Writers */
	private List<IWriterConsumer> _writerClasses = new LinkedList<IWriterConsumer>();

	/** Time when the current consume() method started */
	private long _currentConsumeStartTime = 0;

	/** */
	private BlockingQueue<WriterQueueContainer> _containerQueue = new LinkedBlockingQueue<WriterQueueContainer>();
	private int _warnQueueSizeThresh = DEFAULT_warnQueueSizeThresh;

	/** How many milliseconds have we maximum spent in <i>consume</i> */
	private long _maxConsumeTime      = 0;
	/** Last time the _maxConsumeTime was bumped */
	private long _maxConsumeTimeStamp = System.currentTimeMillis();

	private boolean _writeInfoOnEachConsumerWrite = DEFAULT_writeInfoOnEachConsumerWrite;

	private long _statisticsMessageLastTime  = System.currentTimeMillis();
	private long _statisticsMessagePeriodSec = DEFAULT_statisticsMessagePeriodSeconds;
	

	/*---------------------------------------------------
	** Constructors
	**---------------------------------------------------
	*/
	public WriterQueueHandler()
	throws Exception
	{
	}

//	public WriterQueueHandler(Configuration props, IObjectLookupInspector objectLookupInspector, ISqlCaptureBroker sqlCaptureBroker)
//	throws Exception
//	{
//		_objectLookupInspector = objectLookupInspector;
//		_sqlCaptureBroker      = sqlCaptureBroker;
//		init(props);
//	}

	public Configuration getConfig()
	{
		return _conf;
	}

	/**
	 * Get a "public" string of how all writer are configured, no not reveal
	 * passwords or sensitive information.
	 */
	public String getConfigStr()
	{
		String configStr = "";

		// loop all writer classes
		for (IWriterConsumer pw : _writerClasses)
		{
			configStr += pw.getName() + "={" + pw.getConfigStr() + "}, ";
		}
		if (configStr.length() > 0)
			configStr = configStr.substring(0, configStr.length()-2);
		
		return configStr;
	}
	
	
	/** Initialize various member of the class */
	public synchronized void init(Configuration props)
	throws Exception
	{
		_conf = props; 
		
		_logger.info("Initializing the 'Writer Queue Handler' functionality.");

		_warnQueueSizeThresh                 = _conf.getIntProperty    (PROPKEY_warnQueueSizeThresh,            DEFAULT_warnQueueSizeThresh);
		_writeInfoOnEachConsumerWrite        = _conf.getBooleanProperty(PROPKEY_writeInfoOnEachConsumerWrite,   DEFAULT_writeInfoOnEachConsumerWrite);

		// Statistics Message Period
		_statisticsMessagePeriodSec          = _conf.getLongProperty   (PROPKEY_statisticsMessagePeriodSeconds, DEFAULT_statisticsMessagePeriodSeconds);

		// property: alarm.handleAlarmEventClass
		// NOTE: this could be a comma ',' separated list
		String writerClasses = _conf.getProperty(PROPKEY_WriterClass, DEFAULT_WriterClass);

		int len = 60;

		_logger.info("Configuration for 'WriterQueueHandler', with full class name '" + getClass().getName() + "'.");
		_logger.info("                   " + StringUtil.left(PROPKEY_WriterClass                   , len) + " = "  + writerClasses);
		_logger.info("                   " + StringUtil.left(PROPKEY_warnQueueSizeThresh           , len) + " = " + _warnQueueSizeThresh);
		_logger.info("                   " + StringUtil.left(PROPKEY_writeInfoOnEachConsumerWrite  , len) + " = " + _writeInfoOnEachConsumerWrite);
		_logger.info("                   " + StringUtil.left(PROPKEY_statisticsMessagePeriodSeconds, len) + " = " + _statisticsMessagePeriodSec);
		
		// Check writer classes
		if (writerClasses == null)
		{
//			throw new Exception("The property 'WriterQueueHandler.WriterClass' is mandatory for the WriterQueueHandler module. It should contain one or several classes that implemets the IWriterConsumer interface. If you have more than one writer, specify them as a comma separated list.");
			_logger.info("No counters will be persisted. The property 'WriterQueueHandler.WriterClass' is not found in configuration for the WriterQueueHandler module. It should contain one or several classes that implemets the IWriterConsumer interface. If you have more than one writer, specify them as a comma separated list.");
		}
		else
		{
			String[] writerClassArray =  writerClasses.split(",");
			for (int i=0; i<writerClassArray.length; i++)
			{
				writerClassArray[i] = writerClassArray[i].trim();
				String writerClassName = writerClassArray[i];
				IWriterConsumer writerClass;
	
				_logger.debug("Instantiating and Initializing WriterClass='" + writerClassName + "'.");
//				_logger.info("Instantiating and Initializing WriterClass='" + writerClassName + "'.");
				try
				{
					Class<?> c = Class.forName( writerClassName );
					writerClass = (IWriterConsumer) c.newInstance();
					_writerClasses.add( writerClass );
				}
				catch (ClassCastException e)
				{
					throw new ClassCastException("When trying to load writerWriter class '" + writerClassName + "'. The writerWriter do not seem to follow the interface 'sek.ase.auditor.wqs.consumers.IWriterConsumer'");
				}
				catch (ClassNotFoundException e)
				{
					throw new ClassNotFoundException("Tried to load writerWriter class '" + writerClassName + "'.", e);
				}
	
				// Now initialize the User Defined AlarmWriter
				writerClass.init(_conf);
				writerClass.printConfig();
				writerClass.startServices();
			}
			if (_writerClasses.size() == 0)
			{
				_logger.warn("No Persistent Counter Writers has been installed, NO counters will be saved.");
			}
		}

		_initialized = true;
	}

	/*---------------------------------------------------
	** Methods
	**---------------------------------------------------
	*/
	
	//////////////////////////////////////////////
	//// Instance
	//////////////////////////////////////////////
	public static WriterQueueHandler getInstance()
	{
		return _instance;
	}

	public static boolean hasInstance()
	{
		return (_instance != null);
	}

	public static void setInstance(WriterQueueHandler inst)
	{
		_instance = inst;
	}

	
	//////////////////////////////////////////////
	//// memory
	//////////////////////////////////////////////
	/** Keep track of how many calls we have been made to lowOnMemoryHandler() */
	private int _lowOnMemoryHandlerCalls = 0;
	private int _lowOnMemoryHandlerCallsThreshold = 20;

	public void lowOnMemoryHandler()
	{
		// If lowOnMemoryHandler() has been called to many times, simply call outOfMemory() to do more extensive cleanup.
		_lowOnMemoryHandlerCalls++;
		if (_lowOnMemoryHandlerCalls > _lowOnMemoryHandlerCallsThreshold)
		{
			_logger.warn("Persistant Counter Handler, lowOnMemoryHandler() has now been called " + _lowOnMemoryHandlerCalls + " times, decided to call outOfMemoryHandler() to do more extensive cleanup.");
			outOfMemoryHandler();
			_lowOnMemoryHandlerCalls = 0;
			return;
		}

		boolean fire = false;

//		// Clear the DDL queues
//		if (_ddlInputQueue.size() > 0)
//		{
//			_logger.warn("Persistant Counter Handler, lowOnMemoryHandler() was called. Emtying the DDL Lookup Input queue, which has " + _ddlInputQueue.size() + " entries.");
//			_ddlInputQueue.clear();
//			fire = true;
//		}
//		if (_ddlStoreQueue.size() > 0)
//		{
//			_logger.warn("Persistant Counter Handler, lowOnMemoryHandler() was called. Emtying the DDL Store/Write queue, which has " + _ddlStoreQueue.size() + " entries.");
//			_ddlStoreQueue.clear();
//			fire = true;
//		}
//
//		// SQL Capture queues
//		if (_sqlCaptureStoreQueue.size() > 0)
//		{
//			_logger.warn("Persistant Counter Handler, lowOnMemoryHandler() was called. Emtying the SQL Capture Store/Write queue, which has " + _sqlCaptureStoreQueue.size() + " entries.");
//			_sqlCaptureStoreQueue.clear();
//			fire = true;
//		}
//
//		// SQL Capture Broker
//		if (_sqlCaptureBroker != null)
//		{
//			_sqlCaptureBroker.lowOnMemoryHandler();
//			fire = true;
//		}

		if (fire)
			fireQueueSizeChange();		
	}

	/** Keep track of how many calls we have been made to outOfMemoryHandler() */
	private int _outOfMemoryHandlerCallCount = 0;
	public void outOfMemoryHandler()
	{
		_outOfMemoryHandlerCallCount++;
		_logger.warn("Persistant Counter Handler, outOfMemoryHandler() was called. callCount=" + _outOfMemoryHandlerCallCount + ".");

		boolean fire = false;

//		// Clear the DDL queues
//		if (_ddlInputQueue.size() > 0)
//		{
//			_logger.warn("Persistant Counter Handler, outOfMemoryHandler() was called. Emtying the DDL Lookup Input queue, which has " + _ddlInputQueue.size() + " entries.");
//			_ddlInputQueue.clear();
//			fire = true;
//		}
//		if (_ddlStoreQueue.size() > 0)
//		{
//			_logger.warn("Persistant Counter Handler, outOfMemoryHandler() was called. Emtying the DDL Store/Write queue, which has " + _ddlStoreQueue.size() + " entries.");
//			_ddlStoreQueue.clear();
//			fire = true;
//		}
//
//		// SQL Capture queues
//		if (_sqlCaptureStoreQueue.size() > 0)
//		{
//			_logger.warn("Persistant Counter Handler, outOfMemoryHandler() was called. Emtying the SQL Capture Store/Write queue, which has " + _sqlCaptureStoreQueue.size() + " entries.");
//			_sqlCaptureStoreQueue.clear();
//			fire = true;
//		}
//
//		// SQL Capture Broker
//		if (_sqlCaptureBroker != null)
//		{
//			_sqlCaptureBroker.outOfMemoryHandler();
//			fire = true;
//		}

		// Clear the PCS queues if the outOfMemoryHandler() has been called more that X number of times since we last emptied the queue
		if (_outOfMemoryHandlerCallCount > 3)
		{
			_logger.warn("Persistant Counter Handler, outOfMemoryHandler() was called. Emtying the Counter Store/Write queue, which has " + _containerQueue.size() + " entries.");
			_containerQueue.clear();
			_outOfMemoryHandlerCallCount = 0;
			fire = true;
		}

		if (fire)
			fireQueueSizeChange();
	}

	
	//////////////////////////////////////////////
	//// xxx
	//////////////////////////////////////////////
	
	public void add(WriterQueueContainer cont)
	{
		if (_writerClasses.size() == 0)
			return;
		if ( ! isRunning() )
		{
			_logger.warn("The Persistent Counter Handler is not running, discarding entry.");
			return;
		}

		int qsize = _containerQueue.size();
		if (qsize > _warnQueueSizeThresh)
		{
			long currentConsumeTimeMs    = System.currentTimeMillis() -_currentConsumeStartTime;
			String currentConsumeTimeStr = "The consumer is currently not active. ";
			if (_currentConsumeStartTime > 0)
				currentConsumeTimeStr = "The current consumer has been active for " + TimeUtils.msToTimeStr(currentConsumeTimeMs) + ". ";

			_logger.warn("The persistent queue has " + qsize + " entries. The persistent writer might not keep in pace. " + currentConsumeTimeStr);

			// call each writes to let them know about this.
			for (IWriterConsumer pw : _writerClasses)
			{
				pw.queueSizeWarning(qsize, _warnQueueSizeThresh);
			}
		}

		_containerQueue.add(cont);
		fireQueueSizeChange();
	}

	private void isInitialized()
	{
		if ( ! _initialized )
		{
			throw new RuntimeException("The Persistent Counter Handler module has NOT yet been initialized.");
		}
	}




	/**
	 * Use all installed writers to store the Persist information
	 * 
	 * @param cont
	 * @param prevConsumeTimeMs
	 */
	private void consume(WriterQueueContainer cont, long prevConsumeTimeMs)
	{
		// Should we CLONE() the cont, this since it will be passed to several writers
		// each of the writer is changing the sessionStartTime in the container.
		// For the moment: copy/restore the cont.sessionStartTime...
//		Timestamp initialSessionStartTime = cont.getSessionStartTime();
		
		// NOTE: 
		// SessionStartTime IS SET and dictaded by the writer (at start the above 'cont.getSessionStartTime()' is most likly NULL
		// and it's the Writer instance that saves/handles the SessionStartDate and does a new startSession()
		//
		
		// loop all writer classes
		for (IWriterConsumer pw : _writerClasses)
		{
			// if we are about to STOP the service
			if ( ! isRunning() )
			{
				_logger.info("The service is about to stop, discarding a consume(ObjectLookupQueueEntry:Input) queue entry.");
				continue;
			}

			// Set/restore the original sessionStartTime
//			cont.setSessionStartTime(initialSessionStartTime);

			// Start the clock
			long startTime = System.currentTimeMillis();

			// CALL THE installed Writer
			// AND catch all runtime errors that might come
			try 
			{
//				_logger.info("Persisting Counters using '" + pw.getName() + "' for sessionStartTime='" + cont.getSessionStartTime() + "', mainSampleTime='" + cont.getMainSampleTime() + "'. Previous persist took " + prevConsumeTimeMs + " ms. inserts=" + pw.getInserts() + ", updates=" + pw.getUpdates() + ", deletes=" + pw.getDeletes() + ", createTables=" + pw.getCreateTables() + ", alterTables=" + pw.getAlterTables() + ", dropTables=" + pw.getDropTables() + ".");

				// BEGIN-OF-SAMPLE If we want to do anything in here
				pw.beginOfSample(cont);

//				// If a Session has not yet been started (or you need to "restart" one), go and do that.
//				if ( ! pw.isSessionStarted() || cont.getStartNewSample() )
//				{
//					Timestamp newTs = cont.getMainSampleTime();
//					cont.setSessionStartTime(newTs);
//					pw.setSessionStartTime(newTs);
//
//					pw.startSession(cont);
//					// note: the above pw.startSession() must call: pw.setSessionStarted(true);
//					// otherwise we will run this everytime
//				}

//				// Set the Session Start Time in the container.
//				Timestamp newTs = pw.getSessionStartTime();
//				cont.setSessionStartTime(newTs);

//				// CREATE-DDL
//				for (CountersModel cm : cont.getCounterObjects())
//				{
//					// only call saveDdl() the first time...
//					if ( ! pw.isDdlCreated(cm) )
//					{
//						if (pw.saveDdl(cm))
//						{
//							pw.markDdlAsCreated(cm);
//						}
//					}
//				}

				// SAVE-SAMPLE
				// In here we can "do it all" 
				// or use: beginOfSample(), saveDdl(), saveCounters(), endOfSample()
				pw.saveSample(cont);

				
				// SAVE-COUNTERS
				for (AuditRecord alarmRecord : cont.getAuditRecords())
				{
					pw.saveRecord(alarmRecord);
				}

				
				// END-OF-SAMPLE If we want to do anything in here
				pw.endOfSample(cont, false);

				// Stop clock and print statistics.
				long execTime = TimeUtils.msDiffNow(startTime);
				setConsumeTime(execTime);

				firePcsConsumeInfo(pw.getName(), cont.getServerName(), cont.getMainSampleTime(), (int)execTime, pw.getStatistics());

				// Reset the statistics
//				pw.resetCounters();
			}
			catch (Throwable t)
			{
				_logger.error("The Persistent Writer got runtime error in consume() in Persistent Writer named '" + pw.getName() + "'. Continuing with next Writer...", t);
				pw.endOfSample(cont, true);
			}
		} // end: consume/writer

		// Is it time to print some statistical messages (this should be at the TOP of the loop, otherwise it wont be printed if no data is fetched.)
		if (TimeUtils.secondsDiffNow(_statisticsMessageLastTime) >= _statisticsMessagePeriodSec)
		{
			// First reset the time
			_statisticsMessageLastTime = System.currentTimeMillis();

			for (IWriterConsumer pw : _writerClasses)
			{
				String writerName = pw.getName();
				WriterQueueStatistics writerStats = pw.getStatistics();
				
				_logger.info("STATISTICS[" + writerName + "]: " + writerStats); 
			}
		}
	}

	private void setConsumeTime(long lastExecTime)
	{
//		_maxConsumeTime = Math.max(_maxConsumeTime, lastExecTime);
		
		if (lastExecTime > _maxConsumeTime)
		{
			_maxConsumeTime      = lastExecTime;
			_maxConsumeTimeStamp = System.currentTimeMillis();
		}
		else
		{
			// if we have a "bump in the road", meaning: one persist took an awfull long time, but normally this does not happen
			// then: lets slowly decrease the '_maxConsumeTime'
			
			// FIXME: how should this algorithm look like?   half the _maxConsumeTime every time, or only after ### ms from _maxConsumeTimeStamp...
			//        for now the PersistWriterToHttpJson.checkSendAlarm() uses a max value of 10 minutes...
			//_maxConsumeTime = _maxConsumeTime / 2;
		}
	}

	/**
	 * Get MAX time in milliseconds we have spent in consume for any writer
	 * @return
	 */
	public long getMaxConsumeTime()
	{
		return _maxConsumeTime;
	}
	
//	/** 
//	 * When we start a new session, lets call this method to get some 
//	 * idea what we are about to sample. 
//	 * @param cont a WriterQueueContainer filled with <b>all</b> the available
//	 *             CounterModels we could sample.
//	 */
//	public void startSession(WriterQueueContainer cont)
//	{
//		Iterator<IWriterConsumer> writerIter = _writerClasses.iterator();
//		while (writerIter.hasNext()) 
//		{
//			IWriterConsumer pw = writerIter.next();
//
//			// CALL THE installed Writer
//			// AND catch all runtime errors that might come
//			try 
//			{
//				_logger.info("Starting Counters Storage Session '" + pw.getName() + "' for sessionStartTime='" + cont.getSessionStartTime() + "', server='" + cont.getServerName() + "', with serverAlias='" + cont.getServerNameAlias() + "', displayName='" + cont.getServerDisplayName() + "'.");
//
//				pw.startSession(cont);
//			}
//			catch (Throwable t)
//			{
//				_logger.error("The Persistent Writer got runtime error when calling the method startSession() in Persistent Writer named '" + pw.getName() + "'. Continuing with next Writer...", t);
//			}
//		}
//	}

	
	/**
	 * Read from the Container "in" queue, and use all Writers to save DATA 
	 */
	@Override
	public void run()
	{
		String threadName = _thread.getName();
		_logger.info("Starting a thread for the module '" + threadName + "'.");

		isInitialized();

		_running = true;
		long prevConsumeTimeMs = 0;

		while(isRunning())
		{
			//_logger.info("Thread '" + _thread.getName() + "', SLEEPS...");
			//try { Thread.sleep(5 * 1000); }
			//catch (InterruptedException ignore) {}
			
			if (_logger.isDebugEnabled())
				_logger.debug("Thread '" + threadName + "', waiting on queue...");

			try 
			{
				WriterQueueContainer cont = _containerQueue.take();
				fireQueueSizeChange();

				// Make sure the container isn't empty.
				if (cont == null || (cont != null && cont.isEmpty()) )
					continue;

				// if we are about to STOP the service
				if ( ! isRunning() )
				{
					_logger.info("The service is about to stop, discarding a consume(WriterQueueContainer) queue entry.");
					continue;
				}

				// Go and store or consume the in-data/container
				_currentConsumeStartTime = System.currentTimeMillis();
				long startTime = System.currentTimeMillis();

				//--------------------------------------------
				// Here is where it's all done
				//--------------------------------------------
				consume( cont, prevConsumeTimeMs );

				long stopTime = System.currentTimeMillis();
				_currentConsumeStartTime = 0;

				prevConsumeTimeMs = stopTime-startTime;
				_logger.debug("It took " + prevConsumeTimeMs + " ms to persist the above information (using all writers).");
			} 
			catch (InterruptedException ex) 
			{
				_running = false;
			}
		}

		_logger.info("Emptying the queue for module '" + threadName + "', which had " + _containerQueue.size() + " entries.");
		_containerQueue.clear();
		fireQueueSizeChange();

		_logger.info("Thread '" + threadName + "' was stopped.");
	}

	/**
	 * Are we running or not
	 */
	public boolean isRunning()
	{
		return _running;
	}

	/**
	 * Start this subsystem
	 */
	public void start()
	{
		if (_writerClasses.size() == 0)
		{
			_logger.warn("No Persistent Counter Writers has been installed, The service thread will NOT be started and NO counters will be saved.");
			return;
		}

		isInitialized();

		// Start the Container Persist Thread
		_thread = new Thread(this);
		_thread.setName("WriterQueueHandler");
		_thread.setDaemon(true);
		_thread.start();
	}

	/**
	 * Stop this subsystem
	 */
	public void stop(boolean clearQueues, int maxWaitTimeInMs)
	{
		_running = false;

		if (clearQueues)
		{
			_containerQueue.clear();
			fireQueueSizeChange();
		}

		if (_thread != null)
		{
			_thread.interrupt();
		}

		// Close the connections to the data store.
		for (IWriterConsumer pw : _writerClasses)
		{
			// Do this first, otherwise the wait time wont work, if close(); closes the db connection
			pw.stopServices(maxWaitTimeInMs);

			// Note: that stop service might stop the service before we disconnect from it
			//       so you could expect some error messages, which could be discarded
			pw.close();
		}

		// Wait for thread to end 
		try { _thread.join(); }
		catch (InterruptedException ignore) {}

		_thread = null;
		_logger.info("Done Stopping all Writers.");
	}
	

	/**
	 * Check if we have any writers installed/attached
	 * @return
	 */
	public boolean hasWriters()
	{
		return (_writerClasses.size() > 0);
	}

	/**
	 * Get installed writer classes
	 * @return
	 */
	public List<IWriterConsumer> getWriters()
	{
		return _writerClasses;
	}
	


	/*---------------------------------------------------
	** Listener stuff
	**---------------------------------------------------
	*/
	/** interface to be invoked when any of the internal queue sizes is changed */
	public interface WriterQueueChangeListener
	{
		/**
		 * This will be called when any of the queue changes in size
		 * @param pcsQueueSize          Queue size of the in bound container queue
		 * @param ddlLookupQueueSize    Queue size of the DDL Lookups to be done
		 * @param ddlStoreQueueSize     Queue size of the DDL Storage to be done
		 */
		public void queueChange(int pcsQueueSize);

//		/**
//		 * This will be called once for each of the Persist Writers
//		 * @param persistWriterName   Name of the Writer
//		 * @param sessionStartTime    Time when the SESSION started to collect data
//		 * @param mainSampleTime      Time for last main period (if sample time is 10 seconds, this will be the "head" time for all subsequent Performance Counters individual times)
//		 * @param persistTimeInMs     Number of milliseconds it took for the Performance Writer to store the data
//		 * @param inserts             Number of insert operations that was done for this save
//		 * @param updates             Number of update operations that was done for this save
//		 * @param deletes             Number of delete operations that was done for this save
//		 * @param createTables        Number of create table operations that was done for this save
//		 * @param alterTables         Number of alter table operations that was done for this save
//		 * @param dropTables          Number of drop table operations that was done for this save
//		 */
//		public void pcsConsumeInfo(String persistWriterName, Timestamp sessionStartTime, Timestamp mainSampleTime, int persistTimeInMs, int inserts, int updates, int deletes, int createTables, int alterTables, int dropTables, int ddlSaveCount, int ddlSaveCountSum);

		/**
		 * This will be called once for each of the Persist Writers
		 * @param persistWriterName   Name of the Writer
		 * @param sessionStartTime    Time when the SESSION started to collect data
		 * @param mainSampleTime      Time for last main period (if sample time is 10 seconds, this will be the "head" time for all subsequent Performance Counters individual times)
		 * @param persistTimeInMs     Number of milliseconds it took for the Performance Writer to store the data
		 * @param writerStatistics    The actual statistical counters object
		 */
		public void consumeInfo(String persistWriterName, Timestamp mainSampleTime, int persistTimeInMs, WriterQueueStatistics writerQueueStatistics);
	}

	/** listeners */
	Set<WriterQueueChangeListener> _queueChangeListeners = new HashSet<WriterQueueChangeListener>();

	/** Add any listeners that want to see changes */
	public void addChangeListener(WriterQueueChangeListener l)
	{
		_queueChangeListeners.add(l);
	}

	/** Remove the listener */
	public void removeChangeListener(WriterQueueChangeListener l)
	{
		_queueChangeListeners.remove(l);
	}

	/** Kicked off when new entries are added */
	protected void fireQueueSizeChange()
	{
		int pcsQueueSize         = _containerQueue.size();

		for (WriterQueueChangeListener l : _queueChangeListeners)
			l.queueChange(pcsQueueSize);
	}

	/** Kicked off when consume is done */ 
	public void firePcsConsumeInfo(String persistWriterName, String serverName, Timestamp mainSampleTime, int persistTimeInMs, WriterQueueStatistics writerQueueStatistics)
	{
		int    pcsQueueSize = _containerQueue.size();

		if (persistWriterName == null) persistWriterName = "-unknown-";
		if (serverName        == null) serverName        = "-unknown-";
		
		_maxLenPersistWriterName = Math.max(_maxLenPersistWriterName, persistWriterName.length());
		_maxLenServerName        = Math.max(_maxLenServerName,        serverName       .length());

		if (_writeInfoOnEachConsumerWrite)
		{
			_logger.info("Persisting Counters using " + StringUtil.left("'" + persistWriterName + "', ", _maxLenPersistWriterName+4)
				+ "for serverName="               + StringUtil.left("'" + serverName        + "', ", _maxLenServerName+4)
//				+ "sessionStartTime='"            + StringUtil.left(sessionStartTime + "",23)     + "', "
				+ "mainSampleTime='"              + StringUtil.left(mainSampleTime   + "",23)     + "'. "
				+ "This persist took ["           + TimeUtils.msToTimeStrShort(persistTimeInMs) + "] " + StringUtil.left(persistTimeInMs + " ms. ", 10)
				+ "qs="                           + StringUtil.left(pcsQueueSize + ".", 4) // ###.
				+ "jvmMemoryLeftInMB="            + Memory.getMemoryLeftInMB() + ". " 
				+ (writerQueueStatistics == null ? "" : writerQueueStatistics) );
		}

		for (WriterQueueChangeListener l : _queueChangeListeners)
			l.consumeInfo(persistWriterName, mainSampleTime, persistTimeInMs, writerQueueStatistics);
	}
	// Below are just used for formating the above string (for readability)
	private int _maxLenPersistWriterName = 0;
	private int _maxLenServerName        = 0;


	//////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////
	//// ---- TEST CODE ---- TEST CODE ---- TEST CODE ---- TEST CODE ----
	//////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////
//	public static void main(String[] args) 
//	{
//	}
}





//    ###### capture SQL
//    RS> Col# Label             JDBC Type Name           Guessed DBMS type Source Table              
//    RS> ---- ----------------- ------------------------ ----------------- --------------------------
//    RS> 1    SPID              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 2    KPID              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 3    BatchID           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 4    LineNumber        java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 5    dbname            java.sql.Types.VARCHAR   varchar(30)       -none-                    
//    RS> 6    procname          java.sql.Types.VARCHAR   varchar(255)      -none-                    
//    RS> 7    Elapsed_ms        java.sql.Types.INTEGER   int               -none-                    
//    RS> 8    CpuTime           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 9    WaitTime          java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 10   MemUsageKB        java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 11   PhysicalReads     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 12   LogicalReads      java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 13   RowsAffected      java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 14   ErrorStatus       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 15   ProcNestLevel     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 16   StatementNumber   java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 17   PagesModified     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 18   PacketsSent       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 19   PacketsReceived   java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 20   NetworkPacketSize java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 21   PlansAltered      java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 22   StartTime         java.sql.Types.TIMESTAMP datetime          master.dbo.monSysStatement
//    RS> 23   EndTime           java.sql.Types.TIMESTAMP datetime          master.dbo.monSysStatement
//    RS> 24   PlanID            java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 25   DBID              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 26   ObjOwnerID        java.sql.Types.INTEGER   int               -none-                    
//    RS> 27   ProcedureID       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    
//    
//    ###### select * from dbo.monSysStatement where SPID not in (select spid from master..sysprocesses p where p.program_name like 'AseTune%')
//    
//    RS> Col# Label             JDBC Type Name           Guessed DBMS type Source Table              
//    RS> ---- ----------------- ------------------------ ----------------- --------------------------
//    RS> 1    SPID              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 2    InstanceID        java.sql.Types.TINYINT   tinyint           master.dbo.monSysStatement
//    RS> 3    KPID              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 4    DBID              java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 5    ProcedureID       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 6    PlanID            java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 7    BatchID           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    ++++RS> 8    ContextID         java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 9    LineNumber        java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 10   CpuTime           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 11   WaitTime          java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 12   MemUsageKB        java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 13   PhysicalReads     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 14   LogicalReads      java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 15   PagesModified     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 16   PacketsSent       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 17   PacketsReceived   java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 18   NetworkPacketSize java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 19   PlansAltered      java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 20   RowsAffected      java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 21   ErrorStatus       java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    +++RS> 22   HashKey           java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    +++RS> 23   SsqlId            java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 24   ProcNestLevel     java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 25   StatementNumber   java.sql.Types.INTEGER   int               master.dbo.monSysStatement
//    RS> 26   DBName            java.sql.Types.VARCHAR   varchar(30)       master.dbo.monSysStatement
//    RS> 27   StartTime         java.sql.Types.TIMESTAMP datetime          master.dbo.monSysStatement
//    RS> 28   EndTime           java.sql.Types.TIMESTAMP datetime          master.dbo.monSysStatement
//    
//    
//    ###### select * from dbo.monSysSQLText   where SPID not in (select spid from master..sysprocesses p where p.program_name like 'AseTune%')
//    RS> Col# Label           JDBC Type Name         Guessed DBMS type Source Table            
//    RS> ---- --------------- ---------------------- ----------------- ------------------------
//    RS> 1    SPID            java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//    RS> 2    InstanceID      java.sql.Types.TINYINT tinyint           master.dbo.monSysSQLText
//    RS> 3    KPID            java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//    RS> 4    ServerUserID    java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//    RS> 5    BatchID         java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//    RS> 6    SequenceInBatch java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//    RS> 7    SQLText         java.sql.Types.VARCHAR varchar(255)      master.dbo.monSysSQLText
//    +----+----------+--------+------------+-------+---------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
//    |SPID|InstanceID|KPID    |ServerUserID|BatchID|SequenceInBatch|SQLText                                                                                                                                                                          |
//    +----+----------+--------+------------+-------+---------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
//    |14  |0         |32571433|1           |989    |1              |select dbname=db_name(), spid=@@spid, username = user_name(), susername =suser_name(), trancount=@@trancount, tranchained=@@tranchained, transtate=@@transtate                   |
//    |14  |0         |32571433|1           |990    |1              |select dbname=db_name(dbid), table_name=object_name(id, dbid), lock_type=type, lock_count=count(*)  from master.dbo.syslocks  where spid = @@spid         group by dbid, id, type|
//    |14  |0         |32571433|1           |992    |1              |select @@tranchained                                                                                                                                                             |
//    |14  |0         |32571433|1           |994    |1              |select @@tranchained                                                                                                                                                             |
//    |14  |0         |32571433|1           |995    |1              |select * from dbo.monSysSQLText   where SPID not in (select spid from master..sysprocesses p where p.program_name like 'AseTune%')                                               |
//    +----+----------+--------+------------+-------+---------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
//    Rows 5
//    
