/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.wqs.consumers;

import sek.ase.auditor.collectors.records.AuditRecord;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.wqs.WriterQueueContainer;
import sek.ase.auditor.wqs.WriterQueueHandler;
import sek.ase.auditor.wqs.WriterQueueStatistics;


/**
 * Implement you'r own forwarding of Alarms to various subsystems.
 * <p>
 * You can choose what methods you want to implement
 * <ul>
 *     <li>beginOfSample(WriterQueueContainer)          -- is called at the start of each Container "loop"</li>
 *     <li>saveSample(WriterQueueContainer)             -- is called once for each container</li>
 *     <li>saveRecord(AuditRecord)                      -- is called once for every AuditRecord in the container</li>
 *     <li>beginOfSample(WriterQueueContainer, boolean) -- is called at the end of each Container "loop"</li>
 * </ul>
 * So you can basically choose what method you like to use
 * 
 * @author Goran Schwarz
 */
public interface IWriterConsumer 
{
	/**
	 * The init() method will be called
	 * so it can configure itself. Meaning reading the "props" and initialize itself.
	 * 
	 * @param props The Configuration (basically a Properties object)
	 * @throws Exception when the initialization fails, you will 
	 *         throw an Exception to tell what was wrong
	 */
	public void init(Configuration props)	
	throws Exception;

	/**
	 * Called when we are about to stop the service, before stopServices (if you want to do anything)
	 */
	public void close();

	/**
	 * Get the configuration which this writer is using.
	 */
	public Configuration getConfig();

	/**
	 * Get a "public" string of how the the writer is configured, no not reveal
	 * passwords or sensitive information.
	 */
	public String getConfigStr();

	/**
	 * Print Writer Configuration to the application log
	 */
	public void printConfig();

//	/** 
//	 * When we start a new session, lets call this method to get some idea what we are about to sample.
//	 * <p>
//	 * @param cont a WriterQueueContainer filled with <b>all</b> the available
//	 *             CounterModels we could sample.
//	 */
//	public void startSession(WriterQueueContainer cont);
	
	/**
	 * Called from the {@link WriterQueueHandler#consume} as the first thing it does.
	 * @param cont 
	 * @see WriterQueueHandler#consume
	 */
	public void beginOfSample(WriterQueueContainer cont);

	/**
	 * Called from the {@link WriterQueueHandler#consume}.
	 * Save a bunch of CM's that we have sampled during this sample interval.
	 * @param cont
	 */
	public void saveSample(WriterQueueContainer cont);

	/**
	 * Called from the {@link WriterQueueHandler#consume} to save a single entry.
	 * Save a single AlarmRecord
	 * @param cont
	 */
	public void saveRecord(AuditRecord ar);

	/**
	 * Called from the {@link WriterQueueHandler#consume} as the last thing it does.
	 * @param cont 
	 * @see WriterQueueHandler#consume
	 */
	public void endOfSample(WriterQueueContainer cont, boolean caughtErrors);



	/**
	 * Start various service threads etc that this module needs
	 * @throws Exception
	 */
	public void startServices() throws Exception;

	/**
	 * Stop various service threads etc that this module started in startServices()
	 * 
	 * @param maxWaitTimeInMs maximum time that that the service can wait before gracefully shutdown.
	 *                                0 means, do shutdown now, or "without wait"
	 */
	public void stopServices(int maxWaitTimeInMs);

	/**
	 * The writer has to have some cind of name...
	 * 
	 * @return name of the Writer
	 */
	public String getName();

	/**
	 * Called when the queue size is higher that the warning threshold<br>
	 * The idea is that you can do various stuff in the Writes to help resolving this issue.
	 * @param queueSize
	 * @param thresholdSize 
	 */
	public void queueSizeWarning(int queueSize, int thresholdSize);

	/**
	 * Get various statistics from the Writer (how it's performing)
	 * @return
	 */
	public WriterQueueStatistics getStatistics();

	/**
	 * Reset the statistics object
	 */
	public void resetStatistics();
}
