/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.collectors;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import sek.ase.auditor.collectors.SybSysProcessesCache.SybSysProcessesEntry;
import sek.ase.auditor.collectors.records.AuditRecordSybAudit;
import sek.ase.auditor.utils.AseUtils;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.wqs.WriterQueueContainer;
import sek.ase.auditor.wqs.WriterQueueContainer.HeaderInfo;

/**
 * Collects information from the Sybase auditing database <code>sybsecurity</code>
 * 
 * @author goran
 */
public class SybAuditCollector
extends SybAbstractCollector
{
	private static Logger _logger = LogManager.getLogger();

	private Timestamp _lastEntryTs;
	
	public static final String  PROPKEY_sleepTimeInSeconds   = "SybAuditCollector.sleepTime.seconds";
	public static final int     DEFAULT_sleepTimeInSeconds   = 10;

	public static final String  PROPKEY_sampleJdbcTimeout    = "SybAuditCollector.sample.jdbc.timout"; 
	public static final int     DEFAULT_sampleJdbcTimeout    = 300;

	public static final String  PROPKEY_printGetData         = "SybAuditCollector.print.getData";
	public static final boolean DEFAULT_printGetData         = false;
	
	private int     _sampleJdbcTimeout    = DEFAULT_sampleJdbcTimeout;
	private boolean _printGetData         = DEFAULT_printGetData;

	@Override
	public void init(Configuration config)
	throws Exception
	{
		super.init(config);
		
		_sampleJdbcTimeout    = getConfig().getIntProperty    (PROPKEY_sampleJdbcTimeout   , DEFAULT_sampleJdbcTimeout);
		_printGetData         = getConfig().getBooleanProperty(PROPKEY_printGetData        , DEFAULT_printGetData);

		setSleepTimeSec( getConfig().getIntProperty(PROPKEY_sleepTimeInSeconds, DEFAULT_sleepTimeInSeconds) );

		// Test to connect
		//   - This will throw Exception on errors, and the service would NOT start, ConfigProp: SybCollector.dbms.exit.if.first.connect.fails = true
		//   - To start anyway, and hope we get a connection later on                ConfigProp: SybCollector.dbms.exit.if.first.connect.fails = false
		atInitTestConnection("sybsecurity");
		
		// Print how we are configured
		printConfig();
	}
	
	@Override
	public void printConfig()
	{
		int len = 60;

		_logger.info("Configuration for '" + getName() + "', with full class name '" + getClass().getName() + "'.");
		_logger.info("                   " + StringUtil.left(PROPKEY_sampleJdbcTimeout , len) + " = " + _sampleJdbcTimeout);
		_logger.info("                   " + StringUtil.left(PROPKEY_sleepTimeInSeconds, len) + " = " + getSleepTimeSec());
		_logger.info("                   " + StringUtil.left(PROPKEY_printGetData      , len) + " = " + _printGetData);

		printBaseConfig(len);
	}

	@Override
	protected WriterQueueContainer getData()
	throws SQLException, InterruptedException
	{
		if (_printGetData)
			_logger.info("------------------------------------ " + getName() + "::getData()");

			_logger.debug("------------------------------------ " + getName() + "::getData()");

		// Get DBMS Connection
		Connection conn = getConnection("sybsecurity");

		// Create a Container 
		HeaderInfo headerInfo = new HeaderInfo(new Timestamp(System.currentTimeMillis()), getAseServerName(), getAseHostName());
		WriterQueueContainer container = new WriterQueueContainer(headerInfo);

		// create a new "holder" object for each read
		// If we later on want to defer this to a later stage (if we want get records from monSysStatement to get 'rowcount' etc, we need to have this at an instance variable, so we can keep SQL Text for a longer time.
		List<AuditRecordSybAudit> list = new ArrayList<>(1000);
		// Alternatively this could be a LinkedHashMap with key=spid,eventtime
		// But *most* entries will only have one row (sequence==1) so the overhead of using a List might not be too bad ;)

		if (_lastEntryTs == null)
			_lastEntryTs = new Timestamp(0);

//		String sql = "exec sybsecurity.dbo.audit_collector";
//		String sql = "exec audit_collector";
		String sql = "{?=call audit_collector()}";
		
		// Execute SQL
		try
		{
			SQLWarning sqlw  = null;
//			Statement stmnt  = conn.createStatement();
			CallableStatement stmnt  = conn.prepareCall(sql);
			ResultSet  rs    = null;
			int rowsAffected = 0;

//			if (getQueryTimeout() > 0)
//				stmnt.setQueryTimeout(getQueryTimeout());
			stmnt.setQueryTimeout(_sampleJdbcTimeout);

			if (_logger.isDebugEnabled()) 
				_logger.debug("EXECUTING: " + sql);
			
			boolean hasRs = false;
			if (stmnt instanceof CallableStatement)
			{
				stmnt.registerOutParameter(1, Types.INTEGER);
				hasRs = stmnt.execute();
			}
			else
			{
				hasRs = stmnt.execute(sql);
			}

			// iterate through each result set
			do
			{
				if(hasRs)
				{
					// Get next resultset to work with
					rs = stmnt.getResultSet();

					// Read the ResultSet and add it to the list
					readResultSet(list, rs, sql);

					// Check for warnings
					// If warnings found, add them to the LIST
					for (sqlw = rs.getWarnings(); sqlw != null; sqlw = sqlw.getNextWarning())
					{
						_logger.info("SQL Warning: " + AseUtils.sqlExceptionOrWarningSingleToString(sqlw));
					}
					rs.clearWarnings();

					// Close it
					rs.close();
				}
				else
				{
					// Treat update/row count(s)
					rowsAffected = stmnt.getUpdateCount();
					if (rowsAffected >= 0)
					{
					}
				}

				// Check if we have more resultsets
				hasRs = stmnt.getMoreResults();

				_logger.trace( "--hasRs=" + hasRs + ", rowsAffected=" + rowsAffected );
			}
			while (hasRs || rowsAffected != -1);

			// Read Procedure Return Code
			if (stmnt instanceof CallableStatement)
			{
				int returnStatus = stmnt.getInt(1);
				if (returnStatus != 0)
					_logger.info("Unexpected Return Code of " + returnStatus + " from the SQL '" + sql + "'.");
			}
			
			// Check for warnings
			for (sqlw = stmnt.getWarnings(); sqlw != null; sqlw = sqlw.getNextWarning())
			{
				_logger.info("SQL Warning: " + AseUtils.sqlExceptionOrWarningSingleToString(sqlw));
			}
			stmnt.clearWarnings();

			// Close the statement
			stmnt.close();
		}
		catch(SQLWarning w)
		{
			_logger.warn("Problems when executing sql: " + sql, w);
		}
		catch (SQLException ex)
		{
			throw ex;
		}

		// Add records to the container
		// Also filter out some records that we can exclude
		for (AuditRecordSybAudit auditRecord : list)
		{
			// If we want to check for application name etc... 
//			if ( ! includeEntry(conn, auditRecord) )
//				continue;
			if ( ! includeProgramName(conn, auditRecord.getSybAuditEntry().getSpid(), -1) ) // KPID=-1  to grab "first" KPID
				continue;
			
			// Extract information from "extrainfo" and fill in some extra fields
			auditRecord.getSybAuditEntry().normalizeExtraInfo();

			// Set ASE Process/User information
			SybSysProcessesEntry sysProcEntry = SybSysProcessesCache.getInstance().getRecord(conn, auditRecord.getSybAuditEntry().getSpid(), -1);
			auditRecord.setSybSysProcessesEntry( sysProcEntry );
			
			container.add(auditRecord);
		}
		
		// reset the SPID/KPID miss/skip map
		SybSysProcessesCache.getInstance().clearSkipMap();
		
		return container;
	}

	private int readResultSet(List<AuditRecordSybAudit> list, ResultSet rs, String sql)
	throws SQLException
	{
		ResultSetMetaData rsmd = rs.getMetaData();

		int expectedColCount = 16;
		int actualColCount   = rsmd.getColumnCount(); 
		if (actualColCount != expectedColCount)
		{
			_logger.error("Unexpected Column Count of " + actualColCount + ", I was expecting " + expectedColCount + " columns in the ResultSet for SQL=|" + sql + "|.");

			// Read out the ResultSet
			while(rs.next())
				;

			// And close it
			rs.close();

			return 0;
		}
		
		AuditRecordSybAudit lastAuditRecord = null;
		int rowsRead = 0;
		while(rs.next())
		{
			rowsRead ++;

			String    event_name = rs.getString   (1);
			int       event      = rs.getInt      (2);
			int       eventmod   = rs.getInt      (3);
			int       spid       = rs.getInt      (4);
			Timestamp eventtime  = rs.getTimestamp(5);
			int       sequence   = rs.getInt      (6);
			int       suid       = rs.getInt      (7);
			int       dbid       = rs.getInt      (8);
			int       objid      = rs.getInt      (9);
			String    xactid     = StringUtil.bytesToHex("0x", rs.getBytes(10), true); // The datatype is 'binary(6)' which will be translated to 0xHexRepresentationOfTheValue
			String    loginname  = rs.getString   (11);
			String    dbname     = rs.getString   (12);
			String    objname    = rs.getString   (13);
			String    objowner   = rs.getString   (14);
			String    extrainfo  = rs.getString   (15);
			int       nodeid     = rs.getInt      (16);

			if (eventtime != null)
			{
				if (eventtime.getTime() > _lastEntryTs.getTime())
					_lastEntryTs = eventtime;
			}
			
			// Handle Strings with null values (assign a known value)
//			if (event_name == null) event_name = NULL_REPLACE;
//			if (loginname  == null) loginname  = NULL_REPLACE;
//			if (dbname     == null) dbname     = NULL_REPLACE;
//			if (objname    == null) objname    = NULL_REPLACE;
//			if (objowner   == null) objowner   = NULL_REPLACE;
////			if (extrainfo  == null) extrainfo  = NULL_REPLACE;

			// Handle Strings with null values (assign empty string)
			if (event_name == null) event_name = "";
			if (loginname  == null) loginname  = "";
			if (dbname     == null) dbname     = "";
			if (objname    == null) objname    = "";
			if (objowner   == null) objowner   = "";
			if (extrainfo  == null) extrainfo  = "";
			
			// Should the Login Name be included or nor
			if ( ! includeLoginName(loginname) )
				continue;

			if (sequence <= 1)
			{
				AuditRecordSybAudit auditRecord = new AuditRecordSybAudit();
				auditRecord.setSybAuditEntry(
						 event_name
						,event     
						,eventmod  
						,spid      
						,eventtime 
						,sequence  
						,suid      
						,dbid      
						,objid     
						,xactid    
						,loginname 
						,dbname    
						,objname   
						,objowner  
						,extrainfo 
						,nodeid    
				);

				// Add it to the list
				list.add(auditRecord);
				
				// Remember Last record, this so we can easily add/append "extrainfo" and increment sequence number
				// this will be used if sequence is above 1
				lastAuditRecord = auditRecord;
			}
			else
			{
				if (lastAuditRecord != null)
				{
					lastAuditRecord.getSybAuditEntry().addExtraInfo(extrainfo, sequence);
				}
//				// Get previous record, and append information to 'extrainfo'
//				for (AuditRecordSybAudit ar : list)
//				{
//					SybAuditEntry sae = ar.getSybAuditEntry();
//					if (sae.getSpid() == spid && sae.getEventTime().getTime() == eventtime.getTime())
//					{
//						sae.addExtraInfo(extrainfo, sequence);
//					}
//				}
			}
		}

		return rowsRead;
	}


//	@Override
//	protected WriterQueueContainer getData()
//	throws SQLException, InterruptedException
//	{
//System.out.println("------------------------------------ " + getName() + "::getData()");
//		if (_logger.isDebugEnabled())
//			_logger.debug("------------------------------------ " + getName() + "::getData()");
//
//		// Get DBMS Connection
//		Connection conn = getConnection("sybsecurity");
//
//		// Create a Container 
//		HeaderInfo headerInfo = new HeaderInfo(new Timestamp(System.currentTimeMillis()), getAseServerName(), getAseHostName());
//		WriterQueueContainer container = new WriterQueueContainer(headerInfo);
//
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
////		container.add(new AuditRecordSybAudit());
//
//		// create a new "holder" object for each read
//		// If we later on want to defer this to a later stage (if we want get records from monSysStatement to get 'rowcount' etc, we need to have this at an instance variable, so we can keep SQL Text for a longer time.
//		List<AuditRecordSybAudit> list = new ArrayList<>(1000);
//		// Alternatively this could be a LinkedHashMap with key=spid,eventtime
//		// But *most* entries will only have one row (sequence==1) so the overhead of using a List might not be too bad ;)
//
//		if (_lastEntryTs == null)
//			_lastEntryTs = new Timestamp(0);
//
//		String sql = ""
//				+ "select \n"
//				+ "     event_name = audit_event_name(event) \n"
//				+ "    ,event     \n"
//				+ "    ,eventmod  \n"
//				+ "    ,spid      \n"
//				+ "    ,eventtime \n"
//				+ "    ,sequence  \n"
//				+ "    ,suid      \n"
//				+ "    ,dbid      \n"
//				+ "    ,objid     \n"
//				+ "    ,xactid    \n"
//				+ "    ,loginname \n"
//				+ "    ,dbname    \n"
//				+ "    ,objname   \n"
//				+ "    ,objowner  \n"
//				+ "    ,extrainfo \n"
//				+ "    ,nodeid    \n"
//				+ "from sysaudits_01 \n"
//				+ "where spid != @@spid \n"
//				+ "  and eventtime > '" + _lastEntryTs + "' \n"
//				+ "";
//		
//TODO; // Call procedure 'audit_collector' instead
//TODO; // Also change to cstmnt.execute(); loop ResultSets if we want to generate MANY ResultSets before we truncate the sysaudit_01 table (if we arn't using several audit tables)
//		
//		
//		// Get SQL TEXT
//		try (Statement stmnt = conn.createStatement())
//		{
//			stmnt.setQueryTimeout(60);
//
//			try (ResultSet rs = stmnt.executeQuery(sql))
//			{
//				while(rs.next())
//				{
//					String    event_name = rs.getString   (1);
//					int       event      = rs.getInt      (2);
//					int       eventmod   = rs.getInt      (3);
//					int       spid       = rs.getInt      (4);
//					Timestamp eventtime  = rs.getTimestamp(5);
//					int       sequence   = rs.getInt      (6);
//					int       suid       = rs.getInt      (7);
//					int       dbid       = rs.getInt      (8);
//					int       objid      = rs.getInt      (9);
//					String    xactid     = StringUtil.bytesToHex("0x", rs.getBytes(10), true); // The datatype is 'binary(6)' which will be translated to 0xHexRepresentationOfTheValue
//					String    loginname  = rs.getString   (11);
//					String    dbname     = rs.getString   (12);
//					String    objname    = rs.getString   (13);
//					String    objowner   = rs.getString   (14);
//					String    extrainfo  = rs.getString   (15);
//					int       nodeid     = rs.getInt      (16);
//
//					if (eventtime != null)
//					{
//						if (eventtime.getTime() > _lastEntryTs.getTime())
//							_lastEntryTs = eventtime;
//					}
//
//					// Should the Login Name be included or nor
//					if ( ! includeLoginName(loginname) )
//						continue;
//
//					if (sequence <= 1)
//					{
//						AuditRecordSybAudit auditRecord = new AuditRecordSybAudit();
//						auditRecord.setSybAuditEntry(
//								 event_name
//								,event     
//								,eventmod  
//								,spid      
//								,eventtime 
//								,sequence  
//								,suid      
//								,dbid      
//								,objid     
//								,xactid    
//								,loginname 
//								,dbname    
//								,objname   
//								,objowner  
//								,extrainfo 
//								,nodeid    
//						);
//
//						// Add it to the list
//						list.add(auditRecord);
//					}
//					else
//					{
//						// Get previous record, and append information to 'extrainfo'
//						for (AuditRecordSybAudit ar : list)
//						{
//							SybAuditEntry sae = ar.getSybAuditEntry();
//							if (sae.getSpid() == spid && sae.getEventTime().getTime() == eventtime.getTime())
//							{
//								sae.addExtraInfo(extrainfo, sequence);
//							}
//						}
//					}
//				}
//			}
//		}
//
//		for (AuditRecordSybAudit auditRecord : list)
//		{
//			// If we want to check for application name etc... 
////			if ( ! includeEntry(conn, auditRecord) )
////				continue;
//			if ( ! includeProgramName(conn, auditRecord.getSybAuditEntry().getSpid(), -1) ) // KPID=-1  to grab "first" KPID
//				continue;
//			
//			// Extract information from "extrainfo" and fill in some extra fields
//			auditRecord.getSybAuditEntry().normalizeExtraInfo();
//
//			// Set ASE Process/User information
//			auditRecord.setSybSysProcessesEntry( SybSysProcessesCache.getInstance().getRecord(conn, auditRecord.getSybAuditEntry().getSpid(), -1) );
//			
//			container.add(auditRecord);
//		}
//		
//		return container;
//	}

//	@Override
//	public boolean includeLoginName(String loginname)
//	{
//		if (_skipLoginNamePattern != null)
//		{
//			return _skipLoginNamePattern.matcher(loginName).matches();
//		}
//
//		return super.includeLoginName(loginName);
//	}

//	@Override
//	public boolean includeProgramName(Connection conn, int spid, int kpid)
//	{
//		return super.includeProgramName(conn, spid, kpid);
//	}

//	public boolean includeEntry(Connection conn, AuditRecordSybAudit auditRecord)
//	{
//		// In here we could lookup stuff from SybSysProccessCache... like ApplicationName / HostName / IpAddress etc
//		try
//		{
//			SybSysProcessesEntry sysProcEntry = SybSysProcessesCache.getInstance().getRecord(conn, auditRecord.getSybAuditEntry().getSpid(), -1);
//
//			if (sysProcEntry != null)
//			{
//				String programName = sysProcEntry.getProgramName();
//				if (programName != null)
//				{
//					if (Version.getAppName().equals(programName) )
//						return false;
//
//					if (programName.startsWith("AseTune"))
//						return false;
//				}
//			}
//			
//			return true;
//		}
//		catch (SQLException ex)
//		{
//			_logger.error("includeEntry(auditRecord=" + auditRecord + "): Problems looking up information from SybSysProcessesCache. Action=The entry will be INCLUDED.  Error=" + ex.getErrorCode() + ", Msg=|" + ex.getMessage() + "|.");
//			return true;
//		}
//	}

}
