/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.collectors;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import sek.ase.auditor.collectors.records.AuditRecordSybStmnt;
import sek.ase.auditor.collectors.records.AuditRecordSybStmnt.MonSqlStatementEntry;
import sek.ase.auditor.utils.AseUtils;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.utils.TimeUtils;
import sek.ase.auditor.wqs.WriterQueueContainer;
import sek.ase.auditor.wqs.WriterQueueContainer.HeaderInfo;

/**
 * Collects SQL Text and other information from the Sybase (MDA) Monitoring tables
 * 
 * @author goran
 */
public class SybStmntCollector
extends SybAbstractCollector
{
	private static Logger _logger = LogManager.getLogger();

//	public static final int     DEFAULT_sqlCap_ase_sqlTextAndPlan_batchIdEntry_keepCount = 2;
//	public static final int     DEFAULT_sqlCap_ase_sqlTextAndPlan_batchIdEntry_keepCount = 10; // 10 should be WAY to much, it's just for testing...
	//NOTE: if we have a keep count higher than 1 or 2, then we should do LOW ON MEMORY to first remove all with a keep-count > 1 or similar...

	public static final String  PROPKEY_skipSqlTextRegex          = "SybStmntCollector.skip.sqlText.regex";
	public static final String  DEFAULT_skipSqlTextRegex          = null;
	
	public static final String  PROPKEY_keepSqlTextRegex          = "SybStmntCollector.keep.sqlText.regex";
	public static final String  DEFAULT_keepSqlTextRegex          = null;
	
	public static final String  PROPKEY_sleepTimeInSeconds        = "SybStmntCollector.sleepTime.seconds";
	public static final int     DEFAULT_sleepTimeInSeconds        = 1;

	public static final String  PROPKEY_clearMonTablesOnConnect   = "SybStmntCollector.clearMonTablesOnConnect";
	public static final boolean DEFAULT_clearMonTablesOnConnect   = false;
	
	public static final String  PROPKEY_printGetData              = "SybStmntCollector.print.getData";
	public static final boolean DEFAULT_printGetData              = false;
	
	public static final String  PROPKEY_sampleJdbcTimeout         = "SybStmntCollector.sample.jdbc.timout"; 
	public static final int     DEFAULT_sampleJdbcTimeout         = 30;

	public static final String  PROPKEY_sampleStatementDetatils   = "SybStmntCollector.sample.statement.details"; 
	public static final boolean DEFAULT_sampleStatementDetatils   = true;

	public static final String  PROPKEY_sampleStatementProcName   = "SybStmntCollector.sample.statement.procName"; 
	public static final boolean DEFAULT_sampleStatementProcName   = false;

	public static final String  PROPKEY_statistics_printSpidSqlTextManagerInfo = "SybStmntCollector.statistics.printSpidSqlTextManagerInfo";
	public static final boolean DEFAULT_statistics_printSpidSqlTextManagerInfo = false;
	
	private int     _sampleJdbcTimeout       = DEFAULT_sampleJdbcTimeout;
	private boolean _sampleStatementDetails  = DEFAULT_sampleStatementDetatils;
	private boolean _clearMonTablesOnConnect = DEFAULT_clearMonTablesOnConnect;
	private boolean _printGetData            = DEFAULT_printGetData;
	
	// static so we can reach it from any private static sub classes
//	private static int _default_batchIdEntry_keepCount = DEFAULT_sqlCap_ase_sqlTextAndPlan_batchIdEntry_keepCount;

	private String _skipSqlTextRegex = DEFAULT_skipSqlTextRegex; 
	private String _keepSqlTextRegex = DEFAULT_keepSqlTextRegex; 

	private Pattern _skipSqlTextPattern = _skipSqlTextRegex == null ? null : Pattern.compile(_skipSqlTextRegex); 
	private Pattern _keepSqlTextPattern = _keepSqlTextRegex == null ? null : Pattern.compile(_keepSqlTextRegex);; 

	// Statistical members
	private long _totalSqlTextSkipCount = 0;  // Note: Should this be AtimicLong ... but this is single thread so...
	private long  _lastSqlTextSkipCount = 0;

	
	public static final String  CFGNAME_aseConfig_enable_monitoring            = "enable monitoring";
	public static final String  PROPKEY_aseConfig_enable_monitoring            = "SybStmntCollector.ase.config.enable_monitoring";
	public static final int     DEFAULT_aseConfig_enable_monitoring            = 1;

	public static final String  CFGNAME_aseConfig_sql_text_pipe_active         = "sql text pipe active";
	public static final String  PROPKEY_aseConfig_sql_text_pipe_active         = "SybStmntCollector.ase.config.sql_text_pipe_active";
	public static final int     DEFAULT_aseConfig_sql_text_pipe_active         = 1;

	public static final String  CFGNAME_aseConfig_sql_text_pipe_max_messages   = "sql text pipe max messages";
	public static final String  PROPKEY_aseConfig_sql_text_pipe_max_messages   = "SybStmntCollector.ase.config.sql_text_pipe_max_messages";
	public static final int     DEFAULT_aseConfig_sql_text_pipe_max_messages   = 1_000;


	public static final String  CFGNAME_aseConfig_statement_pipe_active        = "statement pipe active";
	public static final String  PROPKEY_aseConfig_statement_pipe_active        = "SybStmntCollector.ase.config.statement_pipe_active";
	public static final int     DEFAULT_aseConfig_statement_pipe_active        = 1;

	public static final String  CFGNAME_aseConfig_statement_pipe_max_messages  = "statement pipe max messages";
	public static final String  PROPKEY_aseConfig_statement_pipe_max_messages  = "SybStmntCollector.ase.config.statement_pipe_max_messages";
	public static final int     DEFAULT_aseConfig_statement_pipe_max_messages  = 10_000;


	public static final String  CFGNAME_aseConfig_statement_statistics_active  = "statement statistics active";
	public static final String  PROPKEY_aseConfig_statement_statistics_active  = "SybStmntCollector.ase.config.statement_statistics_active";
	public static final int     DEFAULT_aseConfig_statement_statistics_active  = 1;

	public static final String  CFGNAME_aseConfig_per_object_statistics_active = "per object statistics active";
	public static final String  PROPKEY_aseConfig_per_object_statistics_active = "SybStmntCollector.ase.config.per_object_statistics_active";
	public static final int     DEFAULT_aseConfig_per_object_statistics_active = 1;

	
	private SpidSqlTextManager _instanceSpidSqlTextManager = new SpidSqlTextManager();
	
	
	//-------------------------------------------------------------------------------
	// Look at how/if we can reuse stuff from DbxTune - SqlCaptureBrokerAse
	// https://github.com/goranschwarz/DbxTune/blob/master/src/com/asetune/pcs/sqlcapture/SqlCaptureBrokerAse.java
	//-------------------------------------------------------------------------------

	private static class MdaCheckMarker
	{
		String _sqlMarker;
		int    _spid;
		int    _kpid;
		int    _batchId;

		@Override
		public String toString()
		{
			return "MdaCheckMarker [spid=" + _spid + ", kpid=" + _kpid + ", batchId=" + _batchId + ", sqlMarker='" + _sqlMarker + "']";
		}
	}
	private MdaCheckMarker _mdaCheckMarker;

	
	@Override
	public void init(Configuration config)
	throws Exception
	{
		super.init(config);

		// Initialize some values based on configuration
		_sampleJdbcTimeout       = getConfig().getIntProperty    (PROPKEY_sampleJdbcTimeout      , DEFAULT_sampleJdbcTimeout);
		_sampleStatementDetails  = getConfig().getBooleanProperty(PROPKEY_sampleStatementDetatils, DEFAULT_sampleStatementDetatils);

		_skipSqlTextRegex        = config.getProperty(PROPKEY_skipSqlTextRegex, DEFAULT_skipSqlTextRegex); 
		_keepSqlTextRegex        = config.getProperty(PROPKEY_keepSqlTextRegex, DEFAULT_keepSqlTextRegex);

		_clearMonTablesOnConnect = getConfig().getBooleanProperty(PROPKEY_clearMonTablesOnConnect, DEFAULT_clearMonTablesOnConnect);
		_printGetData            = getConfig().getBooleanProperty(PROPKEY_printGetData           , DEFAULT_printGetData);

		setSleepTimeSec( getConfig().getIntProperty(PROPKEY_sleepTimeInSeconds, DEFAULT_sleepTimeInSeconds) );


		// Initialize some variables that depends on the configuration
		_skipSqlTextPattern      = _skipSqlTextRegex == null ? null : Pattern.compile(_skipSqlTextRegex); 
		_keepSqlTextPattern      = _keepSqlTextRegex == null ? null : Pattern.compile(_keepSqlTextRegex);; 
		
		// Test to connect
		//   - This will throw Exception on errors, and the service would NOT start, ConfigProp: SybCollector.dbms.exit.if.first.connect.fails = true
		//   - To start anyway, and hope we get a connection later on                ConfigProp: SybCollector.dbms.exit.if.first.connect.fails = false
		atInitTestConnection("master");

		// Print how we are configured
		printConfig();
	}

	@Override
	public void printConfig()
	{
		int len = 60;

		_logger.info("Configuration for '" + getName() + "', with full class name '" + getClass().getName() + "'.");
		_logger.info("                   " + StringUtil.left(PROPKEY_sampleStatementDetatils, len) + " = " + _sampleStatementDetails);
		_logger.info("                   " + StringUtil.left(PROPKEY_sampleJdbcTimeout      , len) + " = " + _sampleJdbcTimeout);
		_logger.info("                   " + StringUtil.left(PROPKEY_skipSqlTextRegex       , len) + " = " + _skipSqlTextRegex);
		_logger.info("                   " + StringUtil.left(PROPKEY_keepSqlTextRegex       , len) + " = " + _keepSqlTextRegex);
		_logger.info("                   " + StringUtil.left(PROPKEY_sleepTimeInSeconds     , len) + " = " + getSleepTimeSec());
		_logger.info("                   " + StringUtil.left(PROPKEY_clearMonTablesOnConnect, len) + " = " + _clearMonTablesOnConnect);
		_logger.info("                   " + StringUtil.left(PROPKEY_printGetData           , len) + " = " + _printGetData);
		
		printBaseConfig(len);
	}

	@Override
	public void onConnect(Connection conn)
	throws SQLException
	{
		checkAseMonitoringConfig(conn);
	}

	private void checkAseMonitoringConfig(Connection conn)
	throws SQLException
	{
		int cfg_enableMonitoring          = getConfig().getIntProperty(PROPKEY_aseConfig_enable_monitoring           , DEFAULT_aseConfig_enable_monitoring);
		int cfg_sqlTextPipeActive         = getConfig().getIntProperty(PROPKEY_aseConfig_sql_text_pipe_active        , DEFAULT_aseConfig_sql_text_pipe_active);
		int cfg_sqlTextPipeMaxMessages    = getConfig().getIntProperty(PROPKEY_aseConfig_sql_text_pipe_max_messages  , DEFAULT_aseConfig_sql_text_pipe_max_messages);
		int cfg_statementPipeActive       = getConfig().getIntProperty(PROPKEY_aseConfig_statement_pipe_active       , DEFAULT_aseConfig_statement_pipe_active);
		int cfg_statementPipeMaxMessages  = getConfig().getIntProperty(PROPKEY_aseConfig_statement_pipe_max_messages , DEFAULT_aseConfig_statement_pipe_max_messages);
		int cfg_statementStatisticsActive = getConfig().getIntProperty(PROPKEY_aseConfig_statement_statistics_active , DEFAULT_aseConfig_statement_statistics_active);
		int cfg_perObjectStatisticsActive = getConfig().getIntProperty(PROPKEY_aseConfig_per_object_statistics_active, DEFAULT_aseConfig_per_object_statistics_active);

		int ase_enableMonitoring          = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_enable_monitoring);

		int ase_sqlTextPipeActive         = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_sql_text_pipe_active);
		int ase_sqlTextPipeMaxMessages    = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_sql_text_pipe_max_messages);

		int ase_statementPipeActive       = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_statement_pipe_active);
		int ase_statementPipeMaxMessages  = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_statement_pipe_max_messages);

		int ase_statementStatisticsActive = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_statement_statistics_active);
		int ase_perObjectStatisticsActive = AseUtils.getAseConfigRunValue(conn, CFGNAME_aseConfig_per_object_statistics_active);
		
		List<String> doReconfigure = new ArrayList<>();
		if (ase_enableMonitoring          < cfg_enableMonitoring)          { doReconfigure.add(CFGNAME_aseConfig_enable_monitoring           ); _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_enable_monitoring            + "' is lower (currently="+ ase_enableMonitoring          +") than the suggested value '" + cfg_enableMonitoring          + "'. Please Execeute: sp_configure '" + CFGNAME_aseConfig_sql_text_pipe_active         + "', " + cfg_enableMonitoring         ); }
		if (ase_sqlTextPipeActive         < cfg_sqlTextPipeActive)         { doReconfigure.add(CFGNAME_aseConfig_sql_text_pipe_active        ); _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_sql_text_pipe_active         + "' is lower (currently="+ ase_sqlTextPipeActive         +") than the suggested value '" + cfg_sqlTextPipeActive         + "'. Please Execeute: sp_configure '" + CFGNAME_aseConfig_sql_text_pipe_active         + "', " + cfg_sqlTextPipeActive        ); }
		if (ase_sqlTextPipeMaxMessages    < cfg_sqlTextPipeMaxMessages)    { doReconfigure.add(CFGNAME_aseConfig_sql_text_pipe_max_messages  ); _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_sql_text_pipe_max_messages   + "' is lower (currently="+ ase_sqlTextPipeMaxMessages    +") than the suggested value '" + cfg_sqlTextPipeMaxMessages    + "'. Please Execeute: sp_configure '" + CFGNAME_aseConfig_sql_text_pipe_max_messages   + "', " + cfg_sqlTextPipeMaxMessages   ); }
		if (ase_statementPipeActive       < cfg_statementPipeActive)       { doReconfigure.add(CFGNAME_aseConfig_statement_pipe_active       ); _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_statement_pipe_active        + "' is lower (currently="+ ase_statementPipeActive       +") than the suggested value '" + cfg_statementPipeActive       + "'. Please Execeute: sp_configure '" + CFGNAME_aseConfig_statement_pipe_active        + "', " + cfg_statementPipeActive      ); }
		if (ase_statementPipeMaxMessages  < cfg_statementPipeMaxMessages)  { doReconfigure.add(CFGNAME_aseConfig_statement_pipe_max_messages ); _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_statement_pipe_max_messages  + "' is lower (currently="+ ase_statementPipeMaxMessages  +") than the suggested value '" + cfg_statementPipeMaxMessages  + "'. Please Execeute: sp_configure '" + CFGNAME_aseConfig_statement_pipe_max_messages  + "', " + cfg_statementPipeMaxMessages ); }

		// The below 2 is NOT Mandatory (I think we will still get the desired SQL Text and some Statement Statistics)
		if (ase_statementStatisticsActive < cfg_statementStatisticsActive) { _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_statement_statistics_active  + "' is lower (currently="+ ase_statementStatisticsActive +") than the NOT mandatory and suggested value '" + cfg_statementStatisticsActive + "'. Consider Execeute: sp_configure '" + CFGNAME_aseConfig_statement_statistics_active  + "', " + cfg_statementStatisticsActive); }
		if (ase_perObjectStatisticsActive < cfg_perObjectStatisticsActive) { _logger.warn("The ASE Configuration '" + CFGNAME_aseConfig_per_object_statistics_active + "' is lower (currently="+ ase_perObjectStatisticsActive +") than the NOT mandatory and suggested value '" + cfg_perObjectStatisticsActive + "'. Consider Execeute: sp_configure '" + CFGNAME_aseConfig_per_object_statistics_active + "', " + cfg_perObjectStatisticsActive); }

		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_enable_monitoring           , 30, "'") + " RunValue is set to: " + StringUtil.left(ase_enableMonitoring         +"", 7) + " Smallest Recommended value: " + cfg_enableMonitoring         );
		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_sql_text_pipe_active        , 30, "'") + " RunValue is set to: " + StringUtil.left(ase_sqlTextPipeActive        +"", 7) + " Smallest Recommended value: " + cfg_sqlTextPipeActive        );
		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_sql_text_pipe_max_messages  , 30, "'") + " RunValue is set to: " + StringUtil.left(ase_sqlTextPipeMaxMessages   +"", 7) + " Smallest Recommended value: " + cfg_sqlTextPipeMaxMessages   );
		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_statement_pipe_active       , 30, "'") + " RunValue is set to: " + StringUtil.left(ase_statementPipeActive      +"", 7) + " Smallest Recommended value: " + cfg_statementPipeActive      );
		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_statement_pipe_max_messages , 30, "'") + " RunValue is set to: " + StringUtil.left(ase_statementPipeMaxMessages +"", 7) + " Smallest Recommended value: " + cfg_statementPipeMaxMessages );
		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_statement_statistics_active , 30, "'") + " RunValue is set to: " + StringUtil.left(ase_statementStatisticsActive+"", 7) + " Non-Mandatory Recommended value: " + cfg_statementStatisticsActive);
		_logger.info("ASE Configuration " + StringUtil.left(CFGNAME_aseConfig_per_object_statistics_active, 30, "'") + " RunValue is set to: " + StringUtil.left(ase_perObjectStatisticsActive+"", 7) + " Non-Mandatory Recommended value: " + cfg_perObjectStatisticsActive);

		if ( ! doReconfigure.isEmpty() )
			throw new SQLException("ASE Configuration " + StringUtil.toCommaStrQuoted("'", doReconfigure) + " needs to be increased.");
	}

	@Override
	protected WriterQueueContainer getData()
	throws SQLException, InterruptedException
	{
		if (_printGetData)
			_logger.info("------------------------------------ " + getName() + "::getData()");

		if (_logger.isDebugEnabled())
			_logger.debug("------------------------------------ " + getName() + "::getData()");

		// Get DBMS Connection
		Connection conn = getConnection("master");

		// Create a Container 
		HeaderInfo headerInfo = new HeaderInfo(new Timestamp(System.currentTimeMillis()), getAseServerName(), getAseHostName());
		WriterQueueContainer container = new WriterQueueContainer(headerInfo);

//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());
//		container.add(new AuditRecordSybMda());

		// If we want to get rid of current records before we start to "poll"
		// if "monTables" are not cleared, we will/might get duplicate entries after any "reconnect" 
		if (_clearMonTablesOnConnect && isNewConnection())
		{
			clearTable(conn, "monSysSQLText");
			clearTable(conn, "monSysStatement");
		}

		// create a new "holder" object for each read
		// If we later on want to defer this to a later stage (if we want get records from monSysStatement to get 'rowcount' etc, we need to have this at an instance variable, so we can keep SQL Text for a longer time.
		SpidSqlTextManager localSpidSqlTextManager = null;
		if ( ! _sampleStatementDetails )
			localSpidSqlTextManager = new SpidSqlTextManager();

//		RS> Col# Label           JDBC Type Name         Guessed DBMS type Source Table            
//		RS> ---- --------------- ---------------------- ----------------- ------------------------
//		RS> 1    SPID            java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//		RS> 2    KPID            java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//		RS> 3    ServerLogin     java.sql.Types.VARCHAR varchar(30)       -none-                  
//		RS> 4    BatchID         java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//		RS> 5    SequenceInBatch java.sql.Types.INTEGER int               master.dbo.monSysSQLText
//		RS> 6    SQLText         java.sql.Types.VARCHAR varchar(255)      master.dbo.monSysSQLText

		String sql = ""
				+ "select \n"
				+ "     SPID \n"
				+ "    ,KPID \n"
				+ "    ,ServerLogin = suser_name(ServerUserID) \n"
				+ "    ,BatchID \n"
				+ "    ,SequenceInBatch \n"
				+ "    ,SQLText \n"
				+ "from master.dbo.monSysSQLText \n"
				+ "";

		boolean sqlMarkerWasFound = false;

		// Get SQL TEXT
		int sqlTextRowCount = 0;
		try (Statement stmnt = conn.createStatement())
		{
			stmnt.setQueryTimeout(_sampleJdbcTimeout);

			try (ResultSet rs = stmnt.executeQuery(sql))
			{
				while(rs.next())
				{
					sqlTextRowCount++;
					int    SPID            = rs.getInt   (1);
					int    KPID            = rs.getInt   (2);
					String ServerLogin     = rs.getString(3);
					int    BatchID         = rs.getInt   (4);
					int    SequenceInBatch = rs.getInt   (5);
					String SQLText         = rs.getString(6);

					// Handle Strings with null values (assign empty string)
					if (ServerLogin == null) ServerLogin = "";
					if (SQLText     == null) SQLText     = "";

					// Check if the previously checked "SQL Marker" is found in the queue.
					if (_mdaCheckMarker != null)
					{
						if (_mdaCheckMarker._sqlMarker != null && _mdaCheckMarker._sqlMarker.equals(SQLText))
						{
							sqlMarkerWasFound = true;
							// Remember what "BatchID" the SQL Marker was sent as... So we can check/detect loss in monSysStatement
							_mdaCheckMarker._spid    = SPID;
							_mdaCheckMarker._kpid    = KPID;
							_mdaCheckMarker._batchId = BatchID;
						}
					}

					// Should the Login Name be included or nor
					if ( ! includeLoginName(ServerLogin) )
						continue;

					if ( ! _sampleStatementDetails )
					{
						// Since the entries may come unsorted (especially in a *busy* multiple engine environment 
						// Add the entries to a "self maintained" manager
						//
						// This could also be usable if we later on want to add information from monSysStatement table (which is populated *after* the Statement is executed)
						// see: https://github.com/goranschwarz/DbxTune/blob/master/src/com/asetune/pcs/sqlcapture/SqlCaptureBrokerAse.java
						//      method: public int doSqlCapture(DbxConnection conn, PersistentCounterHandler pch)
						// for a more detailed explanation
						localSpidSqlTextManager.addSqlText(SPID, KPID, BatchID, SequenceInBatch, SQLText, ServerLogin);
					}
					else
					{
						_instanceSpidSqlTextManager.addSqlText(SPID, KPID, BatchID, SequenceInBatch, SQLText, ServerLogin);
					}
				}
			}
		}
		// If we want to check for "overflow" we might do it here...
		// Or we may post a SQL "/*check:monSysSQLText: ${UUID}*/" and if we can't find that Statement, then we have an "MDA Overflow" where the memory queue has been full... 
		if (_logger.isDebugEnabled())
			_logger.debug("SQL Marker Info[monSysSQLText]: sqlMarkerWasFound=" + sqlMarkerWasFound + ", _mdaCheckMarker=" + _mdaCheckMarker);

		if (_mdaCheckMarker != null && !sqlMarkerWasFound)
		{
			_logger.warn("When checking for 'SQL Marker[monSysSQLText]' previously sent... it wasn't found in 'monSysSQLText'. Searched for SQL Marker Text '" + _mdaCheckMarker._sqlMarker + "'. Consider to increase ASE Configuration '" + CFGNAME_aseConfig_sql_text_pipe_max_messages + "'.");
			_mdaCheckMarker = null; // No need to check 
		}

		// If _sampleStatementDetails is NOT enabled, then SEND SQL Text NOW
		// NOTE: (if _sampleStatementDetails is ENABLED) We might have to do the SQL Text filtering here (early) and then add record to the global/instance '_instanceSpidSqlTextManager'
		if ( ! _sampleStatementDetails )
		{
			// Send a "SQL Marker", which we will check for on next iteration... This to detect if the MDA Queue Sizes ('sql text pipe max messages' and 'statement pipe max messages')
			sendSqlMarker(conn);

			// Get values from spidSqlTextManager to the container
			for (SpidEntry spidEntry : localSpidSqlTextManager._spidMap.values())
			{
				for (BatchIdEntry batchEntry : spidEntry._batchIdMap.values())
				{
					// If we want to check for application name etc... 
					if ( ! includeProgramName(conn, batchEntry._spid, batchEntry._kpid) )
						continue;

					String sqlText = batchEntry.getSqlText();

					// If we want to check for SQL Text (keep/skip) 
					if ( ! includeSqlText(conn, sqlText) )
						continue;

					// Create a AuditRecord
					AuditRecordSybStmnt auditRecord = new AuditRecordSybStmnt();
					
					// Set SQL TEXT
					auditRecord.setSqlTextEntry(
							batchEntry._spid,
							batchEntry._kpid,
							batchEntry._batchId,
							batchEntry.getServerLogin(),
							//batchEntry.getSqlText()
							sqlText, // // Use slqText instead of 'batchEntry.getSqlText()' so we do not have to do "toString()" several times
							null // no MdaStatementRecords
							);

					// Set ASE Process/User information
					auditRecord.setSybSysProcessesEntry( SybSysProcessesCache.getInstance().getRecord(conn, batchEntry._spid, batchEntry._kpid) );

					// Add it to container
					container.add(auditRecord);
				}
			}

			// reset the SPID/KPID miss/skip map
			SybSysProcessesCache.getInstance().clearSkipMap();
			
			return container;  // <<<<<<<<<<<------------- RETURN if _sampleStatementDetails is NOT enabled
		}
		
		//-------------------------------------------------------------------
		// Below is only done if _sampleStatementDetails is enabled
		//-------------------------------------------------------------------
//		sql = ""
//				+ "select \n"
//				+ "     SPID \n"
//				+ "    ,KPID \n"
//				+ "    ,BatchID \n"
//				+ "    ,DBID \n"
//				+ "    ,DBName \n"    // added in 15.0.2 ESD#3
//				+ "    ,ProcedureID \n"
//				+ "    ,PlanID \n"
//				+ "    ,ContextID \n"
//				+ "    ,LineNumber \n"
////				+ "    ,ObjOwnerID = CASE WHEN SsqlId > 0 THEN 0 ELSE object_owner_id(ProcedureID, DBID) END \n"   // added in 15.0.3
//				+ "    ,ObjOwnerID = CASE WHEN SsqlId > 0 THEN 0 ELSE 999 END \n"   // added in 15.0.3
//				+ "    ,HashKey \n"   // added in 15.0.2
//				+ "    ,SsqlId \n"    // added in 15.0.2
//				+ "    ,ProcName = CASE WHEN SsqlId > 0 THEN isnull(object_name(SsqlId,2), '*##'+right('0000000000'+convert(varchar(20),SsqlId),10)+'_'+right('0000000000'+convert(varchar(20),HashKey),10)+'##*') \n"
//				+ "                    ELSE coalesce(object_name(ProcedureID,DBID), object_name(ProcedureID,2), object_name(ProcedureID,db_id('sybsystemprocs'))) \n"
//				+ "               END \n"
//TODO; // fix 'object_owner_id(ProcedureID, DBID)' and 'object_name(ProcedureID,DBID)' they causes SUID not allowed in database XXX
//				+ "    ,CpuTime \n"
//				+ "    ,WaitTime \n"
//				+ "    ,MemUsageKB \n"
//				+ "    ,PhysicalReads \n"
//				+ "    ,LogicalReads \n"
//				+ "    ,RowsAffected \n"   // added in 15.0.0 ESD#2
//				+ "    ,ErrorStatus \n"    // added in 15.0.0 ESD#2
//				+ "    ,ProcNestLevel \n" // added in 15.0.3
//				+ "    ,StatementNumber \n" // added in 15.0.3
//				+ "    ,QueryOptimizationTime \n" // added in 16.0 SP3 <<<<<< NOTE: This might be to high for SEK
//				+ "    ,PagesModified \n"
//				+ "    ,PacketsSent \n"
//				+ "    ,PacketsReceived \n"
//				+ "    ,NetworkPacketSize \n"
//				+ "    ,PlansAltered \n"
//				+ "    ,StartTime \n"
//				+ "    ,EndTime \n"
//				+ "    ,Elapsed_ms = CASE WHEN datediff(day, StartTime, EndTime) >= 24 THEN -1 ELSE  datediff(ms, StartTime, EndTime) END \n"
//				+ "    ,sampleTime = getdate() \n"
//
//				+ "from master.dbo.monSysStatement \n"
//				+ "";

		// GET SQL Statement to execute: 'select ... from master.dbo.monSysStatement'
		sql = MonSqlStatementEntry.getSql();

		boolean statementMarkerWasFound = false;

		int statementRowCount = 0;
		try (Statement stmnt = conn.createStatement())
		{
			stmnt.setQueryTimeout(_sampleJdbcTimeout);

			try (ResultSet rs = stmnt.executeQuery(sql))
			{
				while(rs.next())
				{
					statementRowCount++;

					// Get an Object for all columns
					MonSqlStatementEntry row = new MonSqlStatementEntry(rs);
					
					// Check if the previously checked "SQL Marker" is found in the queue.
					if (_mdaCheckMarker != null)
					{
						if (_mdaCheckMarker._sqlMarker != null 
								&& _mdaCheckMarker._spid    == row.SPID 
								&& _mdaCheckMarker._kpid    == row.KPID 
								&& _mdaCheckMarker._batchId == row.BatchID
							)
						{
							statementMarkerWasFound = true;
						}
					}

					// ADD the entry/row to the SPID/KPID/BatchID
					_instanceSpidSqlTextManager.addStatementRecord(row.SPID, row.KPID, row.BatchID, row);
				}
			}
		}
		// If we want to check for "overflow" we might do it here...
		// Or we may post a SQL "/*check:monSysSQLText: ${UUID}*/" and if we can't find that Statement, then we have an "MDA Overflow" where the memory queue has been full... 
		if (_logger.isDebugEnabled())
			_logger.debug("SQL Marker Info[monSysStatement]: statementMarkerWasFound=" + statementMarkerWasFound + ", _mdaCheckMarker=" + _mdaCheckMarker);

		if (_mdaCheckMarker != null && !statementMarkerWasFound)
		{
			_logger.warn("When checking for 'SQL Marker[monSysStatement]' previously sent... it wasn't found in 'monSysStatement'. Searched for SQL Marker [SPID, KPID, BatchID] '" + _mdaCheckMarker + "'. Consider to increase ASE Configuration '" + CFGNAME_aseConfig_statement_pipe_max_messages + "'.");
		//	_mdaCheckMarker = null; // No need to check   ???? 
		}

		
		// Send a "SQL Marker", which we will check for on next iteration... This to detect if the MDA Queue Sizes ('sql text pipe max messages' and 'statement pipe max messages')
		sendSqlMarker(conn);

		//-----------------------------------------------------
		// Post processing of data
		//   - Send: SQL and Statement info
		//-----------------------------------------------------
		// Get values from spidSqlTextManager to the container
		for (SpidEntry spidEntry : _instanceSpidSqlTextManager._spidMap.values())
		{
			for (BatchIdEntry batchEntry : spidEntry._batchIdMap.values())
			{
				// If we want to check for application name etc... 
				if ( ! includeProgramName(conn, batchEntry._spid, batchEntry._kpid) )
				{
					batchEntry.doMarkForRemoval();
					continue;
				}

				String sqlText = batchEntry.getSqlText();

				// If we want to check for SQL Text (keep/skip) 
				if ( ! includeSqlText(conn, sqlText) )
				{
					batchEntry.doMarkForRemoval();
					continue;
				}
				
				boolean doSend = false;
				
				// If the current BatchId is LOWER than the last BatchId added to the manager, then client has sent NEW SQL Text
				// So we can consider that as "complete" and send it! 
				if (batchEntry._batchId < spidEntry._maxBatchId)
					doSend = true;

				// If the Statement is a "one liner" (no procedure or multiple SQL statements in batch) we can also send it now
				// otherwise we have to wait and send until next SQLText/BatchID is received (because we don't know if it's a LONG running query/statement)
				if ( ! doSend && batchEntry.isOkToSend() )
					doSend = true;

				// If the SPID is NOT anymore logged in to ASE... Then send it
				if ( ! SybSysProcessesCache.getInstance().exists(conn, batchEntry._spid, batchEntry._kpid) )
					doSend = true;
				
				// NOTE: The below is NOT IMPLEMENTED... (I just kept the idea if we want to filter out Zero RowsAffected)
				// One final filter... 'RowsAffected' and 'ErrorStatus'
				//  * check if 'RowsAffected' is above 0 (for all statements)
				//  * but if 'ErrorStatus' != 0 then we will always send (we want to see errors)
				//if (doSend)
				//{
				//	if (batchEntry.getSumRowsAffected() <= 0) // it will be negative if there are no StatementEntries
				//		doSend = false;
                //
				//	if (batchEntry.getErrorCount() > 0)
				//		doSend = true;
				//}

				// Should we SEND it...
				if (doSend)
				{
					// Create a AuditRecord
					AuditRecordSybStmnt auditRecord = new AuditRecordSybStmnt();

					// Set SQL TEXT
					auditRecord.setSqlTextEntry(
							batchEntry._spid,
							batchEntry._kpid,
							batchEntry._batchId,
							batchEntry.getServerLogin(),
							sqlText, // // Use slqText instead of 'batchEntry.getSqlText()' so we do not have to do "toString()" several times
							batchEntry.getStatementRecords()
							);

					// Set ASE Process/User information
					auditRecord.setSybSysProcessesEntry( SybSysProcessesCache.getInstance().getRecord(conn, batchEntry._spid, batchEntry._kpid) );

					// Add it to container
					container.add(auditRecord);

					// Mark this entry for removal
					batchEntry.doMarkForRemoval();
				}
			}
		}
		
		// Remove "stuff" that are not used anymore
		_instanceSpidSqlTextManager.processMarkedForRemoval();

//System.out.println(">>>>>>>>>>>>>>>>>>> " + _instanceSpidSqlTextManager);
		
		return container;
	}


	private void sendSqlMarker(Connection conn)
	{
		boolean isEnabled = true;
		if ( ! isEnabled )
			return;

		String sqlMarker = "/*SQL-Marker:monSysSQLText: " + (new Timestamp(System.currentTimeMillis())) + "*/ declare @dummy int";

		try (Statement stmnt = conn.createStatement())
		{
			stmnt.executeUpdate(sqlMarker);

			if (_mdaCheckMarker == null)
				_mdaCheckMarker = new MdaCheckMarker();
			_mdaCheckMarker._sqlMarker = sqlMarker;
		}
		catch(SQLException ex)
		{
			_logger.warn("Problems executing 'MDA Check Marker' using SQL '" + sqlMarker + "'. Caught: " + AseUtils.sqlExceptionOrWarningAllToString(ex));
			_mdaCheckMarker = null; // No need to check 
		}
		
	}


//	private boolean includeSqlText(Connection conn, String sqlText)
//	{
//		boolean include = true;
//
//		if (include && _keepSqlTextPattern != null)
//		{
//			include = _keepSqlTextPattern.matcher(sqlText).matches();
//			if ( ! include )
//				_sqlTextSkipCount++;
//		}
//
//		if (include && _skipSqlTextPattern != null)
//		{
//			include = ! _skipSqlTextPattern.matcher(sqlText).matches();
//			if ( ! include )
//				_sqlTextSkipCount++;
//		}
//
//		return include;
//	}
	private boolean includeSqlText(Connection conn, String sqlText)
	{
		if (_skipSqlTextPattern != null)
		{
			if (_skipSqlTextPattern.matcher(sqlText).matches())
			{
				_totalSqlTextSkipCount++;
				return false;
			}
		}

		if (_keepSqlTextPattern != null)
		{
			if ( ! _keepSqlTextPattern.matcher(sqlText).matches() )
			{
				_totalSqlTextSkipCount++;
				return false;
			}
		}

		return true;
	}

	
	private void clearTable(Connection conn, String tabName)
	{
		String sql = "select count(*) from master.dbo." + tabName;
		
		_logger.info("BEGIN: Discarding everything in the transient '" + tabName + "' table in the first sample.");

		long startTime = System.currentTimeMillis();

		try (Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql);)
		{
			int discardCount = 0;
			while(rs.next()) 
			{
				discardCount = rs.getInt(1);
			}

			_logger.info("END:   Discarding everything in the transient '" + tabName + "' table in the first sample. discardCount=" + discardCount + ", this took " + TimeUtils.msToTimeStr(System.currentTimeMillis()-startTime) + ".");
		}
		catch(SQLException ex)
		{
//			_logger.error("END:   Discarding everything in the transient '" + tabName + "' table in the first sample FAILED. Caught: " + AseConnectionUtils.sqlExceptionToString(ex));
			_logger.error("END:   Discarding everything in the transient '" + tabName + "' table in the first sample FAILED. Caught: Error=" + ex.getErrorCode() + ", Msg='" + ex.getMessage().trim() + "'");
		}
	}


	@Override
	protected String getStatisticsMessageAndReset()
	{
		boolean printSpidSqlTextManagerInfo = getConfig().getBooleanProperty(PROPKEY_statistics_printSpidSqlTextManagerInfo, DEFAULT_statistics_printSpidSqlTextManagerInfo);
		if (printSpidSqlTextManagerInfo)
		{
			if (_instanceSpidSqlTextManager != null)
				_logger.info("STATISTICS[" + getName() + "]: DEBUG: printSpidSqlTextManagerInfo: " + _instanceSpidSqlTextManager.toString());
		}

		long tmpTotalSqlTextSkipCount = _totalSqlTextSkipCount;
		long diffTotalSqlTextSkipCount = tmpTotalSqlTextSkipCount - _lastSqlTextSkipCount;

		_lastSqlTextSkipCount = tmpTotalSqlTextSkipCount;
		
		String superMsg = super.getStatisticsMessageAndReset();

		String thisMsg = ", TotalSqlTextSkipCount=" + tmpTotalSqlTextSkipCount + " (" + diffTotalSqlTextSkipCount + ")";
		
		return superMsg + thisMsg;
	}



	//--------------------------------------------------------------------------
	// BEGIN: SPID - SqlText and SqlPlan - manager 
	//--------------------------------------------------------------------------
//	private SpidSqlTextAndPlanManager _spidSqlTextAndPlanManager = new SpidSqlTextAndPlanManager();

	//----------------------------------------------------------------------------------------
	private static class SpidSqlTextManager
	{
		//              spid
		private HashMap<Integer, SpidEntry> _spidMap = new HashMap<>();
		
		/** If the spid do NOT exist create a new entry */
		private SpidEntry getSpidEntry(int spid, int kpid)
		{
			SpidEntry spidEntry = _spidMap.get(spid);
			if (spidEntry == null)
			{
				//System.out.println("  +++ Adding a new SPID entry: spid="+spid+", kpid="+kpid);
				spidEntry = new SpidEntry(spid, kpid);
				_spidMap.put(spid, spidEntry);
			}
			return spidEntry;
		}

		/**
		 * Remove entries that are marked for removals
		 */
		public void processMarkedForRemoval()
		{
			for (SpidEntry spidEntry : _spidMap.values())
			{
				// Removes all batchIdEntries with is marked for removal 
				spidEntry._batchIdMap.values().removeIf(batchEntry -> batchEntry.isMarkedForRemoval());
			}
			
			// remove all SPID's with no batchId's
			_spidMap.values().removeIf(spidEntry -> spidEntry._batchIdMap.isEmpty());
		}

		/**
		 * If the SPID/KPID/BatchId do not exist in cache, do not add 
		 * @return null if not found, otherwise the BatchIdEntry
		 */
		public BatchIdEntry addStatementRecord(int spid, int kpid, int batchID, MonSqlStatementEntry mdaStatement)
		{
			if (mdaStatement == null)
				return null;

			SpidEntry spidEntry = _spidMap.get(spid);
			if (spidEntry != null)
			{
				BatchIdEntry batchEntry = spidEntry.getBatchIdEntryLazy(spid, kpid, batchID);
				if (batchEntry != null)
				{
					batchEntry.addStatementRecord(mdaStatement);
					return batchEntry;
				}
			}
			return null;
		}

		public void addSqlText(int spid, int kpid, int batchId, int sequenceNum, String sqlText, String serverLogin)
		{
//if ("noroles".equals(serverLogin))
//	System.out.println("                0: SpidSqlTextManager.addSqlText(sequenceNum="+sequenceNum+", sqlText=|"+sqlText.replace('\n', ' ')+"|, serverLogin='"+serverLogin+"')");
			getSpidEntry(spid, kpid).addSqlText(spid, kpid, batchId, sequenceNum, sqlText, serverLogin);
		}

		@Override
		public String toString()
		{
			String str = "";
			String comma = "";
			for (SpidEntry spidEntry : _spidMap.values())
			{
				str += comma + "{spid=" + spidEntry._spid + ", kpid=" + spidEntry._kpid +", maxBatchId=" + spidEntry._maxBatchId + ", batchIdMap.size()=" + spidEntry._batchIdMap.size() + "}";
				comma = ", ";
			}
			return "SpidSqlTextManager: spidMap.size()=" + _spidMap.size() + ", spidMap=" + str;
		}
	}

	//----------------------------------------------------------------------------------------
	private static class SpidEntry
	{
		private int _spid;
		private int _kpid;
		private int _maxBatchId;
		
		//              kpid
		private HashMap<Integer, BatchIdEntry> _batchIdMap = new HashMap<>();

		public SpidEntry(int spid, int kpid)
		{
			_spid = spid;
			_kpid = kpid;
		}

		/** Do not add any new BatchIdEntry if they do NOT exists, just return null if no existence */
		public BatchIdEntry getBatchIdEntryLazy(int spid, int kpid, int batchId)
		{
			if (kpid != _kpid)
				return null;

			return _batchIdMap.get(batchId);
		}

		private BatchIdEntry getBatchIdEntry(int spid, int kpid, int batchId)
		{
			// If it's a NEW kpid... then clear the batchId Map
			if (kpid != _kpid)
			{
				//System.out.println("  --- Found NEW KPID for SPID clearing old batchMap: spid="+spid+", new-kpid="+kpid+", current-kpid="+_kpid);
				_batchIdMap.clear();
				_kpid = kpid;
				_maxBatchId = 0;
			}
			
			// Remember the highest batch id (used by "endOfScan" to send/cleanup)
			_maxBatchId = Math.max(batchId, _maxBatchId);

			// Find the BatchId (or create a new one)
			BatchIdEntry batchIdEntry = _batchIdMap.get(batchId);
			if (batchIdEntry == null)
			{
				batchIdEntry = new BatchIdEntry(this, spid, kpid, batchId);
				_batchIdMap.put(batchId, batchIdEntry);
			}

			return batchIdEntry;
		}
		
		public void addSqlText(int spid, int kpid, int batchId, int sequenceNum, String sqlText, String serverLogin)
		{
//if ("noroles".equals(serverLogin))
//	System.out.println("                1: SpidEntry.addSqlText(sequenceNum="+sequenceNum+", sqlText=|"+sqlText.replace('\n', ' ')+"|, serverLogin='"+serverLogin+"')");
			getBatchIdEntry(spid, kpid, batchId).addSqlText(sequenceNum, sqlText, serverLogin);
		}
	}

	//----------------------------------------------------------------------------------------
	private static class BatchIdEntry
	{
		private SpidEntry _parent;
		private int _spid;
		private int _kpid;
		private int _batchId;

		private boolean _markedForRemoval = false;
		private boolean _isOkToSend = false;

		private String  _dynamicSqlName;
		private String  _dynamicSqlText;
		private boolean _isDynamicSqlPrepare = false;

		private StringBuilder _sqlText  = new StringBuilder();
		private String        _srvLogin = "";
		private List<MonSqlStatementEntry> _mdaStatements;

		public BatchIdEntry(SpidEntry spidEntry, int spid, int kpid, int batchId)
		{
			_parent  = spidEntry;
			_spid    = spid;
			_kpid    = kpid;
			_batchId = batchId;
		}

		public boolean isOkToSend() { return _isOkToSend; }

		public void    doMarkForRemoval()   { _markedForRemoval = true; }
		public boolean isMarkedForRemoval() {return _markedForRemoval; }

		public void addStatementRecord(MonSqlStatementEntry mdaStatement)
		{
			if (_mdaStatements == null)
				_mdaStatements = new ArrayList<>();

			_mdaStatements.add(mdaStatement);
			
			// below is some logic to mark the batch as "OK to send"
			// - If it's a Stored Procedure, the proc can be running for a LONG time, but the last StatementNumber in MonSysStatement entry "should" always be 0
			// - If it's a SsqlId (Statement held by the Statement cache)... 
			//      NOTE: if a prepared statement with MANY SQL Statements we will make it on first != 0 even if there might be several ones
			//            My guess is that it's UNUSUAL to have this behavior 
			if (mdaStatement.SsqlId != 0)
				_isOkToSend = true;

			if (mdaStatement.StatementNumber == 0)
				_isOkToSend = true;
			
			// If we can't set '_isOkToSend = true' in here...
			// we will have to wait for "next batchId" to be sent by client before we can "mark" it as "send-able"
		}

		public List<MonSqlStatementEntry> getStatementRecords()
		{
			return _mdaStatements;
		}

		public boolean isDynamicSql()
		{
			return _dynamicSqlName != null;
		}

		/**
		 * Append SQL Text to the buffer (called for every row we get from monSysSQLText)
		 * @param sequenceNum
		 * @param sqlText
		 * @param serverLogin
		 */
		public void addSqlText(int sequenceNum, String sqlText, String serverLogin)
		{
//if ("noroles".equals(serverLogin))
//		System.out.println("                3: BatchIdEntry.addSqlText(sequenceNum="+sequenceNum+", sqlText=|"+sqlText.replace('\n', ' ')+"|, serverLogin='"+serverLogin+"')");
			if (sqlText == null)
				return;

			// Figure out if this is a "Dynamic SQL" entry (probably from CT-Lib)
			if (sqlText.startsWith("DYNAMIC_SQL "))
			{
				_dynamicSqlName = sqlText;
				//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"] :: _dynamicSqlName="+_dynamicSqlName);
			}
			if (_dynamicSqlName != null && StringUtils.startsWithIgnoreCase(sqlText, "create proc "))
			{
				_isDynamicSqlPrepare = true;
				//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"]     -- IS-PREPARE :: _dynamicSqlName="+_dynamicSqlName+"");
				int startPos = StringUtils.indexOfIgnoreCase(sqlText, " as ");
				if (startPos != -1)
				{
					startPos += " as ".length();
					_dynamicSqlText = sqlText.substring(startPos);
					//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"]     -- IS-PREPARE :: _dynamicSqlName="+_dynamicSqlName+", _dynamicSqlText=|"+_dynamicSqlText+"|.");
				}
			}
			if (_isDynamicSqlPrepare && sequenceNum > 1)
			{
				if (_dynamicSqlText == null)
					_dynamicSqlText = sqlText;
				else
					_dynamicSqlText += sqlText;
				//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"]     -- APPEND :: _dynamicSqlName="+_dynamicSqlName+", _dynamicSqlText=|"+_dynamicSqlText+"|.");
			}

			_sqlText.append(sqlText);
			_srvLogin = serverLogin;
		}
		
		/**
		 * Get SQL Text for this BatchID. <br>
		 * If current BatchID is a "Dynamic SQL", the SQL Text is probably NOT located in this BatchID (the execution batchId)...
		 * The "declare" part ("create proc ... as ..." part, which holds the SQL Text) is done in a earlier BatchID, so search the parent BatchID map) 
		 * 
		 * @return
		 */
		public String getSqlText()
		{
			// If the entry is a Dynamic SQL, then the SQLText is NOT stored in the same BatchID as the "execution" part of the Dynamic SQL (prepare, execute..., close)
			// So go and grab any "previous" BatchID where the "full" SqlText are stored.
			if (_dynamicSqlName != null && !_isDynamicSqlPrepare)
			{
				//System.out.println("    ??? getSqlText(): IS-DYNAMIC-SQL: SpidEntry[spid="+_spid+", kpid="+_kpid+", batchId="+_batchId+"] -- DynamicName='"+_dynamicSqlName+"', DynamicSqlText=|"+_dynamicSqlText+"|.");
				for (BatchIdEntry be : _parent._batchIdMap.values())
				{
					if (be._isDynamicSqlPrepare && _dynamicSqlName.equals(be._dynamicSqlName))
					{
						if (be._dynamicSqlText != null)
							return "/* DYNAMIC-SQL: */ " + be._dynamicSqlText;
						return "/* DYNAMIC-SQL: */ " + be._sqlText.toString();
					}
				}
			}
			return _sqlText.toString();
		}

		public String getServerLogin()
		{
			return _srvLogin;
		}
	}

	//--------------------------------------------------------------------------
	// END: SPID - SqlText and SqlPlan - manager 
	//--------------------------------------------------------------------------
	
	
	
	
	



//	//--------------------------------------------------------------------------
//	// BEGIN: SPID - SqlText and SqlPlan - manager 
//	//--------------------------------------------------------------------------
////	private SpidSqlTextAndPlanManager _spidSqlTextAndPlanManager = new SpidSqlTextAndPlanManager();
//
//	//----------------------------------------------------------------------------------------
//	private static class SpidSqlTextAndPlanManager
//	{
//		private HashMap<Integer, SpidEntry> _spidMap = new HashMap<>();
//		
//		private SpidEntry getSpidEntry(int spid, int kpid)
//		{
//			SpidEntry spidEntry = _spidMap.get(spid);
//			if (spidEntry == null)
//			{
//				//System.out.println("  +++ Adding a new SPID entry: spid="+spid+", kpid="+kpid);
//				spidEntry = new SpidEntry(spid, kpid);
//				_spidMap.put(spid, spidEntry);
//			}
//			return spidEntry;
//		}
//
//		public int getBatchIdSize()
//		{
//			int size = 0;
//			for (SpidEntry spidEntry : _spidMap.values())
//			{
//				size += spidEntry._batchIdMap.size();
//			}
//			return size;
//		}
//		public int getSpidSize()
//		{
//			return _spidMap.size();
//		}
//		
//		/** Do not add any new SpidEntry or BatchIdEntry if they do NOT exists, just return null if no existence */
//		public BatchIdEntry getBatchIdEntryLazy(int spid, int kpid, int batchId)
//		{
//			SpidEntry spidEntry = _spidMap.get(spid);
//			if (spidEntry == null)
//				return null;
//
//			return spidEntry.getBatchIdEntryLazy(spid, kpid, batchId);
//		}
//
//		/** Check if a entry exists */
//		public boolean exists(int spid, int kpid, int batchId)
//		{
//			return getBatchIdEntryLazy(spid, kpid, batchId) != null;
//		}
//		
//		public BatchIdEntry getBatchIdEntry(int spid, int kpid, int batchId)
//		{
//			return getSpidEntry(spid, kpid).getBatchIdEntry(spid, kpid, batchId);
//		}
//		
//		public void addSqlText(int spid, int kpid, int batchId, int sequenceNum, String sqlText, String serverLogin)
//		{
//			getSpidEntry(spid, kpid).addSqlText(spid, kpid, batchId, sequenceNum, sqlText, serverLogin);
//		}
////		public String getSqlText(int spid, int kpid, int batchId)
////		{
////			return getSpidEntry(spid, kpid).getSqlText(spid, kpid, batchId);
////		}
//
//		
////		public String getServerLogin(int spid, int kpid, int batchId)
////		{
////			return getSpidEntry(spid, kpid).getServerLogin(spid, kpid, batchId);
////		}
//		
//
//		public void addPlanText(int spid, int kpid, int batchId, int sequenceNum, String planText)
//		{
//			getSpidEntry(spid, kpid).addPlanText(spid, kpid, batchId, sequenceNum, planText);
//		}
////		public String getPlanText(int spid, int kpid, int batchId)
////		{
////			return getSpidEntry(spid, kpid).getPlanText(spid, kpid, batchId);
////		}
//
////		/**
////		 * Get all active SPID's in ASE <br>
////		 * Remove all SPID's from '_spidSqlTextAndPlanManager' that no longer exists in ASE
////		 * 
////		 * @param conn
////		 * @return a Set of SPID's that was removed... which can be used elsewhere, to do the same thing.
////		 */
////		public Set<Integer> removeUnusedSlots(DbxConnection conn)
////		{
//////			String sql = "select SPID, KPID, BatchID from master.dbo.monProcess";
////			String sql = "select spid from master.dbo.sysprocesses";
////
////			try ( Statement stmnt = conn.createStatement(); ResultSet rs = stmnt.executeQuery(sql); )
////			{
////				Set<Integer> dbmsSpidSet = new HashSet<>();
////				while(rs.next())
////				{
////					int spid    = rs.getInt(1);
//////					int kpid    = rs.getInt(2);
//////					int BatchId = rs.getInt(3);
////
////					dbmsSpidSet.add(spid);
////				}
////
////				// 1: Take a copy of SPID MAP
////				// 2: Remove all spid's that exists in ASE 
////				// 2: Remove all entries that is NOT active in ASE  
////				Set<Integer> spidsToBeRemoved = new HashSet<>(_spidMap.keySet());
////				spidsToBeRemoved.removeAll(dbmsSpidSet);
////				// now remove the entries
////				_spidMap.keySet().removeAll(spidsToBeRemoved);
////
////				if ( ! spidsToBeRemoved.isEmpty() )
////					_logger.debug("Removed the following SPIDs from the SqlText/PlanText structure, which was no longer present in ASE. size=" + spidsToBeRemoved.size() + ", removeSet=" + spidsToBeRemoved);
////				
//////				if ( ! spidsToBeRemoved.isEmpty() )
//////					_logger.info("Removed the following SPIDs from the SqlText/PlanText structure, which was no longer present in ASE. size=" + spidsToBeRemoved.size() + ", removeSet=" + spidsToBeRemoved);
////
////				return spidsToBeRemoved;
////			}
////			catch(Exception ex)
////			{
////				_logger.error("Problem cleaning up unused SPID/KPID/BatchID SqlTest/PlanText slots, using SQL='" + sql + "'. Caught: " + ex);
////			}
////
////			return Collections.emptySet();
////		}
//
//		/** Remove any SqlText and PlanText that is no longer used (save only last BatchId) */
//		public void endOfScan()
//		{
//			for (SpidEntry spidEntry : _spidMap.values())
//			{
//				spidEntry.endOfScan();
//			}
//		}
//
////		public int getBatchIdCountAndReset()
////		{
////			int count = 0;
////			for (SpidEntry spidEntry : _spidMap.values())
////			{
////				count += spidEntry.getBatchIdCountAndReset();
////			}
////			return count;
////		}
//		
//
//		/** Clear the structure, can for example be used if we starting to get LOW on memory */
//		public void clear()
//		{
//			_spidMap.clear();
////			_spidMap = new HashMap<>();
//		}
//	}
//
//	//----------------------------------------------------------------------------------------
//	private static class SpidEntry
//	{
//		private int _spid;
//		private int _kpid;
//		private int _maxBatchId;
////		private int _batchIdCount;
//		
//		private HashMap<Integer, BatchIdEntry> _batchIdMap = new HashMap<>();
//
//		public SpidEntry(int spid, int kpid)
//		{
//			_spid = spid;
//			_kpid = kpid;
//		}
//
//		/** Remove all BatchId entries that is no longer used */ 
//		public void endOfScan()
//		{
//			if (_batchIdMap.size() <= 1)
//				return;
//
//			BatchIdEntry maxBatchIdEntry = _batchIdMap.get(_maxBatchId);
//			//System.out.println("    EOS -- _batchIdMap.size="+_batchIdMap.size()+", maxBatchIdEntry=[spid="+maxBatchIdEntry._spid+", batchId="+maxBatchIdEntry._batchId+"] -- DynamicName='"+maxBatchIdEntry._dynamicSqlName+"', DynamicSqlText=|"+maxBatchIdEntry._dynamicSqlText+"|.");//, sqlText="+maxBatchIdEntry._sqlText);
//
//			// Loop all entries... and remove the ones that we no longer need
//			for(Iterator<Entry<Integer, BatchIdEntry>> it = _batchIdMap.entrySet().iterator(); it.hasNext(); )
//			{
//				// Get Key/value from the iterator
//				Entry<Integer, BatchIdEntry> entry = it.next();
//				int          batchId = entry.getKey();
//				BatchIdEntry be      = entry.getValue();
//
//				if(batchId < _maxBatchId) 
//				{
//					boolean remove = true;
//
//					// if "last/max" entry has same "Dynamic SQL Name" as the entry we are looking at... 
//					// then KEEP the entry (since we need "all/previous" batchId to get SqlText) 
//					if (maxBatchIdEntry.isDynamicSql())
//					{
//						if (maxBatchIdEntry._dynamicSqlName.equals(be._dynamicSqlName) && be._isDynamicSqlPrepare)
//						{
//							//System.out.println("    <-- KEEP-DYNAMIC-SQL: SpidEntry[spid="+_spid+", kpid="+_kpid+", batchId="+be._batchId+"] -- DynamicName='"+be._dynamicSqlName+"', DynamicSqlText=|"+be._dynamicSqlText+"|, sqlText="+be._sqlText);
//							remove = false;
//						}
//					}
//					
//					// if we want to keep a BatchId's (and it's content) for more than one -END-OF-SCAN- the "keepCounter" can be used.
//					// Note: The initial value for the keepCounter is set in BatchIdEntry class
//					if (be._keepCounter > 0)
//					{
//						be._keepCounter--;
//						remove = false;
//					}
//					
//					if (remove)
//					{
//						//System.out.println("    <-- SpidEntry[spid="+_spid+", kpid="+_kpid+", maxBatchId="+_maxBatchId+"] -- removing: batchId=" + entry.getValue()._batchId + ", sqlText="+entry.getValue().getSqlText());
//						it.remove();
//						
//						// Should we remove entries for SpidInfo here since it wont be referenced anymore
//						// When entries from monSysStatements has "aged out"... they wont be needed anymore
//						//_spidInfoManager.remove(be._spid, be._kpid, be._batchId);
//						//_waitInfo       .remove(be._spid, be._kpid, be._batchId);
//					}
//				}
//			}
//		}
//
//		/** Do not add any new BatchIdEntry if they do NOT exists, just return null if no existence */
//		public BatchIdEntry getBatchIdEntryLazy(int spid, int kpid, int batchId)
//		{
//			if (kpid != _kpid)
//				return null;
//
//			return _batchIdMap.get(batchId);
//		}
//
//		private BatchIdEntry getBatchIdEntry(int spid, int kpid, int batchId)
//		{
//			// If it's a NEW kpid... then clear the batchId Map
//			if (kpid != _kpid)
//			{
//				//System.out.println("  --- Found NEW KPID for SPID clearing old batchMap: spid="+spid+", new-kpid="+kpid+", current-kpid="+_kpid);
//				_batchIdMap.clear();
//				_maxBatchId = 0;
//				_kpid = kpid;
//			}
//			
//			// Remember the highest batch id (used by endOfScan() to cleanup)
//			_maxBatchId = Math.max(batchId, _maxBatchId);
////			if (batchId > _maxBatchId)
////			{
////				_batchIdCount += batchId - _maxBatchId;
////				_maxBatchId = batchId;
////			}
//
//			// Find the BatchId (or create a new one)
//			BatchIdEntry batchIdEntry = _batchIdMap.get(batchId);
//			if (batchIdEntry == null)
//			{
//				batchIdEntry = new BatchIdEntry(this, spid, kpid, batchId);
//				_batchIdMap.put(batchId, batchIdEntry);
//			}
//
//			return batchIdEntry;
//		}
//		
////		public int getBatchIdCountAndReset()
////		{
////			int count = _batchIdCount;
////			_batchIdCount = 0;
////			return count;
////		}
//
//		public void addSqlText(int spid, int kpid, int batchId, int sequenceNum, String sqlText, String serverLogin)
//		{
//			getBatchIdEntry(spid, kpid, batchId).addSqlText(sequenceNum, sqlText, serverLogin);
//		}
////		public String getSqlText(int spid, int kpid, int batchId)
////		{
////			return getBatchIdEntry(spid, kpid, batchId).getSqlText();
////		}
//
////		public String getServerLogin(int spid, int kpid, int batchId)
////		{
////			return getBatchIdEntry(spid, kpid, batchId).getServerLogin();
////		}
//
//		public void addPlanText(int spid, int kpid, int batchId, int sequenceNum, String planText)
//		{
//			getBatchIdEntry(spid, kpid, batchId).addPlanText(sequenceNum, planText);
//		}
////		public String getPlanText(int spid, int kpid, int batchId)
////		{
////			return getBatchIdEntry(spid, kpid, batchId).getPlanText();
////		}
//	}
//
//	//----------------------------------------------------------------------------------------
//	private static class BatchIdEntry
//	{
//		private SpidEntry _parent;
//		private int _spid;
//		private int _kpid;
//		private int _batchId;
//
//		// if we want to keep a BatchId's (and it's content) for more than one -END-OF-SCAN- the "keepCounter" can be used.
//		// This is decremented in SpidEntry.endOfScan(), and when it reaches 0 it is removed.
//		public int _keepCounter = 0; 
//
//		private String  _dynamicSqlName;
//		private String  _dynamicSqlText;
//		private boolean _isDynamicSqlPrepare = false;
////		private BatchIdEntry _dynamicSqlPrepareDirectLink; // if we have already have back-traced the origin BatchIdEntry which holds the "create proc text" lets remember it...
//
//		private StringBuilder _sqlText  = new StringBuilder();
//		private StringBuilder _planText = new StringBuilder();
//		private String        _srvLogin = "";
//
//		public BatchIdEntry(SpidEntry spidEntry, int spid, int kpid, int batchId)
//		{
//			_parent  = spidEntry;
//			_spid    = spid;
//			_kpid    = kpid;
//			_batchId = batchId;
//			
//			_keepCounter = _default_batchIdEntry_keepCount;
//		}
//
//		public boolean isDynamicSql()
//		{
//			return _dynamicSqlName != null;
//		}
//
//		/**
//		 * Append SQL Text to the buffer (called for every row we get from monSysSQLText)
//		 * @param sequenceNum
//		 * @param sqlText
//		 * @param serverLogin
//		 */
//		public void addSqlText(int sequenceNum, String sqlText, String serverLogin)
//		{
//			if (sqlText == null)
//				return;
//
//			// Figure out if this is a "Dynamic SQL" entry (probably from CT-Lib)
//			if (sqlText.startsWith("DYNAMIC_SQL "))
//			{
//				_dynamicSqlName = sqlText;
//				//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"] :: _dynamicSqlName="+_dynamicSqlName);
//			}
//			if (_dynamicSqlName != null && StringUtils.startsWithIgnoreCase(sqlText, "create proc "))
//			{
//				_isDynamicSqlPrepare = true;
//				//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"]     -- IS-PREPARE :: _dynamicSqlName="+_dynamicSqlName+"");
//				int startPos = StringUtils.indexOfIgnoreCase(sqlText, " as ");
//				if (startPos != -1)
//				{
//					startPos += " as ".length();
//					_dynamicSqlText = sqlText.substring(startPos);
//					//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"]     -- IS-PREPARE :: _dynamicSqlName="+_dynamicSqlName+", _dynamicSqlText=|"+_dynamicSqlText+"|.");
//				}
//			}
//			if (_isDynamicSqlPrepare && sequenceNum > 1)
//			{
//				if (_dynamicSqlText == null)
//					_dynamicSqlText = sqlText;
//				else
//					_dynamicSqlText += sqlText;
//				//System.out.println("    --> DYNAMIC_SQL [spid="+_spid+", batchId="+_batchId+"]     -- APPEND :: _dynamicSqlName="+_dynamicSqlName+", _dynamicSqlText=|"+_dynamicSqlText+"|.");
//			}
//
//			_sqlText.append(sqlText);
//			_srvLogin = serverLogin;
//		}
//		
//		/**
//		 * Get SQL Text for this BatchID. <br>
//		 * If current BatchID is a "Dynamic SQL", the SQL Text is probably NOT located in this BatchID (the execution batchId)...
//		 * The "declare" part ("create proc ... as ..." part, which holds the SQL Text) is done in a earlier BatchID, so search the parent BatchID map) 
//		 * 
//		 * @return
//		 */
//		public String getSqlText()
//		{
//			// If the entry is a Dynamic SQL, then the SQLText is NOT stored in the same BatchID as the "execution" part of the Dynamic SQL (prepare, execute..., close)
//			// So go and grab any "previous" BatchID where the "full" SqlText are stored.
//			if (_dynamicSqlName != null && !_isDynamicSqlPrepare)
//			{
//				//System.out.println("    ??? getSqlText(): IS-DYNAMIC-SQL: SpidEntry[spid="+_spid+", kpid="+_kpid+", batchId="+_batchId+"] -- DynamicName='"+_dynamicSqlName+"', DynamicSqlText=|"+_dynamicSqlText+"|.");
//				for (BatchIdEntry be : _parent._batchIdMap.values())
//				{
//					if (be._isDynamicSqlPrepare && _dynamicSqlName.equals(be._dynamicSqlName))
//					{
//						if (be._dynamicSqlText != null)
//							return "/* DYNAMIC-SQL: */ " + be._dynamicSqlText;
//						return "/* DYNAMIC-SQL: */ " + be._sqlText.toString();
//					}
//				}
//			}
//			return _sqlText.toString();
//		}
//
//		public String getServerLogin()
//		{
//			return _srvLogin;
//		}
//
//		public void addPlanText(int sequenceNum, String planText)
//		{
//			_planText.append(planText);
//		}
//		public String getPlanText()
//		{
//			return _planText.toString();
//		}
//	}
//
//	//--------------------------------------------------------------------------
//	// END: SPID - SqlText and SqlPlan - manager 
//	//--------------------------------------------------------------------------
}
