/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.collectors.records;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.ObjectMapper;

import sek.ase.auditor.Version;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.utils.TimeUtils;
import sek.ase.auditor.wqs.WriterQueueContainer;

public abstract class AuditRecord
{
	private static Logger _logger = LogManager.getLogger();

	protected WriterQueueContainer _container;

	private static final int JSON_MESSAGE_VERSION = 1;

	/**
	 * Set the Container object so we can reach "stuff" in there<br>
	 * Typically called from: {@link WriterQueueContainer#add(AuditRecord)}  
	 * 
	 * @param container  The container which holds this Record
	 */
	public void setContainer(WriterQueueContainer container)
	{
		_container = container;
	}

	/**
	 * Get the container that Record was added to
	 * @return
	 */
	public WriterQueueContainer getContainer()
	{
		return _container;
	}
	
	
	public boolean isEmpty()
	{
		return false;
	}

	/**
	 * Create JSON Content for a specific Record Type
	 * @return
	 */
	public abstract void createJsonForRecord(JsonGenerator gen) throws IOException;


	/**
	 * Create JSON Content 
	 * @return
	 */
	public void createJson(JsonGenerator gen) 
	throws IOException
	{
		gen.writeStartObject();

		// The 'head' Object
		gen.writeFieldName("head");
		gen.writeStartObject();
			gen.writeNumberField("messageVersion"         , JSON_MESSAGE_VERSION);
			gen.writeStringField("appName"                , Version.getAppName());
			gen.writeStringField("appVersion"             , Version.getVersionStr());
			gen.writeStringField("appBuildString"         , Version.getBuildStr());
//			gen.writeStringField("collectorHostname"      , collectorHostname);
//			gen.writeNumberField("collectorSampleInterval", collectorSampleInterval);
//			gen.writeStringField("collectorCurrentUrl"    , collectorCurrentUrl);
//			gen.writeStringField("collectorInfoFile"      , collectorInfoFile);

			WriterQueueContainer cont = getContainer();
//			gen.writeStringField("sessionStartTime"       , TimeUtils.toStringIso8601(cont.getSessionStartTime()));
//			gen.writeStringField("sessionSampleTime"      , TimeUtils.toStringIso8601(cont.getMainSampleTime()));
			gen.writeStringField("sampleTime"             , TimeUtils.toStringIso8601(cont.getMainSampleTime()));
			gen.writeStringField("serverName"             , cont.getServerName());
			gen.writeStringField("onHostname"             , cont.getOnHostname());
//			gen.writeStringField("serverNameAlias"        , cont.getServerNameAlias());
//			gen.writeStringField("serverDisplayName"      , cont.getServerDisplayName());
			gen.writeStringField("recordType"             , this.getClass().getSimpleName());
		gen.writeEndObject();

		//-----------------------------------------------------------
		// Create Any specific JSON entries for any Record Type
		//-----------------------------------------------------------
		createJsonForRecord(gen);

		// end JSON Object
		gen.writeEndObject();
	}


	/**
	 * Transform the Audit Record into a JSON String
	 * @return
	 */
	public String toJson()
	throws IOException
	{
		StringWriter sw = new StringWriter();

		// use PersistContainer.toJsonMessage() as an template
		JsonFactory jfactory = new JsonFactory();
		JsonGenerator gen = jfactory.createGenerator(sw);
		gen.setPrettyPrinter(new DefaultPrettyPrinter());
		gen.setCodec(new ObjectMapper(jfactory));

		// Do the WORK
		createJson(gen);

		gen.close();
		
		// And make it to a string
		String jsonStr = sw.toString();
		
		// Debug write to a file, if property is set
		if (Configuration.getCombinedConfiguration().hasProperty("AuditRecordSybStmnt.debug.toJson.writeToFile"))
		{
			//-------------------------------------------------------------------------------------------------------------------
			// Property example: PersistWriterToHttpJson.debug.writeToFile = c:\tmp\PersistWriterToHttpJson.tmp.json
			//-------------------------------------------------------------------------------------------------------------------
			String debugToFileNameStr = Configuration.getCombinedConfiguration().getProperty("PersistContainer.debug.toJson.writeToFile", "");
			if (StringUtil.hasValue(debugToFileNameStr))
			{
				if ("STDOUT".equalsIgnoreCase(debugToFileNameStr))
				{
					System.out.println("#### BEGIN JSON #####################################################################");
					System.out.println(jsonStr);
					System.out.println("#### END JSON #######################################################################");
				}
				else
				{
					File debugToFileName = new File(debugToFileNameStr);
					
					try
					{
						_logger.info("Writing JSON message to DEBUG file '" + debugToFileName.getAbsolutePath() + "'.");
						FileUtils.writeStringToFile(debugToFileName, jsonStr, StandardCharsets.UTF_8);
					}
					catch(Exception ex)
					{
						_logger.error("PROBLEMS Writing JSON message to DEBUG file '" + debugToFileName.getAbsolutePath() + "', skipping and continuing.", ex);
					}
				}
			}
		}
		
		return jsonStr;
	}
	
}
