/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.wqs.consumers;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.ObjectMapper;

import sek.ase.auditor.Version;
import sek.ase.auditor.collectors.records.AuditRecord;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.utils.TimeUtils;
import sek.ase.auditor.wqs.WriterQueueContainer;
import sek.ase.auditor.wqs.WriterQueueStatistics;

public class WriterToFile
implements IWriterConsumer, Runnable
{
	private static Logger _logger = LogManager.getLogger();

	public static final String  PROPKEY_saveThreadCount = "WriterToFile.saveThreadCount";
	public static final int     DEFAULT_saveThreadCount = 1;

	public static final String  PROPKEY_saveDir          = "WriterToFile.saveDir";
	public static final String  DEFAULT_saveDir          = System.getProperty("java.io.tmpdir") + Version.getAppName() + "/WriterToFile";

	public static final String  PROPKEY_jsonPrettyPrint    = "WriterToFile.json.prettyPrint";
	public static final boolean DEFAULT_jsonPrettyPrint    = false;

	private String  _saveDir         = DEFAULT_saveDir;
	private boolean _jsonPrettyPrint = DEFAULT_jsonPrettyPrint;
	
	private Configuration _conf;

	private int _threadCount;
	private List<FileSaver> _writers = new ArrayList<>();

	WriterQueueStatisticsLocal _stats = new WriterQueueStatisticsLocal();
	
//	private BlockingQueue<WriterQueueContainer> _containerQueue = new LinkedBlockingQueue<WriterQueueContainer>();
	private BlockingQueue<AuditRecord> _auditRecordQueue = new LinkedBlockingQueue<AuditRecord>();
	
	@Override
	public void init(Configuration conf) throws Exception
	{
		_conf = conf;

		_saveDir         = _conf.getProperty(       PROPKEY_saveDir        , DEFAULT_saveDir);
		_jsonPrettyPrint = _conf.getBooleanProperty(PROPKEY_jsonPrettyPrint, DEFAULT_jsonPrettyPrint);

		_threadCount = _conf.getIntProperty(PROPKEY_saveThreadCount, DEFAULT_saveThreadCount);
		
		// Check config for mandatory parameters
		if (_threadCount <= 0)                          throw new Exception("Number of threads cant be " + _threadCount + ". Change this to be 1 or above. Config: " + PROPKEY_saveThreadCount + " = " + DEFAULT_saveThreadCount);
		if (StringUtil.isNullOrBlank(_saveDir))         throw new Exception("Save Dir cant be empty. Config: "          + PROPKEY_saveDir         + " = /full/path/to/save/dir");

	
		// Check if SAVE Directory EXISTS (and that we can write to it)
		File f = new File(_saveDir);
		if ( ! f.exists() )
			throw new Exception("Save Directory '" + f + "' does NOT exist. Please create it.");

		// Check if we are allowed to write to the SAVE Directory
		File tmpFileName = new File(_saveDir, "dummyFile.deleteme");
		try
		{
			FileUtils.writeStringToFile(tmpFileName, "Test String", StandardCharsets.UTF_8);
		}
		catch (IOException ex)
		{
			throw new Exception("Problems writing a test file in directory '" + f + "'. Caught:" + ex, ex);
		}
		finally
		{
			tmpFileName.delete();
		}
	}

	@Override
	public void close()
	{
	}

	@Override
	public Configuration getConfig()
	{
		return _conf;
	}

	@Override
	public String getConfigStr()
	{
		return "";
	}

	@Override
	public void printConfig()
	{
		int len = 40;

		_logger.info("Configuration for '" + getName() + "', with full class name '" + getClass().getName() + "'.");
		_logger.info("                   " + StringUtil.left(PROPKEY_saveDir          , len) + " = " + _saveDir);
		_logger.info("                   " + StringUtil.left(PROPKEY_jsonPrettyPrint  , len) + " = " + _jsonPrettyPrint);
		_logger.info("                   " + StringUtil.left(PROPKEY_saveThreadCount  , len) + " = " + _threadCount);
	}

	@Override
	public void beginOfSample(WriterQueueContainer cont)
	{
		// all done in saveRecord();
	}

	@Override
	public void saveSample(WriterQueueContainer cont)
	{
		// all done in saveRecord();
	}

	@Override
	public void saveRecord(AuditRecord ar)
	{
		_auditRecordQueue.add(ar);
	}

	@Override
	public void endOfSample(WriterQueueContainer cont, boolean caughtErrors)
	{
		// all done in saveRecord();
	}

	@Override
	public void startServices() throws Exception
	{
		_logger.info("Starting " + _threadCount + " FileSaver(s).");

		for (int i = 0; i < _threadCount; i++)
		{
			FileSaver fs = new FileSaver(this, i);
			fs.start();
			
			_writers.add(fs);
		}

//		_logger.info("Done Starting " + _threadCount + " FileSaver(s).");
	}

	@Override
	public void stopServices(int maxWaitTimeInMs)
	{
		_logger.info("Stopping " + _writers.size() + " FileSaver(s).");

		for (FileSaver splunkSender : _writers)
		{
			splunkSender.stop();
		}
		
		for (FileSaver fileSaver : _writers)
		{
			try { fileSaver._thread.join(); }
			catch (InterruptedException ignore) {}
		}
		
		_logger.info("Done Stopping " + _writers.size() + " FileSaver(s).");
	}

	@Override
	public String getName()
	{
		return this.getClass().getSimpleName();
	}

	@Override
	public void queueSizeWarning(int queueSize, int thresholdSize)
	{
		// TODO Auto-generated method stub
		
	}

	@Override
	public WriterQueueStatistics getStatistics()
	{
		return _stats;
	}

	@Override
	public void resetStatistics()
	{
		// TODO Auto-generated method stub
		
	}

	@Override
	public void run()
	{
		// TODO Auto-generated method stub
		
	}

	private static class WriterQueueStatisticsLocal
	extends WriterQueueStatistics
	{
		private AtomicLong _totalSaveCount     = new AtomicLong();
		private AtomicLong _totalFailCount     = new AtomicLong();
		private long       _lastTotalSaveCount = 0;
		private long       _lastTotalFailCount = 0;
		
		public long incrementSaveCount()
		{
			return _totalSaveCount.incrementAndGet();
		}
		public long incrementFailCount()
		{
			return _totalFailCount.incrementAndGet();
		}
		
		@Override
		public String toString()
		{
			long tmpTotalSaveCount = _totalSaveCount.get();
			long tmpTotalFailCount = _totalFailCount.get();

			long diffTotalSaveCount = tmpTotalSaveCount - _lastTotalSaveCount;
			long diffTotalFailCount = tmpTotalFailCount - _lastTotalFailCount;

			_lastTotalSaveCount = tmpTotalSaveCount;
			_lastTotalFailCount = tmpTotalFailCount;

			return "totalSendCount=" + _totalSaveCount + " (" + diffTotalSaveCount + "), totalFailCount=" + _totalFailCount + " (" + diffTotalFailCount + ")";
		}
	}

	//---------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------
	/**
	 * This the class that are responsible for grabbing entries from the internal queue and SAVING information to a file
	 * <p>
	 * NOTE: That it can be <b>many</b> instances of this, if we are saving records to File in parallel 
	 */
	private static class FileSaver
	implements Runnable
	{
		private Thread _thread;
		private WriterToFile _parent;
		private int _id;

		private boolean _running;

		public String getName()
		{
			return "FileSaver-" + _id;
		}
		
		public FileSaver(WriterToFile parent, int id)
		{
			_parent = parent;
			_id     = id + 1;
		}

		public void start()
		{
			_thread = new Thread(this, getName());
			_thread.setDaemon(true);
			_thread.start();
		}
		
		public void stop()
		{
			if (_thread != null)
			{
				_thread.interrupt();
			}
		}

		public boolean isRunning()
		{
			return _running;
		}

		@Override
		public void run()
		{
			_running = true;

			_logger.info("Starting Thread for: " + getName());
			while(_running)
			{
				try
				{
					// GET A RECORD
					AuditRecord auditRecord = _parent._auditRecordQueue.take();

					// Make sure the auditRecord isn't empty.
					if (auditRecord == null || (auditRecord != null && auditRecord.isEmpty()) )
						continue;

					// APPLY FILTER
					if ( ! isIncluded(auditRecord) )
						continue;

					// if we are about to STOP the service
					if ( ! isRunning() )
					{
						_logger.info("The service is about to stop, discarding a consume(AuditRecord) queue entry.");
						continue;
					}

					//--------------------------------------------
					// Here is where it's all done
					//--------------------------------------------
					consume( auditRecord );

				}
				catch (InterruptedException ex) 
				{
					_logger.info("Received 'InterruptedException' in '" + getName() + "', time to stop...");
					_running = false;
				}
				catch(Throwable t)
				{
					_logger.error("Found some issue in '" + getName() + "', continuing with next message. Caught: " + t, t);
				}
			}
			_logger.info("Ending Thread for " + getName());
		}

		private boolean isIncluded(AuditRecord auditRecord)
		{
			return true;
		}

		private void consume(AuditRecord auditRecord)
		{
			try
			{
				StringWriter sw = new StringWriter();

				// use PersistContainer.toJsonMessage() as an template
				JsonFactory jfactory = new JsonFactory();
				JsonGenerator gen = jfactory.createGenerator(sw);
				if (_parent._jsonPrettyPrint)
					gen.setPrettyPrinter(new DefaultPrettyPrinter());
				gen.setCodec(new ObjectMapper(jfactory));

				// Create a JSON Message that is representing the AuditRecord
				auditRecord.createJson(gen);

				gen.close();
				
				// And make it to a string
				String jsonStr = sw.toString();

				// SAVE to a file
//				String ts = "" + System.currentTimeMillis();
				String ts = TimeUtils.getCurrentTimeForFileNameYmdHmsMs();
				long nanoTs = System.nanoTime();
				File fileName = new File(_parent._saveDir, getName() + "." + ts + "." + nanoTs + ".json");
				FileUtils.writeStringToFile(fileName, jsonStr, StandardCharsets.UTF_8);
				
				_logger.info(getName() + ": Saved JSON content to file '" + fileName + "'.");

				_parent._stats.incrementSaveCount();
			}
			catch (IOException ex)
			{
				_logger.error("Problems saving JSON message to File.", ex);

				_parent._stats.incrementFailCount();
			}
		}
	}
}
