/*******************************************************************************
 * SEK ASE Auditor -- Created by: goran.schwarz@executeit.se
 ******************************************************************************/
package sek.ase.auditor.wqs.consumers;

import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.ObjectMapper;

import sek.ase.auditor.collectors.records.AuditRecord;
import sek.ase.auditor.utils.Configuration;
import sek.ase.auditor.utils.StringUtil;
import sek.ase.auditor.wqs.WriterQueueContainer;
import sek.ase.auditor.wqs.WriterQueueStatistics;

public class WriterToSplunk
implements IWriterConsumer, Runnable
{
	private static Logger _logger = LogManager.getLogger();

	public static final String  PROPKEY_senderThreadCount = "WriterToSplunk.senderThreadCount";
	public static final int     DEFAULT_senderThreadCount = 3;

	public static final String  PROPKEY_splunkUrl          = "WriterToSplunk.url";
	public static final String  DEFAULT_splunkUrl          = null;
//	public static final String  DEFAULT_splunkUrl          = "http://splunk-1-gs:8088/services/collector";

//	public static final String  PROPKEY_splunkIndexName    = "WriterToSplunk.index.name";
//	public static final String  DEFAULT_splunkIndexName    = null;
//	public static final String  DEFAULT_splunkIndexName    = "sek_ase_auditing_index";

	public static final String  PROPKEY_splunkAccessToken  = "WriterToSplunk.access.token";
	public static final String  DEFAULT_splunkAccessToken  = null;
//	public static final String  DEFAULT_splunkAccessToken  = "Splunk 4bccb251-43f7-413b-8a03-029040d2e330";

	public static final String  PROPKEY_jsonPrettyPrint    = "WriterToSplunk.json.prettyPrint";
	public static final boolean DEFAULT_jsonPrettyPrint    = false;

	public static final String  PROPKEY_printJsonBeforeSend = "WriterToSplunk.json.printBeforeSend";
	public static final boolean DEFAULT_printJsonBeforeSend = false;

	public static final String  PROPKEY_json_onErrorPrintXchars = "WriterToSplunk.json.on.error.print.numOfChars";
	public static final int     DEFAULT_json_onErrorPrintXchars = 512;

	public static final String  PROPKEY_httpConnPoolMaxSize = "WriterToSplunk.http.conn.pool.maxSize";
	public static final int     DEFAULT_httpConnPoolMaxSize = 100;

	public static final String  PROPKEY_httpConnPoolMaxSizePerRoute = "WriterToSplunk.http.conn.pool.maxSizePerRoute";
	public static final int     DEFAULT_httpConnPoolMaxSizePerRoute = 20;

	
	private String _splunkUrl         = DEFAULT_splunkUrl;
//	private String _splunkIndexName   = DEFAULT_splunkIndexName;
	private String _splunkAccessToken = DEFAULT_splunkAccessToken;
	
	private boolean _jsonPrettyPrint     = DEFAULT_jsonPrettyPrint;
	private boolean _jsonPrintBeforeSend = DEFAULT_printJsonBeforeSend;
	
	private Configuration _conf;

	private int _threadCount;
	private List<SplunkSender> _senders = new ArrayList<>();

	WriterQueueStatisticsLocal _stats = new WriterQueueStatisticsLocal();
	
//	private BlockingQueue<WriterQueueContainer> _containerQueue = new LinkedBlockingQueue<WriterQueueContainer>();
	private BlockingQueue<AuditRecord> _auditRecordQueue = new LinkedBlockingQueue<AuditRecord>();
	
	PoolingHttpClientConnectionManager _apacheHttpConnectionManager;

	@Override
	public void init(Configuration conf) 
	throws Exception
	{
		_conf = conf;

		_splunkUrl           = _conf.getProperty       (PROPKEY_splunkUrl          , DEFAULT_splunkUrl);
//		_splunkIndexName     = _conf.getProperty       (PROPKEY_splunkIndexName    , DEFAULT_splunkIndexName);
		_splunkAccessToken   = _conf.getProperty       (PROPKEY_splunkAccessToken  , DEFAULT_splunkAccessToken);
		
		_jsonPrettyPrint     = _conf.getBooleanProperty(PROPKEY_jsonPrettyPrint    , DEFAULT_jsonPrettyPrint);
		_jsonPrintBeforeSend = _conf.getBooleanProperty(PROPKEY_printJsonBeforeSend, DEFAULT_printJsonBeforeSend);

		_threadCount         = _conf.getIntProperty    (PROPKEY_senderThreadCount  , DEFAULT_senderThreadCount);
		
		// Check config for mandatory parameters
		if (_threadCount <= 0)                            throw new Exception("Number of threads cant be " + _threadCount + ". Change this to be 1 or above. Config: " + PROPKEY_senderThreadCount + " = " + DEFAULT_senderThreadCount);
		if (StringUtil.isNullOrBlank(_splunkUrl))         throw new Exception("Splunk URL cant be empty. Config: "          + PROPKEY_splunkUrl         + " = http://hostname:port/...");
//		if (StringUtil.isNullOrBlank(_splunkIndexName))   throw new Exception("Splunk Index Name cant be empty. Config: "   + PROPKEY_splunkIndexName   + " = nameOfIndexToStoreInfo");
		if (StringUtil.isNullOrBlank(_splunkAccessToken)) throw new Exception("Splunk Access Token cant be empty. Config: " + DEFAULT_splunkAccessToken + " = accessToken");

		// Create a connection pool for the http connections, otherwise we will/may run into "java.net.BindException: Address already in use: JVM_Bind"
		_apacheHttpConnectionManager = new PoolingHttpClientConnectionManager();
		_apacheHttpConnectionManager.setMaxTotal          (conf.getIntProperty(PROPKEY_httpConnPoolMaxSize        , DEFAULT_httpConnPoolMaxSize));
		_apacheHttpConnectionManager.setDefaultMaxPerRoute(conf.getIntProperty(PROPKEY_httpConnPoolMaxSizePerRoute, DEFAULT_httpConnPoolMaxSizePerRoute));

	}

	@Override
	public void close()
	{
	}

	@Override
	public Configuration getConfig()
	{
		return _conf;
	}

	@Override
	public String getConfigStr()
	{
		return "";
	}

	@Override
	public void printConfig()
	{
		int len = 60;

		_logger.info("Configuration for '" + getName() + "', with full class name '" + getClass().getName() + "'.");
		_logger.info("                   " + StringUtil.left(PROPKEY_splunkUrl           , len) + " = " + _splunkUrl);
//		_logger.info("                   " + StringUtil.left(PROPKEY_splunkIndexName     , len) + " = " + _splunkIndexName);
		_logger.info("                   " + StringUtil.left(PROPKEY_splunkAccessToken   , len) + " = " + _splunkAccessToken);
		_logger.info("                   " + StringUtil.left(PROPKEY_jsonPrettyPrint     , len) + " = " + _jsonPrettyPrint);
		_logger.info("                   " + StringUtil.left(PROPKEY_printJsonBeforeSend , len) + " = " + _jsonPrintBeforeSend);
		_logger.info("                   " + StringUtil.left(PROPKEY_senderThreadCount   , len) + " = " + _threadCount);
	}

	@Override
	public void beginOfSample(WriterQueueContainer cont)
	{
		// all done in saveRecord();
	}

	@Override
	public void saveSample(WriterQueueContainer cont)
	{
		// all done in saveRecord();
	}

	@Override
	public void saveRecord(AuditRecord ar)
	{
		_auditRecordQueue.add(ar);
	}

	@Override
	public void endOfSample(WriterQueueContainer cont, boolean caughtErrors)
	{
		// all done in saveRecord();
	}

	@Override
	public void startServices() throws Exception
	{
		_logger.info("Starting " + _threadCount + " SplunkSender(s).");

		for (int i = 0; i < _threadCount; i++)
		{
			SplunkSender sr = new SplunkSender(this, i);
			sr.start();
			
			_senders.add(sr);
		}

//		_logger.info("Done Starting " + _threadCount + " SplunkSender(s).");
	}

	@Override
	public void stopServices(int maxWaitTimeInMs)
	{
		_logger.info("Stopping " + _senders.size() + " SplunkSender(s).");

		for (SplunkSender splunkSender : _senders)
		{
			splunkSender.stop();
		}
		
		for (SplunkSender splunkSender : _senders)
		{
			try { splunkSender._thread.join(); }
			catch (InterruptedException ignore) {}
		}
		
		_logger.info("Done Stopping " + _senders.size() + " SplunkSender(s).");
	}

	@Override
	public String getName()
	{
		return this.getClass().getSimpleName();
	}

	@Override
	public void queueSizeWarning(int queueSize, int thresholdSize)
	{
		// TODO Auto-generated method stub
		
	}

	@Override
	public WriterQueueStatistics getStatistics()
	{
		return _stats;
	}

	@Override
	public void resetStatistics()
	{
		// TODO Auto-generated method stub
		
	}

	@Override
	public void run()
	{
		// TODO Auto-generated method stub
		
	}


	private static class WriterQueueStatisticsLocal
	extends WriterQueueStatistics
	{
		private AtomicLong _totalSendCount     = new AtomicLong();
		private AtomicLong _totalFailCount     = new AtomicLong();
		private long       _lastTotalSendCount = 0;
		private long       _lastTotalFailCount = 0;
		private int        _lastRetStatus      = 0;
		private String     _lastResponseString = "";
		
		public long incrementSendCount()
		{
			return _totalSendCount.incrementAndGet();
		}
		public long incrementFailCount()
		{
			return _totalFailCount.incrementAndGet();
		}
		public void setLastRetStatus(int status)
		{
			_lastRetStatus = status;
		}
		
		public void setLastResponceString(String responseString)
		{
			_lastResponseString = responseString;
		}

		@Override
		public String toString()
		{
			long tmpTotalSendCount = _totalSendCount.get();
			long tmpTotalFailCount = _totalFailCount.get();

			long diffTotalSendCount = tmpTotalSendCount - _lastTotalSendCount;
			long diffTotalFailCount = tmpTotalFailCount - _lastTotalFailCount;

			_lastTotalSendCount = tmpTotalSendCount;
			_lastTotalFailCount = tmpTotalFailCount;

			return "totalSendCount=" + _totalSendCount + " (" + diffTotalSendCount + "), totalFailCount=" + _totalFailCount + " (" + diffTotalFailCount + "), lastRetStatus=" + _lastRetStatus + ", _lastResponseString='" + _lastResponseString + "'";
		}
	}

	//---------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------
	/**
	 * This the class that are responsible for grabbing entries from the internal queue and SENDING information to Splunk
	 * <p>
	 * NOTE: That it can be <b>many</b> instances of this, if we are sending records to Splunk in parallel 
	 */
	private static class SplunkSender
	implements Runnable
	{
		private Thread _thread;
		private WriterToSplunk _parent;
		private int _id;

		private boolean _running;

		public String getName()
		{
			return "SplunkSender-" + _id;
		}
		
		public SplunkSender(WriterToSplunk parent, int id)
		{
			_parent = parent;
			_id     = id + 1;
		}

		public void start()
		{
			_thread = new Thread(this, getName());
			_thread.setDaemon(true);
			_thread.start();
		}
		
		public void stop()
		{
			if (_thread != null)
			{
				_thread.interrupt();
			}
		}

		public boolean isRunning()
		{
			return _running;
		}

		@Override
		public void run()
		{
			_running = true;

			_logger.info("Starting Thread for: " + getName());
			while(_running)
			{
				try
				{
					// GET A RECORD
					AuditRecord auditRecord = _parent._auditRecordQueue.take();

					// Make sure the auditRecord isn't empty.
					if (auditRecord == null || (auditRecord != null && auditRecord.isEmpty()) )
						continue;

					// APPLY FILTER
					if ( ! isIncluded(auditRecord) )
						continue;

					// if we are about to STOP the service
					if ( ! isRunning() )
					{
						_logger.info("The service is about to stop, discarding a consume(AuditRecord) queue entry.");
						continue;
					}

					//--------------------------------------------
					// Here is where it's all done
					//--------------------------------------------
					consume( auditRecord );

				}
				catch (InterruptedException ex) 
				{
					_logger.info("Received 'InterruptedException' in '" + getName() + "', time to stop...");
					_running = false;
				}
				catch(Throwable t)
				{
					_logger.error("Found some issue in '" + getName() + "', continuing with next message. Caught: " + t, t);
				}
			}
			_logger.info("Ending Thread for " + getName());
		}

		private boolean isIncluded(AuditRecord auditRecord)
		{
			return true;
		}

		private void consume(AuditRecord auditRecord)
		{
//			System.out.println("---- " + getName() + " TODO: send autidRecord to SPLUNK: " + auditRecord);
//			try
//			{
//				System.out.println("---- " + getName() + " TODO: send autidRecord to SPLUNK: " + auditRecord.toJson());
//			}
//			catch (IOException ex)
//			{
//				_logger.error("Problems sending JSON message to SPLUNK.", ex);
//			}

			
			try
			{
				StringWriter sw = new StringWriter();

				// use PersistContainer.toJsonMessage() as an template
				JsonFactory jfactory = new JsonFactory();
				JsonGenerator gen = jfactory.createGenerator(sw);
				if (_parent._jsonPrettyPrint)
					gen.setPrettyPrinter(new DefaultPrettyPrinter());
				gen.setCodec(new ObjectMapper(jfactory));


				gen.writeStartObject();

				// BEGIN: The Splunk HEC 'event' Object
				gen.writeFieldName("event");

				// Create a JSON Message that is representing the AuditRecord
				auditRecord.createJson(gen);

				// END: The Splunk HEC 'event' Object
				gen.writeEndObject();

				gen.close();
				
				// And make it to a string
				String jsonStr = sw.toString();

				// Debug print
				if (_parent._jsonPrintBeforeSend)
				{
					System.out.println(""
							+ "=================================================================\n" 
							+ "---- " + getName() + " -- Print JSON Before Send \n"
							+ "---- " + getName() + " -- To Turn this off: " + PROPKEY_printJsonBeforeSend + " = false \n"
							+ "-----------------------------------------------------------------\n" 
							+ jsonStr + "\n"
							+ "-----------------------------------------------------------------\n" 
					);
				}
				
				
				// SEND to Splunk
				//---------------------------------------------------------------
				// Using Connection Pooling to avoid  "java.net.BindException: Address already in use: JVM_Bind"
				// which we get after a while when using: try (CloseableHttpClient httpclient = HttpClients.createDefault())
				// Note 1: do NOT close the connection... Then it wont be pooled, and we are seeing "other" problems
				// Note 2: I also tried to use a regular "HttpURLConnection" but that had the same "BindException" issue
				//         probably we are running out of local ports on the client side
				CloseableHttpClient httpclient = HttpClients.custom().setConnectionManager(_parent._apacheHttpConnectionManager).build(); 
				HttpUriRequest request = RequestBuilder.post()
						.setUri(_parent._splunkUrl)
						.setHeader(HttpHeaders.CONTENT_TYPE, "application/json")
						.setHeader(HttpHeaders.AUTHORIZATION, _parent._splunkAccessToken)
						.setCharset(StandardCharsets.UTF_8)
						.setEntity(new StringEntity(jsonStr, StandardCharsets.UTF_8)) // <<<--- add request body
						.build();

				
				// Executing POST request...
				HttpResponse response = httpclient.execute(request);

				// Get status code
				int statusCode = response.getStatusLine().getStatusCode();
//				System.out.println("Status code: " + statusCode);
				if (statusCode != 200)
				{
					//String statusCodeStr = org.apache.commons.httpclient.HttpStatus.getStatusText(statusCode); // This is not available in classpath
					
					int numberOfChars = _parent.getConfig().getIntProperty(PROPKEY_json_onErrorPrintXchars, DEFAULT_json_onErrorPrintXchars);
					String jsonMessageFirstXChars = jsonStr.substring(0, Math.min(numberOfChars, jsonStr.length())).replace('\n', ' ').replace('\r', ' ') + "...";
					_logger.error("HTTP Response code from Splunk was " + statusCode + ", while 200 was expected. Extra Info: Splunk URL='" + _parent._splunkUrl + "', AccessToken='" + _parent._splunkAccessToken + "', jsonMessageFirstXChars='" + jsonMessageFirstXChars + "'.");

					// TODO: Handle errors (meaning should we save the JSON String and re-send it later)
				}

				_parent._stats.setLastRetStatus(statusCode);


				// and Response String
				String responseString = new BasicResponseHandler().handleResponse(response);
//				System.out.println("Response: " + responseString);
				if (responseString != null && !responseString.contains("\"Success\""))
				{
					_logger.error(getName() + ": Unexpected responce string '" + responseString + "'.");
					// TODO: Handle errors
				}
				_parent._stats.setLastResponceString(responseString);

				_parent._stats.incrementSendCount();
				
//-------------------------------------------------
// BEGIN: do not use old/test code that is NOT POOLED
//-------------------------------------------------
//				try (CloseableHttpClient httpclient = HttpClients.createDefault())
//				{
//					HttpPost httpPost = new HttpPost(_parent._splunkUrl);
//					httpPost.addHeader(HttpHeaders.CONTENT_TYPE, "application/json");
//					httpPost.addHeader(HttpHeaders.AUTHORIZATION, _parent._splunkAccessToken);
//					
//					httpPost.setEntity(new StringEntity(jsonStr, StandardCharsets.UTF_8));
//					
//					// Executing POST request...
//					HttpResponse response = httpclient.execute(httpPost);
//
//					// Get status code
//					int statusCode = response.getStatusLine().getStatusCode();
////					System.out.println("Status code: " + statusCode);
//					if (statusCode != 200)
//					{
//						_logger.error("HTTP Response code from Splunk at '" + _parent._splunkUrl + "' was " + statusCode + ", while 200 was expected.");
//					}
//					else
//					{
//						// TODO: Handle errors
//					}
//					_parent._stats.setLastRetStatus(statusCode);
//
//
//					// and Response String
//					String responseString = new BasicResponseHandler().handleResponse(response);
////					System.out.println("Response: " + responseString);
//					if (responseString != null && !responseString.contains("\"Success\""))
//					{
//						_logger.error(getName() + ": Unexpected responce string '" + responseString + "'.");
//						// TODO: Handle errors
//					}
//					_parent._stats.setLastResponceString(responseString);
//
//					_parent._stats.incrementSendCount();
//				}
//-------------------------------------------------
// END: do not use old/test code that is NOT POOLED
//-------------------------------------------------
			}
			catch (IOException ex)
			{
				_logger.error("Problems sending JSON message to SPLUNK at '" + _parent._splunkUrl + "'.", ex);
				
				_parent._stats.incrementFailCount();
				
				// TODO: If Splunk is DOWN: Should we save the content of the message we failed to send, and send it LATER on
			}
			
			
			
//			try { Thread.sleep(3);}
//			catch (InterruptedException ignore) {}
		}
	}
}
