/*
** rmp1510.sql
**
** RMP Version: 15.6.0
** Revision:    1
** RS Version:  15.1.0 - The Rep Server version associated with the scripts
** Date:        08/19/2010
**
** This script contains stored procedures used by the RMP to query information
** from the RSSD. The script is loaded into the Rep Server's RSSD by the RMP.
** This is one in a set of files used by the RMP. There is one file for each
** version of the Rep Server supported by the RMP. The file name identifies the
** version of the Rep Server (i.e. rmp1510.sql contains stored procedures for
** Rep Server version 15.1.0).
**
** ******************************* WARNING ************************************
**
** Do not attempt to change the text, variable names, or formatting of the
** following stored procedure. Any modifications might result in the server
** being unable to parse the file or the server incorrectly identifying the
** version numbers.
**
** ******************************* WARNING ************************************
*/

/*
** RMP_PROC_VERSION
**      This procedure identifies the version of the RMP and Rep Server that
**      is associated with the stored procedures in the RSSD. If the values
**      in the RMP and Rep Server do not match the values returned by this
**      procedure, then the stored procedures are out-of-date and must be
**      upgraded.
**
**	This is the dummy procedure to be used at the beginning of the stored
**      proc scripts. The actual stored proc that returns the valid values is
**      at the end of the script. If RMP runs into errors creating any stored
**      proc in this script, RMP will abort the attempt. Any later checks for
**      this stored proc will result in finding the invalid version numbers and
**      the RSSD procedures will be marked invalid If all sprocs are created
**      successfully, the actual stored proc will be created at the end. This
**      dummy stored procedure should never be changed unless the number of
**      columns it returns changes.
*/

if exists (
	select * from sysobjects
	where name = 'rmp_proc_version' and type = 'P')
begin
	drop procedure rmp_proc_version
end
go
create procedure rmp_proc_version
as
	select "0", "0", "0"
go

/********************************* Utilities *********************************/

/*
** RMP_REVERSE_BYTES
**
** Reverse the order of the bytes in the input parameter bytestr and
** return it in the revstr parameter.  This routine is used instead of
** the reverse function.  The reverse function does not work correctly
** in multi-byte character sets.  It reverses characters, not bytes.
*/

if exists (
	select * from sysobjects
	where name = 'rmp_reverse_bytes' and type = 'P')
begin
	drop procedure rmp_reverse_bytes
end
go
create procedure rmp_reverse_bytes
    @bytestr    varbinary(255),
    @start      int,
    @length     int,
	@revstr		varbinary(255) output
as

declare @end    int
declare @ind    int

select @end = @start + @length - 1
select @revstr = substring(@bytestr, @end, 1)
select @ind = @end - 1

while (@ind >= @start)
begin
    select @revstr = @revstr + substring(@bytestr, @ind, 1)
    select @ind = @ind - 1
end
go

/*
** RMP_FIND_CONFIG_VALUE
**
** Figure out proper "run-time" configuration parameter
**
** db_packet_size
** dsi_check_lock_wait
** dsi_commit_control
** dsi_commit_wait_time
** dsi_exec_request_sproc
** dsi_ignore_underscore_name
** dsi_max_xacts_in_group
** dsi_num_retries
** dsi_partitioning_rule
** dsi_replication
** dsi_text_convert_multiplier
** exec_cmds_per_timeslice
** exec_sqm_write_request_limit
** md_sqm_write_request_limit
** sub_sqm_write_request_limit
*/
if exists (
	select * from sysobjects
	where name ='rmp_find_config_value' and type ='P')
begin
	drop procedure rmp_find_config_value
end
go
create procedure rmp_find_config_value
	@objid		rs_id,
	@configparam	varchar(30),
	@result		varchar(255) output
as

/*
** First check to see if the value actually exists.  Some parameters are
** simply not in either the rs_result or rs_tvalues tables.
*/
if exists (select status from rs_tvalues where name = @configparam) or
   exists (select * from rs_config where optionname = @configparam)
   begin
      /*
      ** Okay, the parameter exists.  First see if we can find a match by
      ** object id (@objid) if it was provided.  If not then we can simply
      ** try and find the best fit.  The one in the rs_tvalues table is
      ** the currently running value IFF it was changed.  In that case the
      ** one in the rs_config table is the new value which will take place
      ** the next time the server restarts.
      */
      select @result = NULL
      if (@objid != NULL)
         select @result   = (select charvalue from rs_config
			     where optionname = @configparam
			     and objid = @objid)

      if (@result = NULL)
         if ((select status from rs_tvalues where name = @configparam) = 1)
           select @result = (select cv from rs_tvalues
			     where name = @configparam)
         else
	    select @result = (select charvalue from rs_config
			      where optionname = @configparam
			      and objid = 0)
      if (@result = NULL)
	 if ((select status from rs_tvalues where name = @configparam) = 0)
	   select @result = (select dv from rs_tvalues
			     where name = @configparam)
   end
else
   begin
      /*
      ** Okay, so the parameter is not in the tables which means it has never
      ** been set to something other than the default.  Simply return the
      ** default value here.
      */
      if      (@configparam = "dsi_keep_triggers")
   	select @result      = "on"
      else if (@configparam = "dsi_large_xact_size")
   	select @result      = "100"
      else if (@configparam = "dsi_sqt_max_cache_size")
   	select @result      = "0"
      else if (@configparam = "dsi_max_xacts_in_group")
   	select @result      = "20"
      else if (@configparam = "dsi_replication")
   	select @result      = "on"
      else if (@configparam = "dsi_sql_data_style")
   	select @result      = NULL
      else if (@configparam = "parallel_dsi")
   	select @result      = "off"
      else if (@configparam = "batch_begin")
   	select @result      = "on"
      else if (@configparam = "dsi_exec_request_sproc")
   	select @result      = "on"
      else if (@configparam = "db_packet_size")
   	select @result      = "512"
      else if (@configparam = "unified_login")
   	select @result      = "not_required"
      else if (@configparam = "mutual_auth")
   	select @result      = "not_required"
      else if (@configparam = "msg_confidentiality")
   	select @result      = "not_required"
      else if (@configparam = "msg_integrity")
   	select @result      = "not_required"
      else if (@configparam = "msg_origin_check")
   	select @result      = "not_required"
      else if (@configparam = "msg_replay_detection")
   	select @result      = "not_required"
      else if (@configparam = "msg_sequence_check")
   	select @result      = "not_required"
      else if (@configparam = "security_mechanism")
   	select @result      = ""
      else if (@configparam = "rsi_xact_with_large_msg")
   	select @result      = "shutdown"
      else if (@configparam = "dsi_check_lock_wait")
   	select @result      = "3000"
      else if (@configparam = "dsi_commit_control")
   	select @result      = "on"
      else if (@configparam = "dsi_commit_wait_time")
   	select @result      = "500"
      else if (@configparam = "dsi_ignore_underscore_name")
   	select @result      = "on"
      else if (@configparam = "dsi_num_retries")
   	select @result      = "0"
      else if (@configparam = "dsi_partitioning_rule")
   	select @result      = "none"
      else if (@configparam = "dsi_text_convert_multiplier")
   	select @result      = "1.0"
      else if (@configparam = "exec_cmds_per_timeslice")
   	select @result      = "5"
      else if (@configparam = "exec_sqm_write_request_limit")
   	select @result      = "16384"
      else if (@configparam = "md_sqm_write_request_limit")
   	select @result      = "100000"
      else if (@configparam = "sub_sqm_write_request_limit")
   	select @result      = "16384"
      else
        begin
	   print "Unknown parameter type!!!!! - "
	   print @configparam
	end
    end
go

/*
** RMP_FIND_SECURITY_CONFIG
**
** Figure out proper "run-time" security configuration. This stored proc
** combines the various security options into one bit mask and returns the
** result as an integer for the given ObjectID. It also returns a value for
** use_security (on, off or default) based on whether any security settings
** have been specified for the given object or not.
*/
if exists (
	select * from sysobjects
	where name = 'rmp_find_security_config' and type ='P')
begin
	drop procedure rmp_find_security_config
end
go
create procedure rmp_find_security_config
	@objid		rs_id,
	@result		int output,
	@use_sec	varchar(10) output
as

declare @val	char(20)
declare @option	int

select @result = 0
execute rmp_find_config_value @objid, "unified_login", @result = @val output
if (@val = "required")
	select @result = @result + 1
execute rmp_find_config_value @objid, "mutual_auth", @result = @val output
if (@val = "required")
	select @result = @result + 2
execute rmp_find_config_value @objid, "msg_confidentiality", @result = @val output
if (@val = "required")
	select @result = @result + 4
execute rmp_find_config_value @objid, "msg_integrity", @result = @val output
if (@val = "required")
	select @result = @result + 8
execute rmp_find_config_value @objid, "msg_origin_check", @result = @val output
if (@val = "required")
	select @result = @result + 16
execute rmp_find_config_value @objid, "msg_replay_detection", @result = @val output
if (@val = "required")
	select @result = @result + 32
execute rmp_find_config_value @objid, "msg_sequence_check", @result = @val output
if (@val = "required")
	select @result = @result + 64

if not exists (select charvalue from rs_config where optionname in
	('unified_login', 'mutual_auth', 'msg_confidentiality',
	'msg_integrity', 'msg_origin_check', 'msg_replay_detection',
	'msg_sequence_check', 'security_mechanism') and objid = @objid)
begin
	/* None of the settings have been specified. Using Global default. */
	select @use_sec = "default"
end
else if (@result > 0)
begin
	/* At least one of the settings has been set. */
	select @use_sec = "on"
end
else
begin
	select @use_sec = "off"
end

go

/******************************** Connections ********************************/

/*
** RMP_CONNECTIONS
**
** Get all connections for the specified Replication Server.
**
** @servername - name of the Replication Server
** @ds - name of the data server or '*' for all servers
** @db - name of the database or '*' for all databases
*/
if exists (
	select * from sysobjects
	where name = 'rmp_connections' and type = 'P')
begin
	drop procedure rmp_connections
end
go
create procedure rmp_connections
	@servername varchar(255),
	@ds varchar(255),
	@db varchar(255)
as
select si.name, db.dsname, db.dbname, db.src_status, db.rowtype, db.ptype,
        ldb.dsname as "ldsname", ldb.dbname as "ldbname"
from rs_sites si, rs_databases db, rs_databases ldb
where si.name = @servername
        and db.prsid = si.id
        and db.ltype = 'P'
        and ldb.dbid =* db.ldbid
        and ldb.ltype = 'L'
        and (@ds = '*' or @ds = db.dsname)
        and (@db = '*' or @db = db.dbname)
order by db.dsname asc
go

/*
** RMP_ALL_CONNECTIONS
**
** Get all connections known to the replication server.
*/
if exists (
	select * from sysobjects
	where name = 'rmp_all_connections' and type = 'P')
begin
	drop procedure rmp_all_connections
end
go
create procedure rmp_all_connections
as
select si.name, db.dsname, db.dbname, db.src_status, db.rowtype, db.ptype,
        ldb.dsname as "ldsname", ldb.dbname as "ldbname"
from rs_sites si, rs_databases db, rs_databases ldb
where db.prsid = si.id
        and db.ltype = 'P'
        and ldb.dbid =* db.ldbid
        and ldb.ltype = 'L'
order by db.dsname asc
go

/*
** RMP_CONNECTION_DETAILS
**
** Get connections details - function string class, error class, maintenance
** userid and password.
**
** @ds - Data Server
** @db - Database
*/
if exists (
	select * from sysobjects
	where name = 'rmp_connection_details' and type = 'P')
begin
	drop procedure rmp_connection_details
end
go
create procedure rmp_connection_details
	@ds varchar(255),
	@db varchar(255)
as
select db.dsname, db.dbname, e.classname as "error class",
        f.classname as "function string class", mu.username, mu.password
from rs_databases db, rs_maintusers mu, rs_classes e, rs_classes f
where db.dbid = mu.destid
        and db.dsname = @ds
        and db.dbname = @db
        and e.classid = db.errorclassid and e.classtype = 'E'
        and f.classid = db.funcclassid and f.classtype = 'F'
order by db.dsname asc
go

/*
** RMP_CONNECTION_DEMAND
**
** Get on-demand database connection information for one connection.
**
** db_packet_size
** disk_affinity
** dsi_check_lock_wait
** dsi_commit_control
** dsi_commit_wait_time
** dsi_exec_request_sproc
** dsi_ignore_underscore_name
** dsi_max_xacts_in_group
** dsi_num_retries
** dsi_partitioning_rule
** dsi_replication
** dsi_text_convert_multiplier
** exec_cmds_per_timeslice
** exec_sqm_write_request_limit
** md_sqm_write_request_limit
** sub_sqm_write_request_limit
*/
if exists (
	select * from sysobjects
	where name = 'rmp_connection_demand' and type = 'P')
begin
	drop procedure rmp_connection_demand
end
go
create procedure rmp_connection_demand
	@repname varchar(255),
	@sqlname varchar(255),
	@dbname  varchar(255)
as

declare	@dbid		rs_id
declare @ldbid		rs_id
declare @rawid		rs_id

select @rawid = (select rdb.dbid
from rs_databases rdb, rs_sites rsi
where rsi.name = @repname
        and rdb.prsid = rsi.id
        and rdb.ltype = 'P'
        and rdb.dsname = @sqlname
        and rdb.dbname = @dbname)

/* If this is a byte-swap machine, adjust rs_id */
if (convert(int, 0x00000100) = 65536)
	execute rmp_reverse_bytes @rawid, 1, 4, @dbid output
else
	select @dbid = @rawid

/* For save_interval, select the logical dbid */
select @rawid = (select rdb.ldbid
                 from rs_databases rdb, rs_sites rsi
                 where rsi.name = @repname
                        and rdb.prsid = rsi.id
                        and rdb.ltype = 'P'
                        and rdb.dsname = @sqlname
                        and rdb.dbname = @dbname)

/* If this a byte-swap machine, adjust rs_id */
if (convert(int, 0x00000100) = 65536)
	execute rmp_reverse_bytes @rawid, 1, 4, @ldbid output
else
	select @ldbid = @rawid

/*
** Setup the return values.  We use variables since the executes can't
** be done as a select.
*/

declare @stat_dist_status		tinyint
declare @stat_src_status		tinyint

select 	@stat_dist_status = r1.dist_status,
	@stat_src_status = r1.src_status
from rs_databases r1
where r1.dbid = @rawid

declare @config_batch                   varchar(8)
declare @config_batch_begin             varchar(8)
declare @config_command_retry           varchar(8)
declare @config_dsi_charset_convert     varchar(8)
declare @config_dsi_cmd_batch_size      varchar(16)
declare @config_dsi_cmd_sep             varchar(8)
declare @config_dsi_fadeout_time        varchar(8)
declare @config_dsi_keep_triggers       varchar(8)
declare @config_dsi_large_xact_size     varchar(8)
declare @config_dsi_max_cmds_to_log     varchar(16)
declare @config_dsi_max_text_to_log     varchar(16)
declare @config_dsi_num_lxact_threads   varchar(8)
declare @config_dsi_num_threads         varchar(16)
declare @config_dsi_replication         varchar(8)
declare @config_dsi_serial_method       varchar(32)
declare @config_dsi_sql_data_style      varchar(16)
declare @config_dsi_sqt_max_cache_size  varchar(8)
declare @config_dsi_xact_group_size     varchar(8)
declare @config_dsi_max_xacts_in_group	varchar(8)
declare @config_dsi_exec_request_sproc  varchar(8)
declare @config_db_packet_size          varchar(8)
declare @config_Dump_Load               varchar(8)
declare @config_parallel_dsi            varchar(8)
declare @config_Save_interval           varchar(8)
declare @config_sec_mechanism           varchar(30)
declare @config_sec_options             int
declare @config_sec_use             	varchar(10)
declare @config_disk_affinity	        varchar(32)
declare @config_dsi_check_lock_wait	varchar(16)
declare @config_dsi_commit_control	varchar(8)
declare @config_dsi_commit_wait_time	varchar(16)
declare @config_dsi_ignore_underscore	varchar(8)
declare @config_dsi_num_retries		varchar(16)
declare @config_dsi_partitioning_rule	varchar(32)
declare @config_dsi_text_multiplier	varchar(16)
declare @config_exec_cmds_timeslice	varchar(16)
declare @config_exec_sqm_limit		varchar(16)
declare @config_md_sqm_limit		varchar(16)
declare @config_sub_sqm_limit		varchar(16)

execute rmp_find_config_value @dbid, "batch",
                               @result = @config_batch output
execute rmp_find_config_value @dbid, "batch_begin",
                               @result = @config_batch_begin output
execute rmp_find_config_value @dbid, "command_retry",
                               @result = @config_command_retry output
execute rmp_find_config_value @dbid, "dsi_charset_convert",
                               @result = @config_dsi_charset_convert output
execute rmp_find_config_value @dbid, "dsi_cmd_batch_size",
                               @result = @config_dsi_cmd_batch_size output
execute rmp_find_config_value @dbid, "dsi_cmd_separator",
                               @result = @config_dsi_cmd_sep output
execute rmp_find_config_value @dbid, "dsi_fadeout_time",
                               @result = @config_dsi_fadeout_time output
execute rmp_find_config_value @dbid, "dsi_keep_triggers",
                               @result = @config_dsi_keep_triggers output
execute rmp_find_config_value @dbid, "dsi_large_xact_size",
                               @result = @config_dsi_large_xact_size output
execute rmp_find_config_value @dbid, "dsi_max_cmds_to_log",
                               @result = @config_dsi_max_cmds_to_log output
execute rmp_find_config_value @dbid, "dsi_max_text_to_log",
                               @result = @config_dsi_max_text_to_log output
execute rmp_find_config_value @dbid, "dsi_num_large_xact_threads",
                               @result = @config_dsi_num_lxact_threads output
execute rmp_find_config_value @dbid, "dsi_num_threads",
                               @result = @config_dsi_num_threads output
execute rmp_find_config_value @dbid, "dsi_replication",
                               @result = @config_dsi_replication output
execute rmp_find_config_value @dbid, "dsi_serialization_method",
                               @result = @config_dsi_serial_method output
execute rmp_find_config_value @dbid, "dsi_sql_data_style",
                               @result = @config_dsi_sql_data_style output
execute rmp_find_config_value @dbid, "dsi_sqt_max_cache_size",
                               @result = @config_dsi_sqt_max_cache_size output
execute rmp_find_config_value @dbid, "dsi_xact_group_size",
                               @result = @config_dsi_xact_group_size output
execute rmp_find_config_value @dbid, "dsi_max_xacts_in_group",
                               @result = @config_dsi_max_xacts_in_group output
execute rmp_find_config_value @dbid, "dsi_exec_request_sproc",
                               @result = @config_dsi_exec_request_sproc output
execute rmp_find_config_value @dbid, "db_packet_size",
                               @result = @config_db_packet_size output
execute rmp_find_config_value @dbid, "dump_load",
                               @result = @config_Dump_Load output
execute rmp_find_config_value @dbid, "parallel_dsi",
                               @result = @config_parallel_dsi output
execute rmp_find_config_value @ldbid, "save_interval",
                               @result = @config_Save_interval output
execute rmp_find_config_value @dbid, "security_mechanism",
                               @result = @config_sec_mechanism output
execute rmp_find_security_config @dbid, @result = @config_sec_options output,
			        @use_sec = @config_sec_use output
execute rmp_find_config_value @dbid, "dsi_check_lock_wait",
		                @result = @config_dsi_check_lock_wait output
execute rmp_find_config_value @dbid, "dsi_commit_control",
			        @result = @config_dsi_commit_control output
execute rmp_find_config_value @dbid, "dsi_commit_wait_time",
				@result = @config_dsi_commit_wait_time output
execute rmp_find_config_value @dbid, "dsi_ignore_underscore_name",
				@result = @config_dsi_ignore_underscore output
execute rmp_find_config_value @dbid, "dsi_num_retries",
				@result = @config_dsi_num_retries output
execute rmp_find_config_value @dbid, "dsi_partitioning_rule",
				@result = @config_dsi_partitioning_rule output
execute rmp_find_config_value @dbid, "dsi_text_convert_multiplier",
				@result = @config_dsi_text_multiplier output
execute rmp_find_config_value @dbid, "exec_cmds_per_timeslice",
				@result = @config_exec_cmds_timeslice output
execute rmp_find_config_value @dbid, "exec_sqm_write_request_limit",
				@result = @config_exec_sqm_limit output
execute rmp_find_config_value @dbid, "md_sqm_write_request_limit",
				@result = @config_md_sqm_limit output
execute rmp_find_config_value @dbid, "sub_sqm_write_request_limit",
				@result = @config_sub_sqm_limit output

/*
** Disk Affinity is not stored with the rest of the configuration parameters.
** The disk affinity value is stored in rs_diskaffinity and is identified
** by the dbid.
*/

select @config_disk_affinity = (select part.logical_name
		from rs_diskpartitions part, rs_diskaffinity da
		where da.dbid_or_siteid = @dbid
                        and da.partition_id = part.id
                        and part.status = 1
                        and (da.status  & 1) = 1)
if (@config_disk_affinity = NULL)
   	select @config_disk_affinity = "off"

/*
** Now return the results to our caller.
*/

select	"dist_status" =			@stat_dist_status,
	"src_status" = 			@stat_src_status,
	"batch" = 			@config_batch,
	"batch_begin" = 		@config_batch_begin,
	"command_retry" = 		@config_command_retry,
	"dsi_charset_convert" =		@config_dsi_charset_convert,
	"dsi_cmd_batch_size" =		@config_dsi_cmd_batch_size,
	"dsi_cmd_separator" =		@config_dsi_cmd_sep,
	"dsi_fadeout_time" =		@config_dsi_fadeout_time,
	"dsi_keep_triggers" =		@config_dsi_keep_triggers,
	"dsi_large_xact_size" =		@config_dsi_large_xact_size,
	"dsi_max_cmds_to_log" =		@config_dsi_max_cmds_to_log,
	"dsi_max_text_to_log" =		@config_dsi_max_text_to_log,
	"dsi_num_large_xact_threads" =	@config_dsi_num_lxact_threads,
	"dsi_num_threads" =		@config_dsi_num_threads,
	"dsi_replication" =		@config_dsi_replication,
	"dsi_serialization_method" =	@config_dsi_serial_method,
	"dsi_sql_data_style" =		@config_dsi_sql_data_style,
	"dsi_sqt_max_cache_size" =	@config_dsi_sqt_max_cache_size,
	"dsi_xact_group_size" =		@config_dsi_xact_group_size,
	"dsi_max_xacts_in_group" =	@config_dsi_max_xacts_in_group,
	"dsi_exec_request_sproc" = 	@config_dsi_exec_request_sproc,
	"db_packet_size" =		@config_db_packet_size,
	"dump_load" =			@config_Dump_Load,
	"parallel_dsi" =		@config_parallel_dsi,
	"save_interval" =		@config_Save_interval,
	"use_security" =		@config_sec_use,
	"sec_mechanism" =		@config_sec_mechanism,
	"sec_options" =			@config_sec_options,
	"disk_affinity" =		@config_disk_affinity,
	"dsi_check_lock_wait" =		@config_dsi_check_lock_wait,
	"dsi_commit_control" =		@config_dsi_commit_control,
	"dsi_commit_wait_time" =	@config_dsi_commit_wait_time,
	"dsi_ignore_underscore_name" =	@config_dsi_ignore_underscore,
	"dsi_num_retries" =		@config_dsi_num_retries,
	"dsi_partitioning_rule" =	@config_dsi_partitioning_rule,
	"dsi_text_convert_multiplier" =	@config_dsi_text_multiplier,
	"exec_cmds_per_timeslice" =	@config_exec_cmds_timeslice,
	"exec_sqm_write_request_limit" = @config_exec_sqm_limit,
	"md_sqm_write_request_limit" =	@config_md_sqm_limit,
	"sub_sqm_write_request_limit" =	@config_sub_sqm_limit
go

/******************************* Exceptions Log ******************************/
/*
** RMP_EXCEPTIONS_LOG
**
** Retrieve the exceptions log
** @ds - Data Server
** @db - Database
** @originds - Origin Data Server
** @origindb - Origin Database
** @hourFilter - Hours used to filter the log
*/
if exists (
	select * from sysobjects
	where name = 'rmp_exceptions_log' and type = 'P')
begin
	drop procedure rmp_exceptions_log
end
go
create procedure rmp_exceptions_log
	@ds varchar(255),
	@db varchar(255),
	@originds varchar(255),
	@origindb varchar(255),
	@hourFilter int
as
declare @idx int
if (convert(int, 0x00000100) = 65536)
        select @idx = 8
else
        select @idx = 5

select 'tran_id' = convert(int, substring(sys_trans_id, @idx, 8)),
        sys_trans_id, app_trans_name,
        orig_site, orig_db, orig_time, orig_user,
        log_time, log_reason, ds_error, ds_errmsg, trans_status,
        error_proc, error_src_line, error_output_line,
        app_usr, app_pwd
from rs_exceptshdr
where error_site = @ds
        and error_db = @db
        and (@originds = '' or orig_site like @originds)
        and (@origindb = '' or orig_db like @origindb)
        and (@hourFilter = 0 or datediff(hh, log_time, getdate()) <= @hourFilter)
order by tran_id
go

/*
** RMP_EXCEPTIONS_LOG_LANG
**
** Retrieve the exception log language command.
** @tranID - Transaction ID
*/
if exists (
	select * from sysobjects
	where name = 'rmp_exceptions_log_lang' and type = 'P')
begin
	drop procedure rmp_exceptions_log_lang
end
go
create procedure rmp_exceptions_log_lang
        @tranID rs_id
as
select t.sequence, t.textval
from rs_exceptscmd e, rs_systext t
where e.sys_trans_id = @tranID
        and e.cmd_id = t.parentid
        and (e.cmd_type = 'L' or e.cmd_type = 'R')
order by e.src_cmd_line asc, e.output_cmd_index asc, t.sequence asc
go

/*
** RMP_EXCEPTIONS_LOG_SRC
**
** Retrieve the exception log source command.
** @tranID - Transaction ID
*/
if exists (
	select * from sysobjects
	where name = 'rmp_exceptions_log_src' and type = 'P')
begin
	drop procedure rmp_exceptions_log_src
end
go
create procedure rmp_exceptions_log_src
        @tranID rs_id
as
select t.sequence, t.textval
from rs_exceptscmd e, rs_systext t
where e.sys_trans_id = @tranID
        and e.cmd_id = t.parentid
        and (e.cmd_type = 'S')
order by e.src_cmd_line asc, e.output_cmd_index asc, t.sequence asc
go

/*
** RMP_DELEXCEPTION
**
** Delete exceptions from the exceptions log.
** @trid - Transaction ID
*/
if exists(
	select name from sysobjects
	where name = 'rmp_delexception')
begin
	drop procedure rmp_delexception
end

go
create procedure rmp_delexception
	@trid varchar(30) = NULL

as

declare @xact int
declare @systran binary(8)
declare @cnt int, @err int
declare @ccnt char(8)
declare @rsname varchar(30)
declare @cmdcount char(9)

select @xact = NULL
set nocount on

/* convert logged tran # to print it */

if (@trid is NULL)
	select @xact = NULL
else
	select @xact = convert(int, @trid)

/* find RS name */

select @rsname = charvalue
from   rs_config
where  optionname = "oserver"

/* Build temp table #tab1 */

create table #tab1
(	orig_site varchar(255),
        orig_db   varchar(255),
        orig_user varchar(255),
        orig_time datetime,
        error_site varchar(255),
        error_db varchar(255),
        log_time datetime,
	reccount int null,
	sys_trans_id binary(8),
	app_usr varchar(255)
)
insert into #tab1 (orig_site, orig_db,
		   orig_user, orig_time,
		   error_site, error_db,
		   log_time, sys_trans_id,
		   app_usr)
select
	orig_site,
	orig_db,
	orig_user,
	orig_time,
      	error_site,
	error_db,
	log_time,
	sys_trans_id,
	app_usr
from	rs_exceptshdr exh

/* add logged command counts to table */

update #tab1
set reccount = (select max(src_cmd_line)
                from rs_exceptscmd exc
		where exc.sys_trans_id = #tab1.sys_trans_id
		group by exc.sys_trans_id)


/* If no parameters print summary */

if (@xact is NULL)
begin

select @cnt = count(sys_trans_id)
from rs_exceptshdr

select @ccnt = convert(char(8), @cnt)

if (@cnt = 0)
begin

print " "
print "         There are 0 Logged Transactions."
print " "
return 0
end

print " "
print "		Summary of Logged Transactions on '%1!'", @rsname
print " "
print " "

if (convert(int, 0x0000100) = 65536)
begin
  select
	"Xact ID" = convert(int, (substring(sys_trans_id, 12, 1) +
				  substring(sys_trans_id, 11, 1) +
				  substring(sys_trans_id, 10, 1) +
				  substring(sys_trans_id, 9, 1) +
				  substring(sys_trans_id, 8, 1) +
				  substring(sys_trans_id, 7, 1) +
				  substring(sys_trans_id, 6, 1) +
				  substring(sys_trans_id, 5, 1))),
	"Org Site" = substring(rtrim(orig_site)+"."+rtrim(orig_db), 1, 15),
	"Org User" = substring(orig_user, 1, 8),
	"Org Date " = convert(char(11), orig_time),
      	"Dest Site" = substring(rtrim(error_site)+"."+rtrim(error_db), 1, 15),
	"# Recs/Xact" = reccount
  from	#tab1
end
else
begin
  select
	"Xact ID" = convert(int, substring(sys_trans_id, 5, 8)),
	"Org Site" = substring(rtrim(orig_site)+"."+rtrim(orig_db), 1, 15),
	"Org User" = substring(orig_user, 1, 8),
	"Org Date " = convert(char(11), orig_time),
      	"Dest Site" = substring(rtrim(error_site)+"."+rtrim(error_db), 1, 15),
	"# Recs/Xact" = reccount
  from	#tab1
end

print " "
print " "
print "		To Delete a Specific Logged Xact., type"
print " "
print "			'rs_delexception {Xact ID}'"
print " "


end


if (@xact IS NOT NULL)
begin

/* Check if @xact exists and figure out right sys_trans_id */

if (convert(int, 0x0000100) = 65536)
begin
  select @systran = sys_trans_id
  from #tab1
  where convert(int, (substring(sys_trans_id, 12, 1) +
        		  substring(sys_trans_id, 11, 1) +
        		  substring(sys_trans_id, 10, 1) +
        		  substring(sys_trans_id, 9, 1) +
        		  substring(sys_trans_id, 8, 1) +
        		  substring(sys_trans_id, 7, 1) +
        		  substring(sys_trans_id, 6, 1) +
        		  substring(sys_trans_id, 5, 1))) = @xact
end
else
begin
  select @systran = sys_trans_id
  from #tab1
  where convert(int, substring(sys_trans_id, 5, 8)) = @xact
end

if (@@rowcount = 0)
begin

print " "
print "  Transaction ID %1! Not Found. Run 'rs_helpexception' for a Complete List.",
	@xact
print " "
return -1
end

/* if logged transaction exists, delete it. */

select @cmdcount = rtrim(convert(char(9), reccount))
from   #tab1
where  sys_trans_id = @systran

print " "
print "	Deleting %1! Commands in Logged Transaction # %2! on '%3!'",
        @cmdcount, @xact, @rsname
print " "
print " "

begin transaction

delete  rs_systext
from    rs_exceptscmd exc,
        rs_exceptshdr exh,
        rs_systext sys
where   exc.sys_trans_id = exh.sys_trans_id
and     exc.cmd_id = sys.parentid
and 	sys.texttype = "C"
and     exh.sys_trans_id  =  @systran

select @err = @@error

if (@err != 0)
begin
print "Deleting rs_systext table failed. Transaction Rolled Back."
rollback transaction
return @err
end


delete  rs_exceptscmd
where   sys_trans_id  =  @systran

select @err = @@error

if (@err != 0)
begin

print "Deleting rs_exceptscmd table failed. Transaction Rolled Back."
rollback transaction
return @err
end

delete  rs_exceptshdr
where   sys_trans_id  =  @systran

select @err = @@error

if (@err != 0)
begin

print "Deleting rs_exceptshdr table failed. Transaction Rolled Back."
rollback transaction
return @err
end

commit transaction

select @err = @@error

if (@err !=0)
begin

print "Executing 'commit transaction'  failed. Transaction Rolled Back."
rollback transaction
return @err
end

else

begin

print " "
print "		Logged Transaction # %1! Successfully Deleted. ", @xact
print " "
print "		Truncate RSSD Transaction Log if Necessary."
print " "
end
end
go

/****************************** Database Rep Def *****************************/

/*
** RMP_ALL_DB_REPDEF
**
** Get all database rep defs in the domain (at a specific rep server)
*/

if exists (
	select * from sysobjects
	where name = 'rmp_all_db_repdef' and type = 'P')
begin
	drop procedure rmp_all_db_repdef
end
go
create procedure rmp_all_db_repdef
as
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers
from rs_dbreps dbrep, rs_databases db, rs_databases ldb, rs_sites rs
where dbrep.dbid = db.dbid
        and dbrep.prsid = rs.id
        and db.ltype = 'P' and db.ldbid *= ldb.dbid
union
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers
from rs_dbreps dbrep, rs_databases db, rs_sites rs
where dbrep.dbid = db.dbid
        and dbrep.prsid = rs.id
        and db.ltype = 'L'
order by 1 asc
go

/*
** RMP_DB_REPDEF
**
** Get all database rep defs for a specific replication server
** @rs - Replication Server
*/

if exists (
	select * from sysobjects
	where name = 'rmp_db_repdef' and type = 'P')
begin
	drop procedure rmp_db_repdef
end
go
create procedure rmp_db_repdef
        @rs varchar(255)
as
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers
from rs_dbreps dbrep, rs_databases db, rs_databases ldb, rs_sites rs
where dbrep.dbid = db.dbid
        and dbrep.prsid = rs.id
        and rs.name = @rs
        and db.ltype = 'P' and db.ldbid *= ldb.dbid
union
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers
from rs_dbreps dbrep, rs_databases db, rs_sites rs
where dbrep.dbid = db.dbid
        and dbrep.prsid = rs.id
        and rs.name = @rs
        and db.ltype = 'L'
order by 1 asc
go

/*
** RMP_DB_REPDEF_FOR_CONN
**
** Get all database rep defs for a specific connection
** @ds - Data Server
** @db - Database
*/

if exists (
	select * from sysobjects
	where name = 'rmp_db_repdef_for_conn' and type = 'P')
begin
	drop procedure rmp_db_repdef_for_conn
end
go
create procedure rmp_db_repdef_for_conn
        @ds varchar(255),
        @db varchar(255)
as
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers
from rs_dbreps dbrep, rs_databases db, rs_databases ldb, rs_sites rs
where dbrep.dbid = db.dbid
        and dbrep.prsid = rs.id
        and db.dsname = @ds
        and db.dbname = @db
        and db.ltype = 'P' and db.ldbid *= ldb.dbid
union
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers
from rs_dbreps dbrep, rs_databases db, rs_sites rs
where dbrep.dbid = db.dbid
        and dbrep.prsid = rs.id
        and db.dsname = @ds
        and db.dbname = @db
        and db.ltype = 'L'
order by 1 asc
go

/*
** RMP_LOAD_DB_REPDEF
**
** Load the details of a database rep def.
** @dbrepdef = database repdef name
** @ds = data server name
** @db = database name
*/

if exists (
	select * from sysobjects
	where name = 'rmp_load_db_repdef' and type = 'P')
begin
	drop procedure rmp_load_db_repdef
end
go
create procedure rmp_load_db_repdef
        @dbrepdef varchar(255),
        @ds varchar(255),
        @db varchar(255)
as
select dbrep.dbrepname, db.dsname, db.dbname, rs.name, dbrep.minvers,
        dbrep.status, dbrep.dbrepid
from rs_dbreps dbrep, rs_databases db, rs_sites rs
where dbrep.dbid = db.dbid and dbrep.prsid = rs.id and dbrep.dbrepname = @dbrepdef
        and db.dsname = @ds and db.dbname = @db
order by dbrep.dbrepname asc
go

/*
** RMP_LOAD_DB_REPDEF_FILTERS
**
** Load the details of a database rep def.
** @dbrepdef = database repdef name
** @ds = data server name
** @db = database name
*/

if exists (
	select * from sysobjects
	where name = 'rmp_load_db_repdef_filters' and type = 'P')
begin
	drop procedure rmp_load_db_repdef_filters
end
go
create procedure rmp_load_db_repdef_filters
        @dbrepdef varchar(255),
        @ds varchar(255),
        @db varchar(255)
as
select dbrep.dbrepname, db.dsname, db.dbname, sub.type, sub.owner, sub.name
from rs_dbreps dbrep, rs_databases db, rs_dbsubsets sub
where dbrep.dbrepid = sub.dbrepid and dbrep.dbid = db.dbid
        and dbrep.dbrepname = @dbrepdef
        and db.dsname = @ds
        and db.dbname = @db
order by sub.type, sub.name asc
go

/******************************** RMP Classes ********************************/

/*
** RMP_ERROR_CLASS
**
** Retrieve the error classes for a Rep Server
** @servername - Replication Server
** @classname - Error class name
*/
if exists (
        select * from sysobjects
        where name = 'rmp_error_class' and type = 'P')
begin
        drop procedure rmp_error_class
end
go

create procedure rmp_error_class
    @servername varchar(255),
	@classname varchar(30) = "*"
as

/* Get classes that are supplied.*/
/* Supplied classes are never local.*/
select servername=@servername, class_name=rs_classes.classname, is_local=0
from rs_classes
where rs_classes.prsid = 0
	and rs_classes.classtype = 'E'
	and rs_classes.parent_classid = convert(binary(8), 0)
	and ((rs_classes.classname = @classname) or
                (@classname = "*"))
union all

/* Get classes that are not supplied and are local.*/
select servername=@servername, class_name=rs_classes.classname, is_local=1
from rs_classes, rs_sites
where rs_classes.prsid = rs_sites.id
	and rs_sites.name = @servername
	and rs_classes.classtype = 'E'
	and rs_classes.parent_classid = convert(binary(8), 0)
	and ((rs_classes.classname = @classname) or
		(@classname = "*"))
union all

/* Get classes that are not supplied and are not local.*/
select servername=@servername, class_name=rs_classes.classname, is_local=0
from rs_classes, rs_sites
where rs_classes.prsid != rs_sites.id
	and rs_sites.name = @servername
	and rs_classes.classtype = 'E'
	and rs_classes.parent_classid = convert(binary(8), 0)
	and ((rs_classes.classname = @classname) or
		(@classname = "*"))
go

/*
** RMP_FSTRING_CLASS
**
** Get the function string classes known.
**
** @current_rep_server_name - Current Rep Server
** @name - Function String Class name
*/
if exists (
        select * from sysobjects
	where name = 'rmp_fstring_class' and type = 'P')
begin
        drop procedure rmp_fstring_class
end
go

create procedure rmp_fstring_class
    @current_rep_server_name varchar(255),
	@name varchar(30) = '*'
as

/*
** Get classes that have no primary site and have a parent.
*/
select site=@current_rep_server_name, name=child.classname, primary_site=NULL,
        parent_class=parent.classname, is_default_class=child.attributes
from rs_classes child, rs_classes parent
where child.prsid = 0
        and child.classtype = 'F'
        and parent.classtype = 'F'
        and child.parent_classid = parent.classid
        and (@name = '*' or child.classname = @name)
union all

/*
** Get classes that have no primary site and have no parent.
*/
select site=@current_rep_server_name, name=rs_classes.classname, primary_site=NULL,
        parent_class=NULL, is_default_class=rs_classes.attributes
from rs_classes
where rs_classes.prsid = 0
        and rs_classes.classtype = 'F'
        and rs_classes.parent_classid = convert(binary(8), 0)
        and (@name = '*' or rs_classes.classname = @name)
union all

/* Get classes that have a primary site and have no parent.*/
select site=@current_rep_server_name, name=rs_classes.classname, primary_site=primary_site.name,
        parent_class=NULL, is_default_class=rs_classes.attributes
from rs_classes, rs_sites current_site, rs_sites primary_site
where rs_classes.prsid = primary_site.id
	and current_site.name = @current_rep_server_name
	and rs_classes.classtype = 'F'
	and rs_classes.parent_classid = convert(binary(8), 0)
	and (@name = '*' or rs_classes.classname = @name)
union all

/* Get classes that have a primary site and have a parent.*/
select site=@current_rep_server_name, name=child.classname, primary_site=primary_site.name,
        parent_class=parent.classname, is_default_class=child.attributes
from rs_classes child, rs_classes parent, rs_sites current_site, rs_sites primary_site
where child.prsid = primary_site.id
	and current_site.name = @current_rep_server_name
	and child.classtype = 'F'
	and parent.classtype = 'F'
	and child.parent_classid = parent.classid
	and (@name = '*' or child.classname = @name)
go

/*************************** RMP Logical Connection **************************/

/*
** RMP_LCONNECTION_DEMAND
**
** Get on-demand database lconnection information for one connection.
**
** @repname - Replication Server
** @sqlname - Data Server
** @dbname - Database
*/
if exists (
	select * from sysobjects
	where name = 'rmp_lconnection_demand' and type = 'P')
begin
	drop procedure rmp_lconnection_demand
end
go
create procedure rmp_lconnection_demand
	@repname varchar(255),
	@sqlname varchar(255),
	@dbname  varchar(255)
as

declare	@dbid		rs_id
declare @rawid		rs_id

select @rawid = (select rdb.dbid
		from rs_databases rdb, rs_sites rsi
		where rsi.name = @repname
		  and rdb.prsid = rsi.id
		  and rdb.ltype = 'L'
		  and rdb.dsname = @sqlname
		  and rdb.dbname = @dbname)

/* If this a byte-swap machine, adjust rs_id */
if (convert(int, 0x00000100) = 65536)
	execute rmp_reverse_bytes @rawid, 1, 4, @dbid output
else
	select @dbid = @rawid

/*
** Setup the return values.  We use variables since the executes can't
** be done as a select.
*/

declare @stat_dist_status		tinyint
declare @stat_src_status		tinyint

select 	@stat_dist_status = r1.dist_status,
	@stat_src_status = r1.src_status
from rs_databases r1
where r1.dbid = @rawid

declare @config_save_interval           varchar(8)
declare @config_mat_save_interval       varchar(8)

execute rmp_find_config_value @dbid, "save_interval",
                               @result = @config_save_interval output
execute rmp_find_config_value @dbid, "materialization_save_interval",
                               @result = @config_mat_save_interval output

/*
** Default values
*/
if (@config_save_interval = NULL)
   	select @config_save_interval = "strict"
if (@config_mat_save_interval = NULL)
   	select @config_mat_save_interval = "strict"

/*
** Now return the results to our caller.
*/

select	"dist_status" =				@stat_dist_status,
	"src_status"  = 			@stat_src_status,
	"save_interval" =			@config_save_interval,
	"materialization_save_interval" =	@config_mat_save_interval
go

/*
** RMP_LOGICAL_SERVERS
**
** Get all logical servers for the replication server
** @rs - rep server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_logical_servers' and type = 'P')
begin
	drop procedure rmp_logical_servers
end
go
create procedure rmp_logical_servers
	@rs varchar(255)
as
select distinct dsname
from rs_databases db, rs_sites si
where si.name = @rs
        and db.ltype = 'L'
        and db.prsid = si.id
order by dsname asc
go


/*********************************** Queues **********************************/

/*
** RMP_QUEUE
**
** Get queue information.
**
** @servername - Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_queue' and type = 'P')
begin
	drop procedure rmp_queue
end
go
create procedure rmp_queue
	@servername varchar(255)
as
set nocount on

create table #subs (rawid binary(8), intid int)

/*
** If this a byte-swap machine, swap the order of the last
** four bytes of the subid
*/

if (convert(int, 0x00000100) = 65536)
	insert #subs
	select rawid = subid,
		   intid = substring(subid, 8, 1) +
				   substring(subid, 7, 1) +
				   substring(subid, 6, 1) +
				   substring(subid, 5, 1)
	from rs_subscriptions
else
	insert #subs
	select rawid = subid, intid = substring(subid, 5, 4)
	from rs_subscriptions

	create table #queues
		 (	name 		varchar(255),
			q_number 	int,
			q_type 		int,
			q_state		int,
			size 		int,
			saved 		int,
			detect_loss 	int,
			ignore_loss 	int,
			first_seg	int,
			q_objid		binary(8),
			q_objid_temp	binary(8),
			xnl_large_msg	varchar(9) NULL )

insert #queues
select distinct dsname + '.' + dbname + '(Inbound)',
	number, 1, q.state, 0, 0, 0, 0, 0, 0, 0, ''
from rs_queues q, rs_databases d
where number = d.dbid and type=1

insert #queues
select distinct
        isnull(convert(varchar(61), name), dsname+'.'+dbname)+'(Outbound)',
	number, 0, q.state, 0, 0, 0, 0, 0, 0, 0, ''
from rs_queues q, rs_databases, rs_sites
where number *= dbid
  and number *= id
  and type=0

insert #queues
select distinct d.dsname+'.'+d.dbname+'(Materialization-'+sub.subname+')',
	number, q.type, q.state, 0, 0, 0, 0, 0, 0, 0, ''
from rs_queues q, rs_subscriptions sub, rs_databases d, #subs
where d.dbid=number
  and sub.dbid=d.dbid
  and #subs.rawid = sub.subid
  and #subs.intid = q.type
  and materializing=1

insert #queues
select distinct d.dsname+'.'+d.dbname+'(Dematerialization-'+sub.subname+')',
	number, q.type, q.state, 0, 0, 0, 0, 0, 0, 0, ''
from rs_queues q, rs_subscriptions sub, rs_databases d, #subs
where d.dbid=number
  and sub.dbid=d.dbid
  and #subs.rawid = sub.subid
  and #subs.intid = q.type
  and dematerializing=1

update #queues
	set size = (select count(*)
	from rs_segments
	where #queues.q_number = rs_segments.q_number
  	  and #queues.q_type = rs_segments.q_type
  	  and used_flag > 0)

update #queues
	set saved = (select count(*)
	from rs_segments
	where #queues.q_number = rs_segments.q_number
  	  and #queues.q_type = rs_segments.q_type
  	  and used_flag > 1)

update #queues
	set detect_loss = detect_loss + (select count(*)
		from rs_oqid
		where #queues.q_number = rs_oqid.q_number
		  and #queues.q_type = rs_oqid.q_type
		  and valid = 1),
		ignore_loss = ignore_loss + (select count(*)
	from rs_oqid
	where #queues.q_number = rs_oqid.q_number
	  and #queues.q_type = rs_oqid.q_type
	  and valid = 2)

update #queues
	set detect_loss = detect_loss + (select count(*)
		from rs_exceptslast
		where #queues.q_number = error_db
	  	  and status = 1),
		ignore_loss = ignore_loss + (select count(*)
	from rs_exceptslast
	where #queues.q_number = error_db
	  and status = 2)

update #queues
	set first_seg = (select isnull(min(logical_seg),0)
			from rs_segments
			where #queues.q_number = rs_segments.q_number
			  and #queues.q_type = rs_segments.q_type)

/* Build the queue objid number to search for wide message flag in rs_config */
update #queues
	set q_objid = convert ( binary(4), q_number )
		+ convert ( binary(4), q_type )

/* Store a copy of the queue objid in case the bytes need to be reversed */
update #queues
	set q_objid_temp = convert ( binary(4), q_number )
		+ convert ( binary(4), q_type )

/* If this a byte-swap machine, reverse the objid */
if (convert(int, 0x00000100) = 65536)
	update #queues
	set q_objid =  substring ( q_objid_temp, 8, 1 ) +
		substring ( q_objid_temp, 7, 1 ) +
		substring ( q_objid_temp, 6, 1 ) +
		substring ( q_objid_temp, 5, 1 ) +
		substring ( q_objid_temp, 4, 1 ) +
		substring ( q_objid_temp, 3, 1 ) +
		substring ( q_objid_temp, 2, 1 ) +
		substring ( q_objid_temp, 1, 1 )

update #queues
	set xnl_large_msg = (select charvalue
				from rs_config
				where rs_config.objid = #queues.q_objid
				and rs_config.optionname =
					'sqm_xact_with_large_msg' )

update #queues
	set xnl_large_msg = 'shutdown' where xnl_large_msg = NULL

select @servername, name, q_number, q_type, size, saved,
	convert(char(40), getdate(), 109),
	detect_loss, ignore_loss, first_seg,  "DOWN", xnl_large_msg
from #queues
where q_number != 0 and q_state != 1
union

select @servername, name, q_number, q_type, size, saved,
	convert(char(40), getdate(), 109),
 	detect_loss, ignore_loss, first_seg,  "UP", xnl_large_msg
from #queues
where q_number != 0 and q_state = 1

/* Cleanup */
drop table #queues
drop table #subs
go

/****************************** Replication Server ***************************/

/*
** RMP_SITE_DATA
**
** Retrieve the site data for the Replication Server.
*/
if exists (
	select * from sysobjects
	where name = 'rmp_site_data' and type = 'P')
begin
	drop procedure rmp_site_data
end
go
create procedure rmp_site_data
	@rs varchar(255)
as
select
        ID = (select id from rs_sites where name=@rs),
        Site_Version = (select version from rs_version where siteid=1),
        System_Version = (select version from rs_version where siteid=0),
        ID_Server = (select charvalue from rs_config where optionname='id_server')
go

/*********************************** Routes **********************************/

/*
** RMP_ROUTES
**
** Get all routes for the replication server. This includes routes where the
** rep server is an intermediate site.
**
** @rs - Controlling Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_routes' and type = 'P')
begin
	drop procedure rmp_routes
end
go
create procedure rmp_routes
	@rs varchar(255)
as
select origin=o.name, thru=t.name, destination=d.name
from rs_routes r, rs_sites o, rs_sites d, rs_sites t
where o.name=@rs
        and o.id=r.source_rsid
        and d.id=r.dest_rsid
        and t.id=r.through_rsid
union all

select origin=o.name, thru=t.name, destination=d.name
from rs_routes r, rs_sites o, rs_sites d, rs_sites t
where d.name=@rs
        and d.id=r.dest_rsid
        and o.id=r.source_rsid
        and t.id=r.through_rsid
union all

select origin=o.name, thru=t.name, destination=d.name
from rs_routes r, rs_sites o, rs_sites d, rs_sites t
where t.name=@rs
        and r.dest_rsid<>r.through_rsid
        and t.id=r.through_rsid
        and o.id=r.source_rsid
        and d.id=r.dest_rsid
order by 1 asc, 3 asc
go

/*
** RMP_ROUTES_AT_ORIGIN
**
** Get all routes that have the same origin site.
**
** @originRS - Origin Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_routes_at_origin' and type = 'P')
begin
	drop procedure rmp_routes_at_origin
end
go
create procedure rmp_routes_at_origin
	@originRS varchar(255)
as
select origin=o.name, thru=t.name, destination=d.name
from rs_routes r, rs_sites o, rs_sites d, rs_sites t
where o.name=@originRS
        and o.id=r.source_rsid
        and d.id=r.dest_rsid
        and t.id=r.through_rsid
order by origin asc, destination asc
go

/*
** RMP_LOAD_ROUTE
**
** Load the route information
**
** @originRS - Origin Rep Server
** @destRS - Destination Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_load_route' and type = 'P')
begin
	drop procedure rmp_load_route
end
go
create procedure rmp_load_route
	@originRS varchar(255),
	@destRS varchar(255)
as
select u.username, v.route_version, v.status
from rs_routes r, rs_sites o, rs_sites d, rs_maintusers u, rs_routeversions v
where o.name=@originRS
        and d.name=@destRS
        and o.id=r.source_rsid
        and d.id=r.dest_rsid
        and o.id=v.source_rsid
        and d.id=v.dest_rsid
        and d.id*=u.destid
go

/*
** RMP_LOAD_ROUTE_STATUS
**
** Load the route status information.
**
** @originRS - Origin Rep Server
** @destRS - Destination Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_load_route_status' and type = 'P')
begin
	drop procedure rmp_load_route_status
end
go
create procedure rmp_load_route_status
	@originRS varchar(255),
	@destRS varchar(255)
as
select r.status, r.suspended
from rs_routes r, rs_sites o, rs_sites d
where o.name=@originRS
        and d.name=@destRS
        and o.id=r.source_rsid
        and d.id=r.dest_rsid
go

/*
** RMP_CHECK_ROUTE_RECOVERY
**
** Check the rs_recovery table for more status information
**
** @destRS - Destination Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_check_route_recovery' and type = 'P')
begin
	drop procedure rmp_check_route_recovery
end
go
create procedure rmp_check_route_recovery
	@destRS varchar(255)
as
select state, action
from rs_recovery
where (action = 1 or action = 2)
        and textlen = 4
        and convert(int, text) =
                (select distinct dest_rsid from rs_routes r, rs_sites d
                 where d.name=@destRS and d.id=r.dest_rsid)
go

/*
** RMP_CHECK_ROUTE_SUBS
**
** Check the subscriptions table for the route's subscriptions
**
** @originRS - Origin Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_check_route_subs' and type = 'P')
begin
	drop procedure rmp_check_route_subs
end
go
create procedure rmp_check_route_subs
	@originRS varchar(255)
as
select subname from rs_subscriptions
where subname like '%_at_' + convert(varchar(15),
        (select distinct source_rsid from rs_routes r, rs_sites o
                where o.name=@originRS and o.id=r.source_rsid))
go

/*
** RMP_ROUTE_DEMAND
**
** Get on-demand route information for one route.
**
** @repserver - Controlling Rep Server
** @origin - Origin Rep Server
** @dest - Destination Rep Server
*/
if exists (
	select * from sysobjects
	where name = 'rmp_route_demand' and type = 'P')
begin
	drop procedure rmp_route_demand
end
go
create procedure rmp_route_demand
	@repserver	varchar(255),
	@origin		varchar(255),
	@dest		varchar(255)
as

declare	@username	varchar(255)

select @username = (select rm.username
		    from rs_maintusers rm, rs_sites rs
		    where rs.name = @dest
		      and rs.id = rm.destid)

/*
** Setup the rest of the return values.  We use variables since the
** executes can't be done as a select.
*/

declare	@site_id		rs_id
declare @raw_id			rs_id

select @raw_id = (select id from rs_sites where name = @dest)

/* If this a byte-swap machine, adjust rs_id */
if (convert(int, 0x00000100) = 65536)
	execute rmp_reverse_bytes @raw_id, 1, 4, @site_id output
else
	select @site_id = @raw_id

declare	@save_interval		varchar(8)
declare	@rsi_batch_size		varchar(8)
declare	@rsi_fadeout_time	varchar(8)
declare	@rsi_packet_size	varchar(8)
declare	@rsi_sync_interval	varchar(8)
declare @config_sec_mechanism   varchar(30)
declare @config_sec_options     int
declare @config_sec_use     	varchar(10)
declare @wide_msg	     	varchar(10)

execute rmp_find_config_value @site_id, "save_interval",
			      @result = @save_interval output

execute rmp_find_config_value @site_id, "rsi_batch_size",
			      @result = @rsi_batch_size output

execute rmp_find_config_value @site_id, "rsi_fadeout_time",
                              @result = @rsi_fadeout_time output

execute rmp_find_config_value @site_id, "rsi_packet_size",
                              @result = @rsi_packet_size output

execute rmp_find_config_value @site_id, "rsi_sync_interval",
                              @result = @rsi_sync_interval output

execute rmp_find_config_value @site_id, "security_mechanism",
                               @result = @config_sec_mechanism output

execute rmp_find_security_config @site_id, @result = @config_sec_options output,
				@use_sec = @config_sec_use output

execute rmp_find_config_value @site_id, "rsi_xact_with_large_msg",
				@result = @wide_msg output

declare	@route_version		int

select @route_version = 1100

if exists (
	select * from sysobjects
	where name = 'rs_routeversions' and type = 'U')
begin
	select @route_version = (select route_version
					from rs_routeversions, rs_sites
					where source_rsid = id
					  and name = @origin
					  and dest_rsid = @raw_id)
end

select	"Username" =		@username,
	"save_interval" =	@save_interval,
	"rsi_batch_size" =	@rsi_batch_size,
	"rsi_fadeout_time" =	@rsi_fadeout_time,
	"rsi_packet_size" =	@rsi_packet_size,
	"rsi_sync_interval" =	@rsi_sync_interval,
	"route_version" = 	@route_version,
	"use_security" =	@config_sec_use,
	"sec_mechanism" =	@config_sec_mechanism,
	"sec_options" =		@config_sec_options,
	"skip_wide_msg" =	@wide_msg
go

/***************************** Table Subscriptions ***************************/

/*
** RMP_SUBSCRIPTION
**
** Get subscriptions at this Replicate Database.
*/

if exists (
	select * from sysobjects
	where name = 'rmp_subscription' and type = 'P')
begin
	drop procedure rmp_subscription
end
go
create procedure rmp_subscription
	@servername varchar(255),
	@repdefname varchar(255) = '*',
	@sqlserver  varchar(255) = '*',
	@database   varchar(255) = '*',
	@subname    varchar(30) = '*'
as
	declare @id		rs_id

	/* First create a temp table with all the subs we are interested in
	** from the parameters. Also initialize each of their where clauses.
	** At the replicate site, we will always find the dbid in rs_databases,
	** But at the primary site, if the routes are not bi-directional,
	** the dbid would only be found in rs_repdbs. Hence match both tables
	** and take a union.
	*/
	create table #subs (subid binary(8), objid binary(8),
		replicate_has_standby int, objtype char(1))

	insert #subs
	select distinct subid, s.objid,
		replicate_has_standby = 0, o.objtype
	from rs_subscriptions s, rs_objects o, rs_databases rd
	where s.objid = o.objid
		and o.ownertype = 'U'
		and s.dbid = rd.dbid
		and (@repdefname = '*' or @repdefname = o.objname)
		and (@sqlserver = '*' or @sqlserver = rd.dsname)
		and (@database = '*' or @database = rd.dbname)
		and (@subname = '*' or @subname = s.subname)

	union

        select distinct subid, s.objid, replicate_has_standby = 0, o.objtype
        from rs_subscriptions s, rs_objects o, rs_repdbs rd
        where s.objid = o.objid
                and o.ownertype = 'U'
                and s.dbid = rd.dbid
                and (@repdefname = '*' or @repdefname = o.objname)
                and (@sqlserver = '*' or @sqlserver = rd.dsname)
                and (@database = '*' or @database = rd.dbname)
                and (@subname = '*' or @subname = s.subname)

	create unique clustered index #subsind on #subs(subid)

	update #subs
	  set replicate_has_standby = 1
	  from #subs, rs_subscriptions s, rs_databases d
	  where #subs.subid = s.subid
	    and s.dbid = d.ldbid
	    and d.ptype = 'S'

	select @id=id from rs_sites where name = @servername

	/* First fetch all subscriptions for which replicate database is
	** controlled by this RS
	*/

	select distinct @servername, objname, subname, dsname, dbname,
		s.status, recovering, method, materializing, dematerializing,
		error_flag, username, replicate_has_standby, rrs = @servername,
		type, o.objtype, o.deliver_as_name, o.repl_objowner, security,
		mechanism
	from rs_subscriptions s, rs_objects o, rs_databases d, #subs, rs_users
	where o.objid=s.objid
	  and o.dbid != s.dbid
	  and d.dbid=s.dbid
	  and d.prsid=@id
	  and #subs.subid = s.subid
	  and s.rownerid *= rs_users.uid
	  and o.ownertype = 'U'

	union all

	/* Now fetch all subscriptions for which replicate database is controlled
	** by a different RS
	*/
	select distinct @servername, objname, subname, dsname, dbname,
		s.status, recovering, method, materializing, dematerializing,
		error_flag, username, replicate_has_standby, rrs = rs_sites.name,
		type, o.objtype, o.deliver_as_name, o.repl_objowner, security,
		mechanism
	from rs_subscriptions s, rs_objects o, rs_repdbs rd, #subs, rs_sites,
		 rs_users
	where o.objid=s.objid
	  and rd.dbid=s.dbid
	  and rd.controllerid = rs_sites.id
	  and rd.controllerid != @id
	  and #subs.subid = s.subid
	  and s.pownerid *= rs_users.uid
	  and o.ownertype = 'U'

drop table #subs
go

/*
** RMP_SOURCE_SUBSCRIPTIONS
**
** Get subscriptions valid on this primary table/database/server.
*/

if exists (
	select * from sysobjects
	where name = 'rmp_source_subscriptions' and type = 'P')
begin
	drop procedure rmp_source_subscriptions
end
go
create procedure rmp_source_subscriptions
	@servername varchar(255),
	@sqlserver  varchar(255) = '*',
	@database   varchar(255) = '*',
	@owner      varchar(255) = '*',
	@tablename  varchar(255) = '*'
as
	declare @id		rs_id

	/* First create a temp table with all the subs we are interested in
	** from the parameters. Also initialize each of their where clauses.
	** Since we are interested in the subscriptions that are defined on
	** a source table, we need to join rs_databases with rs_objects, so
	** that we can get the primary database reliably.
	*/
	create table #subs (subid binary(8), objid binary(8),
		replicate_has_standby int, objtype char(1))
	insert into #subs
	select distinct subid, s.objid, replicate_has_standby = 0, o.objtype
	from rs_subscriptions s, rs_objects o, rs_databases rd
	where s.objid = o.objid
		and o.ownertype = 'U'
		and o.dbid = rd.dbid
		and (@sqlserver = '*' or @sqlserver = rd.dsname)
		and (@database = '*' or @database = rd.dbname)
		and (@owner = '*' or @owner = o.phys_objowner or
		        (@owner = 'dbo' and
		        (rtrim(o.phys_objowner) is NULL or
		        rtrim(o.phys_objowner) = '')))
		and (@tablename = '*' or @tablename = o.phys_tablename)
	create unique clustered index #subsind on #subs(subid)

	update #subs
	  set replicate_has_standby = 1
	  from #subs, rs_subscriptions s, rs_databases d
	  where #subs.subid = s.subid
	    and s.dbid = d.ldbid
	    and d.ptype = 'S'

	select @id=id from rs_sites where name = @servername

	/* First fetch all subscriptions for which replicate database is
	** controlled by this RS
	*/

	select distinct @servername, objname, subname, dsname, dbname,
		s.status, recovering, method, materializing, dematerializing,
		error_flag, username, replicate_has_standby, rrs = @servername,
		type, o.objtype, o.deliver_as_name, o.repl_objowner, security,
		mechanism
	from rs_subscriptions s, rs_objects o, rs_databases d, #subs, rs_users
	where o.objid=s.objid
	  and o.dbid != s.dbid
	  and d.dbid=s.dbid
	  and d.prsid=@id
	  and #subs.subid = s.subid
	  and s.rownerid *= rs_users.uid
	  and o.ownertype = 'U'

	union all

	/* Now fetch all subscriptions for which replicate database is controlled
	** by a different RS
	*/
	select distinct @servername, objname, subname, dsname, dbname,
		s.status, recovering, method, materializing, dematerializing,
		error_flag, username, replicate_has_standby, rrs = rs_sites.name,
		type, o.objtype, o.deliver_as_name, o.repl_objowner, security,
		mechanism
	from rs_subscriptions s, rs_objects o, rs_repdbs rd, #subs, rs_sites,
		 rs_users
	where o.objid=s.objid
	  and rd.dbid=s.dbid
	  and rd.controllerid = rs_sites.id
	  and rd.controllerid != @id
	  and #subs.subid = s.subid
	  and s.pownerid *= rs_users.uid
	  and o.ownertype = 'U'

drop table #subs
go

/*
** RMP_TARGET_SUBSCRIPTIONS
**
** Get subscriptions that target this replicate table/database/server.
*/

if exists (
	select * from sysobjects
	where name = 'rmp_target_subscriptions' and type = 'P')
begin
	drop procedure rmp_target_subscriptions
end
go
create procedure rmp_target_subscriptions
	@servername varchar(255),
	@sqlserver  varchar(255) = '*',
	@database   varchar(255) = '*',
	@owner      varchar(255) = '*',
	@tablename  varchar(255) = '*'
as
	declare @id		rs_id

	/* First create a temp table with all the subs we are interested in
	** from the parameters. Also initialize each of their where clauses.
	*/
	create table #subs (subid binary(8), objid binary(8),
		replicate_has_standby int, objtype char(1))

	insert #subs
	select distinct subid, s.objid, replicate_has_standby = 0, o.objtype
	from rs_subscriptions s, rs_objects o, rs_databases rd
	where s.objid = o.objid
		and o.ownertype = 'U'
		and s.dbid = rd.dbid
		and (@sqlserver = '*' or @sqlserver = rd.dsname)
		and (@database = '*' or @database = rd.dbname)
		and (@owner = '*' or @owner = o.repl_objowner or
		        (@owner = 'dbo' and
		        (rtrim(o.repl_objowner) is NULL or
		        rtrim(o.repl_objowner) = '')))
		and (@tablename = '*' or @tablename = o.deliver_as_name)
	create unique clustered index #subsind on #subs(subid)

	update #subs
	  set replicate_has_standby = 1
	  from #subs, rs_subscriptions s, rs_databases d
	  where #subs.subid = s.subid
	    and s.dbid = d.ldbid
	    and d.ptype = 'S'

	select @id=id from rs_sites where name = @servername

	/* First fetch all subscriptions for which replicate database is
	** controlled by this RS
	*/

	select distinct @servername, objname, subname, dsname, dbname,
		s.status, recovering, method, materializing, dematerializing,
		error_flag, username, replicate_has_standby, rrs = @servername,
		type, o.objtype, o.deliver_as_name, o.repl_objowner, security,
		mechanism
	from rs_subscriptions s, rs_objects o, rs_databases d, #subs, rs_users
	where o.objid=s.objid
	  and o.dbid != s.dbid
	  and d.dbid=s.dbid
	  and d.prsid=@id
	  and #subs.subid = s.subid
	  and s.rownerid *= rs_users.uid
	  and o.ownertype = 'U'

	union all

	/* Now fetch all subscriptions for which replicate database is controlled
	** by a different RS
	*/
	select distinct @servername, objname, subname, dsname, dbname,
		s.status, recovering, method, materializing, dematerializing,
		error_flag, username, replicate_has_standby, rrs = rs_sites.name,
		type, o.objtype, o.deliver_as_name, o.repl_objowner, security,
		mechanism
	from rs_subscriptions s, rs_objects o, rs_repdbs rd, #subs, rs_sites,
		 rs_users
	where o.objid=s.objid
	  and rd.dbid=s.dbid
	  and rd.controllerid = rs_sites.id
	  and rd.controllerid != @id
	  and #subs.subid = s.subid
	  and s.pownerid *= rs_users.uid
	  and o.ownertype = 'U'

drop table #subs
go

/*************************** Replication Definitions *************************/

/*
** RMP_TABLE_SOURCE_PROJECTIONS
**
** Get replication definitions for a sqlserver/database/owner/table This
** procedure is usually used at a primary site (RSSD) to get the projections
** defined on a particular table. But it can also be used at a replicate site
** (RSSD) to get the list of valid projections at that replicate site for the
** given *primary table*.
*/
if exists (
	select * from sysobjects
	where name = 'rmp_table_source_projections' and type = 'P')
begin
	drop procedure rmp_table_source_projections
end
go
create procedure rmp_table_source_projections
	@servername varchar(255), @sqlserver varchar(255) = '*',
	@database varchar(255) = '*', @owner varchar(255) = '*',
	@tablename varchar(255) = '*'
as
select distinct @servername, objname, objtype, phys_objowner,
        phys_tablename, repl_objowner, deliver_as_name, dsname,
        dbname, name, rs_objects.attributes, rs_objects.minvers
from rs_objects, rs_databases, rs_sites
where (@sqlserver = '*' or dsname = @sqlserver)
        and (@database = '*' or dbname = @database)
        and (@owner = '*' or phys_objowner = @owner
                or @owner = 'dbo')
        and (@tablename = '*'
                or phys_tablename = @tablename)
        and rs_objects.dbid = rs_databases.dbid
        and rs_sites.id = rs_objects.prsid
        and ownertype = 'U'
go

/*
** RMP_REP_DEF_BY_NAME
**
** Retrieve a replication definition by name
**
** @redef - Replication Definition name
*/
if exists (
	select * from sysobjects
	where name = 'rmp_rep_def_by_name' and type = 'P')
begin
	drop procedure rmp_rep_def_by_name
end
go
create procedure rmp_rep_def_by_name
	@repdef varchar(255)
as
select distinct rd.objname, rd.phys_objowner, rd.phys_tablename,
        rd.repl_objowner, rd.deliver_as_name, conn.dsname, conn.dbname,
        aconn.dsname, aconn.dbname, rs.name, rd.attributes, rd.minvers
from rs_objects rd, rs_databases conn, rs_databases aconn, rs_sites rs
where rd.objname = @repdef
        and rd.dbid = conn.dbid
        and rs.id = rd.prsid
        and rd.ownertype = 'U'
        and rd.objtype = 'R'
        and conn.ldbid *= aconn.ldbid
        and aconn.ltype = 'P'
        and aconn.ptype = 'A'
        and aconn.dbid != aconn.ldbid
go

/*
** RMP_LOAD_REP_DEF
**
** Load the detail information for a replication definition
**
** @table - table name
** @redef - Replication Definition name
** @ds - data server name
** @db - database name
*/
if exists (
	select * from sysobjects
	where name = 'rmp_load_rep_def' and type = 'P')
begin
	drop procedure rmp_load_rep_def
end
go
create procedure rmp_load_rep_def
    @table varchar(255),
	@repdef varchar(255),
	@ds varchar(255),
	@db varchar(255)
as
select distinct cols.colname,  obj.objname, obj.phys_tablename,
        cols.searchable, cols.rowtype, db.dsname, db.dbname, obj.minvers,
        obj.repl_objowner, obj.deliver_as_name, obj.attributes, cols.coltype,
        cols.length, dt.name, cols.status, cols.repl_colname, cols.primary_col,
        cols.publ_length, publ_dt.name, cols.colnum
from rs_columns cols, rs_objects obj, rs_databases db, rs_tvalues dt,
        rs_tvalues publ_dt
where cols.objid = obj.objid and cols.prsid = obj.prsid
        and db.dbid = obj.dbid and obj.phys_tablename = @table
        and obj.objname = @repdef and obj.objtype = 'R'
        and db.dsname = @ds and db.dbname = @db
        and dt.type = 'DT' and cols.coltype *= dt.value
        and publ_dt.type = 'DT' and cols.publ_base_coltype *= publ_dt.value
order by cols.colnum asc
go

/****************************** RMP Proc Version *****************************/

/*
** RMP_PROC_VERSION
**      This procedure identifies the version of the RMP and Rep Server that
**      is associated with the stored procedures in the RSSD. If the values
**      in the RMP and Rep Server do not match the values returned by this
**      procedure, then the stored procedures are out-of-date and must be
**      upgraded.
**
** Version Acceptability checks:
**      1. The rmp_version is the version of the RMP. It must match the version
**         of the RMP that is hardcoded in the Version class.
**	2. The rmp_revision is incremented when an RMP ebf/esd updates this
**         file. It must match the revision of the RMP that is hardcoded in the
**         Version class.
**
**	These above two numbers track the compatibility status between the
**      RMP binary and the stored procedures in thr RSSD. These numbers must
**      change between releases. The numbers may remain constant between maint
**	releases if there are no changes to this file. Please note that these
**      numbers must be the _same_ for all rmp*.sql scripts for a given RMP
**      release.
**
**	3. The rs_script_version identifies the version of the Rep Server that
**         is associated with the stored procedures. The RMP compares this number
**         to the version of the Rep Server. If the Rep Server was upgraded,
**         a new set of stored procedures must be reloaded.
*/
if exists (
	select * from sysobjects
	where name = 'rmp_proc_version' and type = 'P')
begin
	drop procedure rmp_proc_version
end
go
create procedure rmp_proc_version
as
	declare @rmp_version varchar(30)
	declare @rmp_revision varchar(30)
	declare @rs_script_version varchar(30)

	select @rmp_version = "1560"
	select @rmp_revision = "1"
	select @rs_script_version = "1510"

	select "RMP Version" = @rmp_version,
                "RMP Revision" = @rmp_revision,
		"Script Version" = @rs_script_version
go

/* Setup security for the procedures. */
grant execute on rmp_reverse_bytes to rs_systabgroup
grant execute on rmp_find_config_value to rs_systabgroup
grant execute on rmp_find_security_config to rs_systabgroup

grant execute on rmp_connections to rs_systabgroup
grant execute on rmp_all_connections to rs_systabgroup
grant execute on rmp_connection_details to rs_systabgroup
grant execute on rmp_connection_demand to rs_systabgroup

grant execute on rmp_exceptions_log to rs_systabgroup
grant execute on rmp_exceptions_log_lang to rs_systabgroup
grant execute on rmp_exceptions_log_src to rs_systabgroup
grant execute on rmp_delexception to rs_systabgroup

grant execute on rmp_all_db_repdef to rs_systabgroup
grant execute on rmp_db_repdef to rs_systabgroup
grant execute on rmp_db_repdef_for_conn to rs_systabgroup
grant execute on rmp_load_db_repdef to rs_systabgroup
grant execute on rmp_load_db_repdef_filters to rs_systabgroup

grant execute on rmp_error_class to rs_systabgroup
grant execute on rmp_fstring_class to rs_systabgroup

grant execute on rmp_lconnection_demand to rs_systabgroup
grant execute on rmp_logical_servers to rs_systabgroup

grant execute on rmp_queue to rs_systabgroup

grant execute on rmp_site_data to rs_systabgroup

grant execute on rmp_routes to rs_systabgroup
grant execute on rmp_routes_at_origin to rs_systabgroup
grant execute on rmp_load_route to rs_systabgroup
grant execute on rmp_load_route_status to rs_systabgroup
grant execute on rmp_check_route_recovery to rs_systabgroup
grant execute on rmp_check_route_subs to rs_systabgroup
grant execute on rmp_route_demand to rs_systabgroup

grant execute on rmp_subscription to rs_systabgroup
grant execute on rmp_source_subscriptions to rs_systabgroup
grant execute on rmp_target_subscriptions to rs_systabgroup

grant execute on rmp_table_source_projections to rs_systabgroup
grant execute on rmp_rep_def_by_name to rs_systabgroup
grant execute on rmp_load_rep_def to rs_systabgroup

grant execute on rmp_proc_version to rs_systabgroup
go
