##===========================================================
## List ALL DbxTune and DbxCentral processes
##===========================================================

##----------------------------------------------
## Set some switch defaults
##----------------------------------------------
param (
    [switch]$l = $false,
    [switch]$long = $false,
    [switch]$n = $false, 
    [switch]$noSummary = $false,
    [switch]$r = $false,
    [switch]$onlyRunning = $false
)

## Handle aliases for switches
$longList = $l -or $long
$printSummary = -not ($n -or $noSummary)
$onlyRunning = $r -or $onlyRunning

##----------------------------------------------
## Source environment
##----------------------------------------------
#$envFile = "$env:USERPROFILE\.dbxtune\DBXTUNE.env.bat"
$envFile = "$PWD\DBXTUNE.env.bat"
if (Test-Path $envFile) {
    ## Run the batch file and capture the environment variables
    $cmdOutput = cmd /c "($envFile) && set"

    ## Apply the changes to PowerShell's environment
    $cmdOutput | ForEach-Object {
        if ($_ -match '^(.*?)=(.*)$') {
            $envName  = $matches[1]
            $envValue = $matches[2]

            ## Only set SOME environment variables for THIS Process
            if ($envName -match "^DBXTUNE" -or $envName -eq "SYBASE") {
                Write-Host " -------------->>>>>>>> setting env var: $envName = |$envValue|"
                [System.Environment]::SetEnvironmentVariable($envName, $envValue, "Process")
                Set-Variable -Name $envName -Value $envValue -Scope Global
            }
        }
    }
}

##----------------------------------------------
## Override some settings/environments
##----------------------------------------------
if (-not [string]::IsNullOrEmpty($env:DBXTUNE_CENTRAL_BASE)) {
    $dbxCentralBase = $env:DBXTUNE_CENTRAL_BASE
} else {
    $dbxCentralBase = "$env:USERPROFILE\.dbxtune\dbxc"
}

##-------------------------------------
## Check for ANY DbxTune and DbxCentral processes
##-------------------------------------
#$pidListDbxTune = @(Get-Process -Name java | Where-Object { $_.CommandLine -match 'com\.dbxtune.*Tune'                    -and $_.Parent.ToString().Contains("java") } )
#$pidListCentral = @(Get-Process -Name java | Where-Object { $_.CommandLine -match 'com\.dbxtune\.central\.DbxTuneCentral' -and $_.Parent.ToString().Contains("java") } )
## Note: Get-Process ... field "CommandLine" is empty if executed from another user... Thats why using Get-CimInstance instead...
$pidListDbxTune    = @(Get-CimInstance -Class Win32_Process -Filter "Name LIKE 'java%'" | Where-Object { $_.CommandLine -match 'com\.gorans\.SleepServer'             -and $_.CommandLine -match '^"' } ) ## There are normally 2 processes one starts with 'java' the otherone starts with "C:\....\java.exe" ...'
$pidListCentral    = @(Get-CimInstance -Class Win32_Process -Filter "Name LIKE 'java%'" | Where-Object { $_.CommandLine -match 'com\.gorans\.central\.DbxTuneCentral' -and $_.CommandLine -match '^"' } ) ## There are normally 2 processes one starts with 'java' the otherone starts with "C:\....\java.exe" ...'
$svcAllDbxTune     = @(Get-Service -Include 'SleepServer*')
$svcRunningDbxTune = @(Get-Service -Include 'SleepServer*' | Where-Object {$_.Status -eq 'Running'})
$svcListCentral    = @(Get-Service -Include 'DbxCentral*')
$svcRunningCentral = @(Get-Service -Include 'DbxCentral*' | Where-Object {$_.Status -eq 'Running'})

##-------------------------------------
## Check if we are in elevated mode
##-------------------------------------
$isElevated = (New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
if ($isElevated) {
    Write-Host ""
    Write-Host " * INFO: Running script in ELEVATED mode..."
} else {
    Write-Host ""
    Write-Host " * INFO: NOT Running in elevated mode... So Dbx Processes started by 'someone else' will NOT be visible..."
    Write-Host "         For examples Processes started as a Service wont be visible, just the service name"
}

## Check if services are up and running!
Write-Host ""
if ($svcAllDbxTune.Count -eq 0) {
    Write-Host " * INFO: NO Service is installed for any DbxTune Collectors..."
    Write-Host ""
} else {
    if ($svcRunningDbxTune.Count -eq 0) {
        Write-Host " * No DbxTune COLLECTOR SERVICES are running..."
        Write-Host ""
    }
}

if ($svcListCentral.Count -eq 0) {
    Write-Host " * INFO: NO Service is installed for DbxCentral..."
    Write-Host ""
} else {
    if ($svcRunningCentral.Count -eq 0) {
        Write-Host " * No DbxTune CENTRAL SERVICE is running..."
        Write-Host ""
    }
}


## Check if pid are up and running!
if ($isElevated)
{
    Write-Host ""
    if ($pidListDbxTune.Count -eq 0) {
        Write-Host " * No DbxTune COLLECTOR processes is running..."
        Write-Host ""
    }
    
    if ($pidListCentral.Count -eq 0) {
        Write-Host " * No DbxTune CENTRAL Server processes is running..."
        Write-Host ""
    }
} 

## If no pids/services, just exit
if ($svcRunningDbxTune.Count -eq 0 -and $svcRunningCentral.Count -eq 0 -and $pidListDbxTune.Count -eq 0 -and $pidListCentral.Count -eq 0) {
    exit 0
}

#$onlyRunning=$true
#$longList=$true

##-------------------------------------
## -r option... Just do THIS Part of the code
##-------------------------------------
if ($onlyRunning) 
{
    ## DbxTune services
    if ($svcRunningDbxTune.Count -gt 0) 
    {
        foreach ($svc in $svcRunningDbxTune) 
        {
            $serviceName = $svc.Name
            $srvName     = $serviceName.Replace('SleepServer__','')

            Write-Host " * SERVICE DbxTune Collector for server '$srvName' is running."
            Write-Host ""
        }
    }

    ## DbxTune processes
    if ($pidListDbxTune.Count -gt 0) {
        foreach ($dbxPid in $pidListDbxTune) 
        {
            $fullCmd = $dbxPid.CommandLine

            # Parse command line arguments
            $srvName        = "unknown-srv"
            $srvNameOrAlias = ""
            $withAliasDesc  = ""
            $appName        = ""
            
            # Extract the server name (-S parameter)
            if ($fullCmd -match '-S\s?([^\s]+)') {
                $srvName = $matches[1] -replace "'", ""
            }
            
            # Extract the alias name (-A or --serverAlias parameter)
            if ($fullCmd -match '(-A|--serverAlias)\s?([^\s]+)') {
                $srvNameOrAlias = $matches[2] -replace "'", ""
                $withAliasDesc = " with alias name '$srvNameOrAlias'"
            }
            
            # Extract the app name
#            if ($fullCmd -match 'com\.dbxtune\.([^\.]+)\.') {
            if ($fullCmd -match 'com\.dbxtune\.([a-zA-Z]+Tune)') {
                $appName = $matches[1]
            }
            
            if (-not $appName) {
                $appName = "DbxTune"
            }
            
            # Compose log file name
            if (-not $DBXTUNE_CENTRAL_LOG_DIR) {
                $DBXTUNE_CENTRAL_LOG_DIR = "$dbxCentralBase/log"
            }
            $logFileBase = if ($srvNameOrAlias) { $srvNameOrAlias } else { $srvName }
            $cleanLogFileBase = $logFileBase -replace "'", ""
            $logFile = Join-Path -Path $DBXTUNE_CENTRAL_LOG_DIR -ChildPath "$cleanLogFileBase.log"
            
            if (-not $longList) {
                Write-Host " >> $appName : $srvName$withAliasDesc, PID=$($dbxPid.ProcessId)"
                if (Test-Path $logFile) {
                    $lastLog = Get-Content $logFile -Tail 1
                    Write-Host "    Last log entry: $lastLog"
                }
            } else {
                Write-Host ""
                Write-Host " >> $appName : $srvName$withAliasDesc, PID=$($dbxPid.ProcessId)"
                Write-Host "======================================================"
                $dbxPid | Format-Table -Wrap ProcessId, @{Name="StartTime";Expression={$_.CreationDate.ToString("yyyy-MM-dd HH:mm")}}, CommandLine
                if (Test-Path $logFile) {
                    Write-Host "==== last log entry =================================="
                    $lastLog = Get-Content $logFile -Tail 1
                    Write-Host "    Last log entry: $lastLog"
                    Write-Host "======================================================"
                }
            }
        }
    }
    
    ## DbxCentral SERVICES
    if ($svcRunningCentral.Count -gt 0) 
    {
        foreach ($svc in $svcRunningCentral) 
        {
            $serviceName = $svc.Name
            $srvName     = $serviceName.Replace('SleepServer__','')

            if (-not $longList) {
                Write-Host " * SERVICE DbxCentral is running."
                Write-Host ""
            }
        }
    }

    ## DbxCentral PID
    foreach ($dbxPid in $pidListCentral) 
    {
        if (-not $longList) {
            Write-Host " >> DbxCentral: DBXTUNE_CENTRAL, PID=$($dbxPid.ProcessId)"
        } else {
            Write-Host ""
            Write-Host " >> DbxCentral: DBXTUNE_CENTRAL, PID=$($dbxPid.ProcessId)"
            Write-Host "======================================================"
            $dbxPid | Format-Table -Wrap ProcessId, @{Name="StartTime";Expression={$_.CreationDate.ToString("yyyy-MM-dd HH:mm")}}, CommandLine
            Write-Host "======================================================"
        }
    }
    
    $totalCount = $svcRunningDbxTune.Count + $svcRunningCentral.Count + $pidListDbxTune.Count + $pidListCentral.Count
    exit $totalCount
}
#Write-Host " >> XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"
#exit 1

##-------------------------------------
## Print REGISTERED Servers
##
## Get list of servers to start from file: ${dbxCentralBase}/conf/SERVER_LIST
## Fields in this file
##   1 - ASE SERVERNAME
##   2 - 1=Enabled, 0=Disabled
##   3 - Some explanation for the role of this server
##-------------------------------------

## Remove known pids from the below list (if registered servers)
$unRegisteredPidListArray = @() + $pidListDbxTune

$srvCount = 0
$srvUpCount = 0
$srvDownCount = 0
$srvDownList = @()

$serverListFile = "$dbxCentralBase\conf\SERVER_LIST"
if (Test-Path $serverListFile) {

    $srvList = Get-Content $serverListFile | Where-Object { $_ -match '\S' -and -not $_.StartsWith('#') } | ForEach-Object {
        $fields = $_ -split ';'
        if ([int]$fields[1] -gt 0) {
            $fields[0].Trim()
        }
    }
    
    foreach ($srvName in $srvList) 
    {
        $srvCount++
        
        # SQL Server specifics. escape \ into \\ for regex
        $escapedSrvName = [regex]::Escape($srvName)
        
        # Check if server is already running
        $runningSrvName = @($pidListDbxTune | Where-Object { $_.CommandLine -match $escapedSrvName } )

        # Server is STARTED
        if ($runningSrvName.Count -gt 0) {
            $srvUpCount++
            $dbxPid = $runningSrvName[0].ProcessId
            
            # Remove from "all" list
            $unRegisteredPidListArray = $unRegisteredPidListArray | Where-Object { $_.ProcessId -ne $dbxPid }
            
            # Get details about this PID
            $fullCmd = $runningSrvName[0].CommandLine
            $srvNameOrAlias = ""
            $withAliasDesc  = ""
            $appName        = ""
            
            # Extract the alias name
            if ($fullCmd -match '(-A|--serverAlias)\s+([^\s]+)') {
                $srvNameOrAlias = $matches[2] -replace "'", ""
                $withAliasDesc = " with alias name '$srvNameOrAlias'"
            }
            
            # Extract app name
            if ($fullCmd -match 'com\.dbxtune\.([a-zA-Z]+Tune)') {
                $appName = $matches[1]
            }
            
            if (-not $appName) {
                $appName = "DbxTune"
            }
            
            if (-not $longList) {
                Write-Host (" >> INFO: {0,-15} - Collector for '$srvName'$withAliasDesc is running... PID=$dbxPid" -f $appName)
            } else {
                Write-Host ""
                Write-Host " >> INFO: $appName - Collector for '$srvName'$withAliasDesc is running... PID=$dbxPid"
                Write-Host "======================================================"
                Write-Host "$fullCmd"
                Write-Host "======================================================"
            }
        } else {
            # Server is DOWN
            $srvDownList += $srvName
            $srvDownCount++
        }
    }
}

foreach ($dbxPid in $pidListCentral) 
{
    if (-not $longList) {
        Write-Host (" >> INFO: {0,-15} - CENTRAL SERVER, PID=$($dbxPid.ProcessId)" -f "DbxCentral")
    } else {
        Write-Host ""
        Write-Host " >> INFO: DbxCentral - CENTRAL SERVER, PID=$($dbxPid.ProcessId)"
        Write-Host "======================================================"
        $dbxPid | Format-Table -Wrap ProcessId, @{Name="StartTime";Expression={$_.CreationDate.ToString("yyyy-MM-dd HH:mm")}}, CommandLine
        Write-Host "======================================================"
    }
}

##-------------------------------------
## Print UN-REGISTERED Servers
##-------------------------------------
if ($unRegisteredPidListArray.Count -gt 0) {
    Write-Host ""
    Write-Host "List of UN-REGISTERED servers (not in $dbxCentralBase\conf\SERVER_LIST)"
    
    foreach ($dbxPid in $unRegisteredPidListArray) {
        $fullCmd = $dbxPid.CommandLine
        $srvName = "-unknown-"
        $appName = "-unknown-"
        
        # Extract server name
        if ($fullCmd -match '-S\s?([^\s]+)') {
            $srvName = $matches[1] -replace "'", ""
        }
        
        # Extract app name
#        if ($fullCmd -match 'com\.dbxtune\.([a-zA-Z]+Tune)') {
        if ($fullCmd -match 'com\.gorans\.(SleepServer)') {
            $appName = $matches[1]
        }
        
#        if (-not $appName) {
#            $appName = "DbxTune"
#        }
        
        if (-not $longList) {
            Write-Host " >> INFO: $appName - Collector for server $srvName is running... PID=$($dbxPid.ProcessId)"
        } else {
            Write-Host ""
            Write-Host " >> INFO: $appName - Collector for server $srvName is running... PID=$($dbxPid.ProcessId)"
            Write-Host "======================================================"
            Write-Host "$fullCmd"
            Write-Host "======================================================"
        }
    }
}

##-------------------------------------
## Print NOT RUNNING REGISTERED Servers
##-------------------------------------
if ($srvDownList.Count -gt 0) {
    Write-Host ""
    Write-Host "======================================================"
    Write-Host " WARNING: Servers are DOWN, below is a list of servers that are NOT started (but exists in $dbxCentralBase\conf\SERVER_LIST)"
    Write-Host "          To start them, RUN: .\start_ALL.ps1"
    Write-Host "======================================================"
    foreach ($srvName in $srvDownList) {
        Write-Host " >> WARN: Collector for '$srvName' is NOT running..."
    }
}

if ($printSummary) {
    Write-Host ""
    Write-Host "DbxTune Collector Count: $($pidListDbxTune.Count)   (in $dbxCentralBase\conf\SERVER_LIST: $srvCount)"
    Write-Host "DbxCentral Server Count: $($pidListCentral.Count)"
    Write-Host ""
    
    if ($srvDownCount -gt 0) {
        Write-Host "WARNING: $srvDownCount Collectors are down: $($srvDownList -join ', ')"
        Write-Host ""
    }
    
    if ($pidListCentral.Count -eq 0) {
        Write-Host "WARNING: DbxTune Central Server IS DOWN"
        Write-Host ""
    }
    
    if ($srvDownCount -eq 0 -and $pidListCentral.Count -gt 0) {
        Write-Host "OK: All registered collectors and DbxCentral are running."
        Write-Host ""
    }
}

$totalCount = $svcRunningDbxTune.Count + $svcRunningCentral.Count + $pidListDbxTune.Count + $pidListCentral.Count
exit $totalCount
