##===========================================================
## START ALL DbxTune and DbxCentral processes
## - only start if NOT already running
##===========================================================

##----------------------------------------------
## Source environment
##----------------------------------------------
#$envFile = "$env:USERPROFILE\.dbxtune\DBXTUNE.env.bat"
$envFile = "$PWD\DBXTUNE.env.bat"
if (Test-Path $envFile) {
	## Run the batch file and capture the environment variables
	$cmdOutput = cmd /c "($envFile) && set"

	## Apply the changes to PowerShell's environment
	$cmdOutput | ForEach-Object {
		if ($_ -match '^(.*?)=(.*)$') 
		{
			$envName  = $matches[1]
			$envValue = $matches[2]

			## Only set SOME environment variables for THIS Process
			if ($envName -match "^DBXTUNE" -or $envName -eq "SYBASE") 
			{
				Write-Host " -------------->>>>>>>> setting env var: $envName = |$envValue|"
				[System.Environment]::SetEnvironmentVariable($envName, $envValue, "Process")
				Set-Variable -Name $envName -Value $envValue -Scope Global
			}
		}
	}
}



##----------------------------------------------
## Override some settings/environments
##----------------------------------------------
$sleepTime=3

## dbxCentralBase
if (-not [string]::IsNullOrEmpty($env:DBXTUNE_CENTRAL_BASE)) {
	$dbxCentralBase = $env:DBXTUNE_CENTRAL_BASE
} else {
	$dbxCentralBase = "$env:USERPROFILE\.dbxtune\dbxc"
}

## SYBASE
if (-not [string]::IsNullOrEmpty($env:SYBASE)) {
	$SYBASE = $env:SYBASE
} else {
	$SYBASE = "$env:USERPROFILE\.dbxtune"
}


##----------------------------------------------
## Check for list_ALL.ps1
##----------------------------------------------
$listAll=".\list_ALL.ps1"
if (-not (Test-Path $listAll)) { $listAll = "$dbxCentralBase\bin\list_ALL.ps1" }
if (-not (Test-Path $listAll)) { $listAll = "$env:DBXTUNE_HOME\bin\dbxc_list_ALL.ps1" }
if (-not (Test-Path $listAll)) { $listAll = "C:\projects\DbxTune\bin\dbxc_list_ALL.ps1" }  ## NOTE: REMOVE THIS AFTER TESTING
if (-not (Test-Path $listAll)) { $listAll = "" }


Write-Host ""
Write-Host " * INFO: Using information from file '${dbxCentralBase}/conf/SERVER_LIST' to start ALL processes."

##----------------------------------------------
## Check that the server names are UNIQUE in the SERVER_LIST
##----------------------------------------------
$tmpSrvList = Get-Content "${dbxCentralBase}/conf/SERVER_LIST" | Where-Object { $_ -notmatch '^\s*$' -and $_ -notmatch '^#' } | ForEach-Object { $_.Split(';')[0].Trim() }
$tmpServerCount = $tmpSrvList.Count
$tmpUniqueCount = ($tmpSrvList | Sort-Object -Unique).Count

if ($tmpServerCount -ne $tmpUniqueCount) {
	Write-Host ""
	Write-Host "========================================================================"
	Write-Host " ERROR: server names are NOT unique"
	Write-Host "        All server names in field 1 (server name) must be unique."
	Write-Host ""
	Write-Host " Duplicate Server Count: $($tmpServerCount - $tmpUniqueCount)"
	Write-Host " Total     Server Count: $tmpServerCount"
	Write-Host " Unique    Server Count: $tmpUniqueCount"
	Write-Host "========================================================================"
	Write-Host " SERVER LIST file: ${dbxCentralBase}/conf/SERVER_LIST"
	Write-Host " Below are ALL entries in the file."
	Write-Host "------------------------------------------------------------------------"
	Write-Host ($tmpSrvList -join "`n")
	Write-Host "------------------------------------------------------------------------"
	Write-Host "Exiting..."
	Write-Host ""
	
	exit 1
}


##------------------------------------
## Function: 
##------------------------------------
function getStartCommandForServer {
	param (
		[string]$srvName
	)

	$startCmd = Get-Content "${dbxCentralBase}/conf/SERVER_LIST" 
		| Where-Object { $_ -notmatch '^\s*$' -and $_ -notmatch '^#' } 
		| ForEach-Object { 
			$fields = $_.Split(';')
			if ($fields[0].Trim() -eq $srvName) {
				return $fields[3].Trim() -replace '<SRVNAME>', $srvName
			}
		}

	## If command starts with '/' or '.' or '$' or '~' then return it "as is", if not append './' before the command
	if ($startCmd -match '^[/.~$]')
	{
		return $startCmd
	}
	else
	{
		return "./$startCmd"
	}
}

##
## Starting DbxCentral
##
<# NOTE: COMMENTED OUT
$cnt = Get-CimInstance -Class Win32_Process | Where-Object { $_.Name -eq 'java.exe' -and $_.CommandLine -match '.central.DbxTuneCentral' } | Measure-Object | Select-Object -ExpandProperty Count
if ($cnt -gt 0) 
{
	Write-Host ""
	Write-Host " * INFO: DbxTune CENTRAL is already running... skipping this."
} 
else 
{
	##----------------------------------------------
	## Start DbxCentral
	##----------------------------------------------

	## Check if we have a service called 'DbxCentral'
	$serviceList = @(Get-Service -Include 'DbxCentral')
	if ($serviceList.Count -gt 0) 
	{
		$service = $serviceList[0]
		if ($service.Status -ne "Running")
		{
			Write-Host ""
			Write-Host " * Starting DbxTune CENTRAL (as a SERVICE)."
			Start-Service -Name 'DbxCentral'
			Write-Host "   Sleeping for ${sleepTime} before continuing with next server"
			Start-Sleep -Seconds $sleepTime
		}
	}
	else 
	{
		Write-Host ""
		Write-Host " * Starting DbxTune CENTRAL (with Start-Process in background), output to ${dbxCentralBase}/log/DBX_CENTRAL.console)."
#		Start-Process -FilePath "${dbxCentralBase}/bin/start_dbxcentral.bat" -RedirectStandardOutput "${dbxCentralBase}/log/DBX_CENTRAL.console" -RedirectStandardError "${dbxCentralBase}/log/DBX_CENTRAL.console" -NoNewWindow -PassThru | ForEach-Object {
#		Start-Process -FilePath "powershell" -ArgumentList "-NoProfile -Command `& { & '${dbxCentralBase}/bin/start_dbxcentral.bat' *> '${dbxCentralBase}/log/DBX_CENTRAL.console' }" -NoNewWindow -PassThru | ForEach-Object {
#			$lastBgPid = $_.Id
#		}
		$logPath = "${dbxCentralBase}/log/DBX_CENTRAL.console"
		$process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c", "${dbxCentralBase}/bin/start_dbxcentral.bat > `"$logPath`" 2>&1" -NoNewWindow -PassThru
		$lastBgPid = $process.Id

		Write-Host "   Sleeping for ${sleepTime} before continuing with next server"
		Start-Sleep -Seconds $sleepTime
		
		# Check if the PID is alive
		$isPidAlive = Get-CimInstance -Class Win32_Process -Filter "ProcessId = $lastBgPid" #-ErrorAction SilentlyContinue
		if (-not $isPidAlive) {
			Write-Host ""
			Write-Host "==================================================================="
			Write-Host "ERROR: Problems Starting DbxTune CENTRAL."
			Write-Host "------ BEGIN output from the start command ------------------------"
			Get-Content "${dbxCentralBase}/log/DBX_CENTRAL.console"
			Write-Host "------ END   output from the start command ------------------------"
			Write-Host ""
		}
	}
}
#>

#exit 1
##
## Get list of servers to start from file: ${dbxCentralBase}/conf/SERVER_LIST
## Fields in this file
##   1 - ASE SERVERNAME
##   2 - 1=Enabled, 0=Disabled
##   3 - Some explanation for the role of this server
##   4 - start command
##
$srvList = Get-Content "${dbxCentralBase}/conf/SERVER_LIST" | Where-Object { $_ -notmatch '^\s*$' -and $_ -notmatch '^#' } | ForEach-Object { 
	$fields = $_.Split(';')
	if ([int]$fields[1] -gt 0) { $fields[0].Trim() }
}

foreach ($srvName in $srvList) 
{
	$escapedSrvName = $srvName -replace '\\', '\\\\'

	$cnt = Get-CimInstance -Class Win32_Process | Where-Object { $_.CommandLine -match "com.dbxtune.*Tune" -and $_.CommandLine -match "\-S\s?$escapedSrvName\s" } | Measure-Object | Select-Object -ExpandProperty Count
	if ($cnt -gt 0) 
	{
		Write-Host ""
		Write-Host " * INFO: Server '$srvName' is already running... continuing with next server."
	} 
	else 
	{
		$startCmd = getStartCommandForServer -srvName $escapedSrvName
		if ([string]::IsNullOrEmpty($startCmd)) 
		{
			Write-Host ""
			Write-Host " ERROR: Problems getting START Command for server '$srvName'. no start command was found."
		} 
		else 
		{
			## Expand any environment variables in the command
#			$startCmdEval = Invoke-Expression -Command $startCmd
#			$startCmdEval = Write-Output $startCmd
			$startCmdEval = $ExecutionContext.InvokeCommand.ExpandString($startCmd)

#exit 1

			$srvNameOrAlias = $srvName
			$withAliasDesc = ""
			$params = $startCmdEval -split '\s+'
			for ($j = 0; $j -lt $params.Length; $j++) 
			{
				if ($params[$j] -eq "-A" -or $params[$j] -eq "--serverAlias") 
				{
					$srvNameOrAlias = $params[$j + 1].Trim("'")
					$withAliasDesc = "with alias name '$srvNameOrAlias'"
				}
			}

			if ($startCmdEval -match '^(/.service )') 
			{
				Write-Host ""
				Write-Host " * Starting monitoring of server '$srvName' $withAliasDesc (as a SERVICE)."
				Start-Service -Name $srvNameOrAlias

				Write-Host "   Sleeping for ${sleepTime} before continuing with next server"
				Start-Sleep -Seconds $sleepTime
			} 
			else 
			{
#				Start-Process -FilePath "powershell" -ArgumentList "-NoProfile -Command `& { & '${dbxCentralBase}/bin/start_dbxcentral.bat' *> '${dbxCentralBase}/log/DBX_CENTRAL.console' }" -NoNewWindow -PassThru | ForEach-Object {
				Write-Host ""
				Write-Host " * Starting monitoring of server '$srvName' $withAliasDesc (with Start-Process in background, output to ${dbxCentralBase}/log/${srvNameOrAlias}.console)."
#				$process = Start-Process -FilePath "powershell" -ArgumentList $startCmdEval -RedirectStandardOutput "${dbxCentralBase}/log/${srvNameOrAlias}.console" -RedirectStandardError "${dbxCentralBase}/log/${srvNameOrAlias}.console" -NoNewWindow -PassThru
#				$lastBgPid = $process.Id

				## We need to start it somehow in background (like Linux 'nohup') but I can't figgure out how to do it in PowerShell
				$logPath = "${dbxCentralBase}/log/${srvNameOrAlias}.console"
				$process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c", "${startCmdEval} > `"$logPath`" 2>&1" -NoNewWindow -PassThru
#				$process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c start /b ${startCmdEval} > `"$logPath`" 2>&1" -NoNewWindow -PassThru
#				Start-Job -ScriptBlock {
#				    Start-Process -FilePath "cmd.exe" -ArgumentList "/c", "${startCmdEval} > `"$using:logPath`" 2>&1" -NoNewWindow
#				}
#				schtasks /create /tn "DetachedProcess" /tr "cmd.exe /c ${startCmdEval} > `"$logPath`" 2>&1" /sc ONCE /st 00:00 /f
#				schtasks /run /tn "DetachedProcess"
#				Invoke-WmiMethod -Class Win32_Process -Name Create -ArgumentList "cmd.exe /c ${startCmdEval} > `"$logPath`" 2>&1"


				$lastBgPid = $process.Id
		
				Write-Host "   Sleeping for ${sleepTime} before continuing with next server"
				Start-Sleep -Seconds $sleepTime

				$isPidAlive = Get-CimInstance -Class Win32_Process -Filter "ProcessId = $lastBgPid"
				if (-not $isPidAlive) {
					Write-Host ""
					Write-Host "==================================================================="
					Write-Host "ERROR: Problems Starting monitoring of server '$srvName'."
					Write-Host "  cmd: $startCmd"
					Write-Host " eval: $startCmdEval"
					Write-Host "------ BEGIN output from the start command ------------------------"
					Get-Content "${dbxCentralBase}/log/${srvNameOrAlias}.console"
					Write-Host "------ END   output from the start command ------------------------"
					Write-Host ""
				}
			}
		}
	}
}

##----------------------------------------------
## Check that they are stopped
##----------------------------------------------
if ([string]::IsNullOrEmpty($listAll)) {
	Write-Host " **** Sorry I couldn't find the 'list_ALL.ps1' command."
	Write-Host " **** So I can NOT list started servers."
	Write-Host ""
} else {
	Write-Host ""
	Write-Host " * Started all servers, Now LIST them"

	& $listAll
}

